#include <netdb.h>
#include <sys/socket.h>
#include <resolv.h>
#include <arpa/inet.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <sys/types.h>
#include <pwd.h>

#define VERSION "2.0"

#define PORT 69
#define MAX_DATA 512
#define RETRIES 5
#define TIMEOUT 10
#define PIDFILE "/var/run/tftpd.pid"
#define LOGFILE "/var/log/tftpd.log"
#define ERRFILE "/var/log/tftpd.err"
#define USER "tftpd"

#define OP_RRQ 1
#define OP_WRQ 2
#define OP_DATA 3
#define OP_ACK 4
#define OP_ERR 5
#define OP_OACK 6

#define E_NODEF 0
#define E_NOENT 1
#define E_ACCES 2
#define E_NOSPC 3
#define E_PERM 4
#define E_UNKN 5
#define E_EXIST 6
#define E_NOUSER 7

#define S_NOENT "File not found"
#define S_ACCES "Access violation"
#define S_NOSPC "Disk full or allocation exceeded"
#define S_PERM "Illegal TFTP operation"
#define S_UNKN "Unknown transfer ID"
#define S_EXIST "File already exists"
#define S_NOUSER "No such user"

#define STDOUT 1
int STDERR = 2;

socklen_t ssin = sizeof(struct sockaddr);

char *pidfile = PIDFILE;
int logfd, srv;

struct tftp_pkt {
  uint16_t opcode;
  union {
    struct {
      uint16_t code;
      char data[MAX_DATA];
    } p;
    char file[MAX_DATA+2];
  } u;
};

void _write (const int filedes, va_list *va, int count)
{
  static char* c;
  for(; count; count--) {
    c = va_arg(*va, char*);
    write(filedes, c, strlen(c));
  }
}

char* ultoa (unsigned long l)
{
  static unsigned long tmp;
  static char* c;
  static char buf[MAX_DATA];

  tmp=l;
  c=buf;
  do {
    c++;
    tmp /= 10;
  } while (tmp>0);
  if (c - buf > MAX_DATA)
    strcpy(buf, "<buf2small>");
  else {
    for (*c = 0, c--; c >= buf; c--, l /= 10)
      *c = l % 10 + '0';
  }
  return buf;
}

char* zs (char* c)
{
  if (strlen(c) < 2) {
    memmove(c+1, c, strlen(c)+1);
    *c = '0';
  }
  return c;
}

void print_timestamp (int fd)
{
  static char* MONATE[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
                            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
  static struct tm *now;
  static time_t tt;
  static char *c;

  time(&tt);

  now = gmtime(&tt);
  now->tm_year += 1900;

  write(fd, "[", 1);
  c = zs(ultoa(now->tm_mday));
  write(fd, c, strlen(c));
  write(fd, "/", 1);
  write(fd, MONATE[now->tm_mon], strlen(MONATE[now->tm_mon]));
  write(fd, "/", 1);
  c = ultoa(now->tm_year);
  write(fd, c, strlen(c));
  write(fd, ":", 1);
  c = zs(ultoa(now->tm_hour));
  write(fd, c, strlen(c));
  write(fd, ":", 1);
  c = zs(ultoa(now->tm_min));
  write(fd, c, strlen(c));
  write(fd, ":", 1);
  c = zs(ultoa(now->tm_sec));
  write(fd, c, strlen(c));
  write(fd, " +0000] ", 8);
}

void die (int count, ...)
{
  va_list va;
  if (count) {
    if (STDERR != 2) print_timestamp(STDERR);
    va_start(va, count);
    _write(STDERR, &va, count);
    va_end(va);
    write(STDERR, "\n", 1);
    count = 1;
  }
  close(srv);
  close(logfd);
  close(STDERR);
  exit(count);
}

void warn (int count, ...)
{
  static va_list va;
  if (STDERR != 2) print_timestamp(STDERR);
  va_start(va, count);
  _write(STDERR, &va, count);
  va_end(va);
  write(STDERR, "\n", 1);
}

void log (struct sockaddr_in *sin, char* method, char* file,
          char* code, char* size)
{
  write(logfd, inet_ntoa(sin->sin_addr), strlen(inet_ntoa(sin->sin_addr)));
  write(logfd, " - - ", 5);
  print_timestamp(logfd);
  write(logfd, "\"", 1);
  write(logfd, method, strlen(method));
  write(logfd, " ", 1);
  write(logfd, file, strlen(file));
  write(logfd, "\" ", 2);
  write(logfd, code, strlen(code));
  write(logfd, " ", 1);
  write(logfd, size, strlen(size));
  write(logfd, " \"-\" \"-\"\n", 9);
}

void sigall (char *signal)
{
  warn(2, signal, " received, tftpd stopped");
  close(srv);
  close(logfd);
  close(STDERR);
  unlink(pidfile);
  exit(0);
}

void sigterm () { sigall("SIGTERM"); }
void sigint () { sigall("SIGINT"); }
void sigchld () { wait(0); }

void tftp_snd (int fd, struct sockaddr_in *sin, struct tftp_pkt *pkt,
               uint16_t opcode, uint16_t code, int len)
{
  pkt->opcode = htons(opcode);
  pkt->u.p.code = htons(code);
  len+=4;
  if (sendto(fd, pkt, len, 0, (struct sockaddr*) sin, ssin) != len)
    die(2, "sendto(): ", strerror(errno));
}

void tftp_err (int fd, struct sockaddr_in *sin, struct tftp_pkt *pkt,
              uint16_t code, char *data)
{
  strcpy((char*)&pkt->u.p.data, data);
  tftp_snd(fd, sin, pkt, OP_ERR, code, strlen(data) + 1);
}

void version ()
{
  write(STDOUT, "fjo's tftpd version " VERSION "\n", 21 + strlen(VERSION));
  exit(0);
}

void usage ()
{
  write(STDOUT,
"usage:\n"
"\n"
"  -V               prints version and exists\n"
"  -h               prints this help screen\n"
"  -n               avoids forking into background\n"
"  -i <ip>|<host>   listens only on specified <ip> or <host>\n"
"  -p <pidfile>     saves pid to <pidfile> instead of " PIDFILE "\n"
"  -l <logfile>     logs to <logfile> instead of " LOGFILE "\n"
"  -e <errfile>     logs errors to <errfile> instead of " ERRFILE "\n"
"  -u <user>        drop root privileges to <user> instead of " USER "\n"
"                   chroot to homedir of <user> instead of " USER "\n"
"\n",
  strlen(PIDFILE) + strlen(LOGFILE) + strlen(ERRFILE) + 2*strlen(USER) + 488);
  version();
}

void setsignal (int signal, void *func)
{
  static struct sigaction sa;

  sa.sa_handler = func;
  sa.sa_flags = 0;
  if (sigaction(signal, &sa, NULL) == -1)
    die(4, "sigaction (", ultoa(signal), "): ", strerror(errno));
}

int udp_socket (struct in_addr *addr, in_port_t port)
{
  static struct sockaddr_in sin;
  static int sock;

  sin.sin_family = AF_INET;
  sin.sin_port = htons(port);
  sin.sin_addr = *addr;
  sock = socket(PF_INET, SOCK_DGRAM, 0);
  if (sock == -1) die(2, "socket(): ", strerror(errno));
  if (bind(sock, (struct sockaddr*) &sin, ssin) == -1)
    die(2, "bind(): ", strerror(errno));
  return sock;
}

void handle_client (struct sockaddr_in *sin, struct in_addr *addr,
                    struct tftp_pkt *pkt)
{
  char size[MAX_DATA], buffer[MAX_DATA];
  int fh, i, len, count = 0, tries;
  struct tftp_pkt tmppkt;
  struct sockaddr_in tmpsin;
  struct stat st;

  close(srv);
  srv = udp_socket(addr, 0);
  pkt->opcode = ntohs(pkt->opcode);
  strncpy(&buffer[0], &pkt->u.file[0], MAX_DATA);

  switch (pkt->opcode) {

    case OP_RRQ:
      fh = open(pkt->u.file, O_RDONLY);
      if (fh == -1) {
        log(sin, "RRQ", &buffer[0], "404", "-");
        tftp_err(srv, sin, pkt, E_NOENT, S_NOENT);
        die(0);
      }
      for (;;) {
        len = read(fh, &pkt->u.p.data[0], MAX_DATA);
        if (len == -1) {
          warn(4, "read(", &buffer[0], "): ", strerror(errno));
          tftp_err(srv, sin, pkt, E_ACCES, S_ACCES);
          close(fh);
          die(0);
        }
        tries = 0;
        ++count;
        for (;;) {
          tftp_snd(srv, sin, pkt, OP_DATA, count, len);
          i = recvfrom(srv, &tmppkt, sizeof(struct tftp_pkt), 0,
                       (struct sockaddr*) &tmpsin, &ssin);
          tmppkt.opcode = ntohs(tmppkt.opcode);
          tmppkt.u.p.code = ntohs(tmppkt.u.p.code);
          if (i == -1) {
            if (errno != EINTR) die(2, "recvfrom(): ", strerror(errno));
            continue;
          }
          if (memcmp(&tmpsin, sin, ssin))
            warn(4, "spoofed packet from ", inet_ntoa(tmpsin.sin_addr), ":",
                 ultoa(tmpsin.sin_port));
          else if ((tmppkt.opcode != OP_ACK) || (tmppkt.u.p.code != count)) {
            if (errno != EINTR) die(2, "recvfrom(): ", strerror(errno));
            if (++tries > RETRIES) {
              close(fh);
              die(4, "timeout while waiting for OP_ACK from ",
                  inet_ntoa(sin->sin_addr), ":", ultoa(sin->sin_port));
            }
          }
          else break;
        }
        if (len < MAX_DATA) {
          if (fstat(fh, &st) == -1) st.st_size = -1;
          close(fh);
          strcpy(&size[0], ultoa(st.st_size));
          log(sin, "RRQ", &buffer[0], "200", &size[0]);
          die(0);
        }
      }
      break;

    case OP_WRQ:
      fh = open(pkt->u.file, O_RDONLY);
      if (fh != -1) {
        log(sin, "WRQ", &buffer[0], "401", "-");
        tftp_err(srv, sin, pkt, E_EXIST, S_EXIST);
        die(0);
      }
      fh = open(pkt->u.file, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);
      if (fh == -1) {
        warn(4, "open(", &buffer[0], "): ", strerror(errno));
        tftp_err(srv, sin, pkt, E_NOSPC, S_NOSPC);
        die(0);
      }
      i = MAX_DATA;
      for (;;) {
        tries = 0;
        for (;;) {
          tftp_snd(srv, sin, pkt, OP_ACK, count, 0);
          if (i < MAX_DATA) {
            if (fstat(fh, &st) == -1) st.st_size = -1;
            if (fchmod(fh, S_IRUSR) == -1) warn(2, "fchmod: ", strerror(errno));
            close(fh);
            strcpy(&size[0], ultoa(st.st_size));
            log(sin, "WRQ", &buffer[0], "200", &size[0]);
            die(0);
          }
          ++count;
          i = recvfrom(srv, &tmppkt, sizeof(struct tftp_pkt), 0,
                       (struct sockaddr*) &tmpsin, &ssin);
          tmppkt.opcode = ntohs(tmppkt.opcode);
          tmppkt.u.p.code = ntohs(tmppkt.u.p.code);
          if (i == -1) {
            if (errno != EINTR) die(2, "recvfrom(): ", strerror(errno));
            continue;
          }
          if (memcmp(&tmpsin, sin, ssin))
            warn(4, "spoofed packet from ", inet_ntoa(tmpsin.sin_addr), ":",
                 ultoa(tmpsin.sin_port));
          else if (i < 4)
            warn(4, "too small udp packet from ", inet_ntoa(sin->sin_addr), ":",
                 ultoa(sin->sin_port));
          else if ((tmppkt.opcode != OP_DATA) || (tmppkt.u.p.code != count)) {
            if (++tries > RETRIES) {
              close(fh);
              die(4, "timeout while waiting for OP_DATA from ",
                  inet_ntoa(sin->sin_addr), ":", ultoa(sin->sin_port));
            }
          }
          else break;
        }
        if (write(fh, &pkt->u.p.data[0], i) != i) {
          warn(4, "write(", &buffer[0], "): ", strerror(errno));
          tftp_err(srv, sin, pkt, E_NOSPC, S_NOSPC);
          close(fh);
          die(0);
        }
      }
      break;

    default:
      die(4, "bogus opcode from ", inet_ntoa(sin->sin_addr), ":",
             ultoa(sin->sin_port));
  }
}

int main (int argc, char *argv[])
{
  struct tftp_pkt pkt;
  struct in_addr addr;
  struct sockaddr_in sin;
  struct hostent *he;
  struct passwd *pass;
  int i, prefork = 1;
  char *c, buffer[MAX_DATA],
       *logfile = LOGFILE,
       *errfile = ERRFILE,
       *user    = USER;

  addr.s_addr = INADDR_ANY;

  for (i = 1; i < argc; ++i) {
    if (!strcmp("-V", argv[i])) version();
    if (!strcmp("-h", argv[i])) usage();
    if (!strcmp("-n", argv[i])) { prefork = 0; continue; }
    if (!strcmp("-i", argv[i])) {
      if (++i == argc)
        die(1, "parameter -i needs an ip address or a host name");

      if (!(he = gethostbyname(argv[i])))
        die(4, "gethostbyname(", argv[i], "): ", hstrerror(h_errno));

      memmove(&addr, *(he->h_addr_list), he->h_length);
      continue;
    }
    if (!strcmp("-p", argv[i])) {
      if (++i == argc) die(1, "parameter -p needs a filepath");
      pidfile = argv[i];
      continue;
    }
    if (!strcmp("-l", argv[i])) {
      if (++i == argc) die(1, "parameter -l needs a filepath");
      logfile = argv[i];
      continue;
    }
    if (!strcmp("-e", argv[i])) {
      if (++i == argc) die(1, "parameter -e needs a filepath");
      errfile = argv[i];
      continue;
    }
    if (!strcmp("-u", argv[i])) {
      if (++i == argc) die(1, "parameter -u needs a username");
      user = argv[i];
      continue;
    }
    warn(2, "unknown parameter: ", argv[i]);
    usage();
  }

  pass = getpwnam(user);
  if (!pass) die(3, "getpwnam(", user, "): no such user");
    
  if (chdir(pass->pw_dir) == -1)
    die(4, "chdir(", pass->pw_dir, "): ", strerror(errno));

  if ((i = open(pidfile, O_RDONLY)) != -1) {
    read(i, &buffer[0], 5);
    close(i);
    buffer[5] = '\0';
    i = atoi(&buffer[0]);
    if (!kill(i, 0)) die(3, "tftpd already running (pid=", &buffer[0], ")");
  }

  (prefork)?(i=fork()):(i=0);
  if (i==-1) die(2, "fork(): ", strerror(errno));
  if (!i) {
    if ((i = open(pidfile, O_WRONLY|O_CREAT, S_IRUSR|S_IWUSR)) == -1)
      die(4, "open(", pidfile, "): ", strerror(errno));

    c = ultoa(getpid());
    write(i, c, strlen(c));
    close(i);

    setsignal(SIGTERM, &sigterm);
    setsignal(SIGINT, &sigint);
    setsignal(SIGCHLD, &sigchld);

    srv = udp_socket(&addr, PORT);

    logfd = open(logfile, O_WRONLY | O_CREAT | O_APPEND, S_IRUSR | S_IWUSR);
    if (logfd == -1) die(4, "open(", logfile, "): ", strerror(errno));

    i = open(errfile, O_WRONLY | O_CREAT | O_APPEND, S_IRUSR | S_IWUSR);
    if (i == -1) die(4, "open(", errfile, "): ", strerror(errno));

    if (chroot(pass->pw_dir) == -1)
      die(4, "chroot(", pass->pw_dir, "): ", strerror(errno));

    if (setuid(pass->pw_uid) == -1)
      die(4, "setuid(", ultoa(pass->pw_uid), "): ", strerror(errno));

    STDERR = i;

    warn(1, "tftpd started");

    for (;;) {
      i = recvfrom(srv, &pkt, sizeof(struct tftp_pkt), 0,
                   (struct sockaddr*) &sin, &ssin);
      if (i==-1) {
        if (errno!=EINTR) die(2, "recvfrom(): ", strerror(errno));
      } else if (i>3) {
        prefork = fork();
        if (prefork == -1) die(2, "fork(): ", strerror(errno));
        if (!prefork) handle_client(&sin, &addr, &pkt);
      } else
        warn(4, "too small udp packet from ", inet_ntoa(sin.sin_addr), ":",
                ultoa(sin.sin_port));
    }
  }
  close(logfd);
  close(STDERR);
  return 0;
}
