/*
Copyright (c) 2012, Felix J. Ogris
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met: 

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer. 
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/* suexec - a replacement for Apache's suexec helper with support for
 *          virtual user and group ids
 * Synopsis:
 * 1. Load the suexec module into your httpd, e.g. in httpd.conf:
 *    LoadModule suexec_module libexec/apache22/mod_suexec.so
 * 2. In each VirtualHost, add a line like this one:
 *    SuexecUserGroup #20000 #20000
 *    Keep the uid between MINUID and MAXUID, and gid between MINGID and
 *    MAXGID (see below)
 */

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <sys/types.h>

#define MINGID 20000
#define MAXGID 21000
#define MINUID 20000
#define MAXUID 21000
#define ALLOWED_GROUP "www"
#define ALLOWED_USER "www"

/* from the real suexec.c */
static const char *const safe_env_lst[] =
{
    /* variable name starts with */
    "HTTP_",
    "SSL_",

    /* variable name is */
    "AUTH_TYPE=",
    "CONTENT_LENGTH=",
    "CONTENT_TYPE=",
    "DATE_GMT=",
    "DATE_LOCAL=",
    "DOCUMENT_NAME=",
    "DOCUMENT_PATH_INFO=",
    "DOCUMENT_ROOT=",
    "DOCUMENT_URI=",
    "GATEWAY_INTERFACE=",
    "HTTPS=",
    "LAST_MODIFIED=",
    "PATH_INFO=",
    "PATH_TRANSLATED=",
    "QUERY_STRING=",
    "QUERY_STRING_UNESCAPED=",
    "REMOTE_ADDR=",
    "REMOTE_HOST=",
    "REMOTE_IDENT=",
    "REMOTE_PORT=",
    "REMOTE_USER=",
    "REDIRECT_HANDLER=",
    "REDIRECT_QUERY_STRING=",
    "REDIRECT_REMOTE_USER=",
    "REDIRECT_STATUS=",
    "REDIRECT_URL=",
    "REQUEST_METHOD=",
    "REQUEST_URI=",
    "SCRIPT_FILENAME=",
    "SCRIPT_NAME=",
    "SCRIPT_URI=",
    "SCRIPT_URL=",
    "SERVER_ADMIN=",
    "SERVER_NAME=",
    "SERVER_ADDR=",
    "SERVER_PORT=",
    "SERVER_PROTOCOL=",
    "SERVER_SIGNATURE=",
    "SERVER_SOFTWARE=",
    "UNIQUE_ID=",
    "USER_NAME=",
    "TZ=",

    "FCGI_ROLE=",

    NULL
};

static int cpyenv(char **envp, char **newenv)
{
  int count;
  int i;
  int j;

  for (count = 0, i = 0; envp[i]; i++)
    for (j = 0; safe_env_lst[j]; j++)
      if (!strncmp(envp[i], safe_env_lst[j], strlen(safe_env_lst[j]))) {
        if (newenv) newenv[count] = envp[i];
        count++;
        break;
      }

  return count;
}

int main(int argc, char **argv, char **envp)
{
  gid_t newgid;
  uid_t newuid;
  struct passwd *pw;
  int count;
  char **newenv;

  if (argc < 4) return -1;

  newgid = atol(argv[2]);
  newuid = atol(argv[1]);

  if ((newgid < MINGID) || (newgid > MAXGID) ||
      (newuid < MINUID) || (newuid > MAXUID))
    return -2;

  if (((pw = getpwnam(ALLOWED_USER)) == NULL) ||
      (pw->pw_gid != getgid()) ||
      (pw->pw_uid != getuid()))
    return -3;

  if ((setgid(newgid) != 0) || (setuid(newuid) != 0))
    return -4;

  count = cpyenv(envp, NULL);

  if ((newenv = malloc((count + 1) * sizeof(char*))) == NULL)
    return -5;

  cpyenv(envp, newenv);

  newenv[count] = NULL;

  return execve(argv[3], argv + 3, newenv);
}
