#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include "libpq-fe.h"
#include "db.h"

/*
 * look for a line containing
 * key = value
 * and let *value point to it
 */

void parse_line (const char *buf, const char *key, char **value)
{
  static char *left, *right;

  if (!value) return;
  left = strstr(buf, key);
  if (!left) return;
  left += strlen(key);
  while ((*left == ' ') || (*left == '\t') || (*left == '=')) ++left;
  right = strchr(left, '\n');
  if (!right) return;
  *right = '\0';
  *value = malloc(right - left + 1);
  if (!*value) return;
  strcpy(*value, left);
  *right = '\n';
}


/*
 * get db connection parameters from config file
 */

ssize_t get_config (char *config, void **buf, char **host, char **port,
                    char **db, char **user, char **pass, char **error_url)
{
  struct stat st;
  ssize_t len;
  int fd;

  fd = open(config, O_RDONLY);
  if (fd < 0) return -1;
  if (fstat(fd, &st) < 0) { close(fd); return -1; }
  len = st.st_size;
  *buf = mmap(NULL, len, PROT_READ|PROT_WRITE, MAP_PRIVATE, fd, 0);
  close(fd);

  if (*buf == MAP_FAILED) return -1;
  parse_line(*buf, "host", host);
  parse_line(*buf, "port", port);
  parse_line(*buf, "db", db);
  parse_line(*buf, "user", user);
  parse_line(*buf, "pass", pass);
  parse_line(*buf, "error_url", error_url);
  return len;
}


/*
 * opens conn to database using config file
 * returns NULL on success, an error message otherwise
 */

char *db_start (char *config, PGconn **conn, char **error_url)
{
  PGresult *res;
  ssize_t len;
  char *host = NULL, *port = NULL, *db = NULL, *user = NULL, *pass = NULL;
  void *mmapp;

  /* read database connection parameters */
  len = get_config(config, &mmapp, &host, &port, &db, &user, &pass, error_url);

  /* connect to the database */
  *conn = PQsetdbLogin(host, port, NULL, NULL, db, user, pass);
  if (len >= 0) munmap(mmapp, len);

  /* cannot connect to the database */
  if (PQstatus(*conn) == CONNECTION_BAD) return PQerrorMessage(*conn);

  res = PQexec(*conn, CHECK_TABLE);

  /* no such table */
  if (PQresultStatus(res) != PGRES_TUPLES_OK) {
    PQclear(res);
    res = PQexec(*conn, CREATE_TABLE);

    /* cannot create our table */
    if (PQresultStatus(res) != PGRES_COMMAND_OK) {
      char *ret = PQresultErrorMessage(res);
      PQclear(res);
      PQfinish(*conn);
      return ret;
    }
  }

  PQclear(res);
  return NULL;
} 
