
#include <ESP8266WebServer.h>
#include <DHT.h>
#include <Adafruit_BMP085_U.h>
#include "wifi.h"
#include "settings.h"

ESP8266WebServer server(80);

const char *ROOTPAGE = "<!DOCTYPE html>\n"
                       "<html>\n"
                       "<head>\n"
                       "<title>ESP8266</title>\n"
                       "<style>\n"
                       "body {\n"
                       "  font-family:sans-serif;\n"
                       "}\n"
                       "body, input {\n"
                       "  font-size:3vh;\n"
                       "}\n"
                       "#error {\n"
                       "  color:red;\n"
                       "  font-weight:bold;\n"
                       "}\n"
                       "</style>\n"
                       "</head>\n"
                       "<body>\n"
                       "<h1><a href=\"/hostname\" id=\"hostname\"></a></h1>\n"
                       "<table>\n"
                       "<tr>\n"
                       "<td><a href=\"/temperature\">Temperature</a>:</td><td id=\"temperature\"></td>\n"
                       "</tr><tr>\n"
                       "<td><a href=\"/humidity\">Humidity</a>:</td><td id=\"humidity\"></td>\n"
                       "</tr><tr>\n"
                       "<td><a href=\"/pressure\">Pressure</a>:</td><td id=\"pressure\"></td>\n"
                       "</tr><tr>\n"
                       "<td><a href=\"/temperature2\">Temperature 2</a>:</td><td id=\"temperature2\"></td>\n"
                       "</tr><tr>\n"
                       "<td><a href=\"/uptime\">Uptime</a>:</td><td id=\"uptime\"></td>\n"
                       "</tr><tr>\n"
                       "<td>Update interval:</td>\n"
                       "<td><input type=\"text\" id=\"updateInterval\" style=\"width:4em\"> seconds</td>\n"
                       "</tr>\n"
                       "</table>\n"
                       "<ul>\n"
                       "<li><a href=\"/json\">JSON</a></li>\n"
                       "<li><a href=\"/rrdupdate\">rrdupdate</a></li>\n"
                       "<li><a href=\"/reset\">Reset me</a></li>\n"
                       "</ul>\n"
                       "<div id=\"error\">&nbsp;</div>\n"
                       "</body>\n"
                       "<script>\n"
                       "function parseUptime (millis) {\n"
                       "  millis /= 1000;\n"
                       "  var uptime = \":\" + String(Math.floor(millis % 60)).padStart(2, \"0\");\n"
                       "  millis /= 60;\n"
                       "  uptime = \":\" + String(Math.floor(millis % 60)).padStart(2, \"0\") + uptime;\n"
                       "  millis /= 60;\n"
                       "  uptime = \"d \" + String(Math.floor(millis % 24)).padStart(2, \"0\") + uptime;\n"
                       "  millis /= 24;\n"
                       "  uptime = \"\" + Math.floor(millis) + uptime;\n"
                       "  return uptime;\n"
                       "}\n"
                       "function processStatus (response) {\n"
                       "  try {\n"
                       "    data = JSON.parse(response);\n"
                       "    document.title = data[\"hostname\"];\n"
                       "    document.getElementById(\"hostname\").innerHTML = data[\"hostname\"];\n"
                       "    document.getElementById(\"temperature\").innerHTML = \"\" + data[\"temperature\"] + \" &deg;C\";\n"
                       "    document.getElementById(\"humidity\").innerHTML = \"\" + data[\"humidity\"] + \" %\";\n"
                       "    document.getElementById(\"pressure\").innerHTML = \"\" + data[\"pressure\"] + \" hPa\";\n"
                       "    document.getElementById(\"temperature2\").innerHTML = \"\" + data[\"temperature2\"] + \" &deg;C\";\n"
                       "    document.getElementById(\"uptime\").innerHTML = parseUptime(data[\"uptime\"]);\n"
                       "  } catch (e) {\n"
                       "    console.log(e);\n"
                       "  }\n"
                       "}\n"
                       "function displayError (errorString) {\n"
                       "  document.getElementById(\"error\").innerHTML = errorString;\n"
                       "}\n"
                       "function getUpdateInterval () {\n"
                       "  var updateInterval = 10;\n"
                       "  var uiElem = document.getElementById(\"updateInterval\");\n"
                       "  try {\n"
                       "    updateInterval = parseInt(uiElem.value);\n"
                       "  } catch (e) {\n"
                       "  }\n"
                       "  if (isNaN(updateInterval) || (updateInterval < 1)) {\n"
                       "    updateInterval = 10;\n"
                       "    uiElem.value = updateInterval.toString();\n"
                       "  }\n"
                       "  return updateInterval * 1000;\n"
                       "}\n"
                       "function startRequest () {\n"
                       "  var xmlHttp = new XMLHttpRequest();\n"
                       "  xmlHttp.onreadystatechange = function () {\n"
                       "    if (xmlHttp.readyState == 4) {\n"
                       "      if (xmlHttp.status == 200) {\n"
                       "        displayError(\"\");\n"
                       "        processStatus(xmlHttp.response);\n"
                       "      } else {\n"
                       "        displayError(\"Cannot fetch data!\");\n"
                       "      }\n"
                       "      window.setTimeout(startRequest, getUpdateInterval());\n"
                       "    }\n"
                       "  };\n"
                       "  xmlHttp.open(\"GET\", \"/json\");\n"
                       "  xmlHttp.timeout = getUpdateInterval() - 500;\n"
                       "  xmlHttp.send();\n"
                       "}\n"
                       "startRequest();\n"
                       "</script>\n"
                       "</html>\n";
const char *RESETPAGE = "<!DOCTYPE html>\n"
                        "<html>\n"
                        "<head>\n"
                        "<meta http-equiv=\"refresh\" content=\"10; URL=/\">\n"
                        "<title>Rebooting...</title>\n"
                        "<style>\n"
                        "body {\n"
                        "  font-family:sans-serif;\n"
                        "}\n"
                        "</style>\n"
                        "</head>\n"
                        "<body>\n"
                        "<p>Rebooting...</p>\n"
                        "</body>\n"
                        "</html>\n";

enum BMPMODE { PRESSURE, TEMPERATURE2 };

#define DHTTYPE DHT22
#define DHTPIN  2 // GPIO2 == D4
DHT dht(DHTPIN, DHTTYPE);

Adafruit_BMP085_Unified bmp = Adafruit_BMP085_Unified(10085);
bool bmp_present = false;

void setup() {
  // put your setup code here, to run once:
  Serial.begin(115200);
  while (!Serial)
    delay(500);
    
  Serial.println("");
  Serial.print("Hostname: ");
  Serial.println(WiFi.hostname());
  Serial.print("Connecting to ");
  Serial.println(WIFISSID);

  WiFi.mode(WIFI_STA);
  WiFi.begin(WIFISSID, WIFIPASS);
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("connected!");
  Serial.print("IP: ");
  Serial.println(WiFi.localIP());
  Serial.print("GW: ");
  Serial.println(WiFi.gatewayIP());
  Serial.print("DNS: ");
  Serial.println(WiFi.dnsIP());
  
  pinMode(DHTPIN, INPUT);
  dht.begin();
  
  if (bmp.begin())
    bmp_present = true;
  else
    Serial.println("no BMP sensor found");
  
  server.onNotFound(handle404);
  server.on("/", handleRoot);
  server.on("/reset", handleReset);
  server.on("/temperature", handleTemperature);
  server.on("/humidity", handleHumidity);
  server.on("/pressure", handlePressure);
  server.on("/temperature2", handleTemperature2);
  server.on("/hostname", handleHostname);
  server.on("/uptime", handleUptime);
  server.on("/json", handleJson);
  server.on("/rrdupdate", handleRrdupdate);
  server.begin();
}

float getTemperature () {
  return dht.readTemperature();
}

float getHumidity () {
  return dht.readHumidity();
}

float getPressureOrTemperature2 (enum BMPMODE m) {
  float retval = 0.0;
  
  if (bmp_present) {
    sensors_event_t event;
    bmp.getEvent(&event);
    if (!event.pressure) {
    } else if (m == PRESSURE) {
      retval = event.pressure + PRESSURE_DELTA;
    } else {
      bmp.getTemperature(&retval);
    }
  }

  return retval;
}

void logger (int code) {
  Serial.print(server.client().remoteIP());
  Serial.print(" ");
  Serial.print(code);
  Serial.print(" ");
  Serial.println(server.uri());
}

void sendGeneric (String ct, String s, int code=200) {
  logger(code);
  server.send(code, ct, s);
}

void sendText (String s, int code=200) {
  sendGeneric("text/plain", s, code);
}

void sendHtml (String h) {
  sendGeneric("text/html", h);
}

void sendJson (String j) {
  sendGeneric("application/json", j);
}

void sendFloat (float f) {
  sendText(String(f));
}

void sendInteger (int i) {
  sendText(String(i));
}

void handle404 () {
  sendText("Not found", 404);
}

String float2string (float f, String s) {
  return (isnan(f) ? s : String(f));
}

String float2json (float f) {
  return float2string(f, "null");
}

String float2rrd (float f) {
  return float2string(f, "U");
}

void handleRoot () {
  sendHtml(ROOTPAGE);
}

void handleReset () {
  sendHtml(RESETPAGE);
  delay(500);
  ESP.restart();
}

void handleTemperature () {
  sendFloat(getTemperature());
}

void handleHumidity () {
  sendFloat(getHumidity());
}

void handlePressure () {
  sendFloat(getPressureOrTemperature2(PRESSURE));
}

void handleTemperature2 () {
  sendFloat(getPressureOrTemperature2(TEMPERATURE2));
}

void handleHostname () {
  sendText(WiFi.hostname());
}

void handleUptime () {
  sendInteger(millis());
}

void handleJson () {
  sendJson(String("{\n") +
           String("\"temperature\" : ")  + float2json(getTemperature())  + String(",\n") +
           String("\"humidity\" : ")     + float2json(getHumidity())     + String(",\n") +
           String("\"pressure\" : ")     + float2json(getPressureOrTemperature2(PRESSURE))     + String(",\n") +
           String("\"temperature2\" : ") + float2json(getPressureOrTemperature2(TEMPERATURE2)) + String(",\n") +
           String("\"hostname\" : \"")   + WiFi.hostname()          + String("\",\n") +
           String("\"uptime\" : ")       + String(millis())         + String("\n") +
           String("}"));
}

void handleRrdupdate () {
  sendText(String("temperature:humidity:pressure:temperature2\nN:") +
           float2rrd(getTemperature())  + String(":") +
           float2rrd(getHumidity())     + String(":") +
           float2rrd(getPressureOrTemperature2(PRESSURE))     + String(":") +
           float2rrd(getPressureOrTemperature2(TEMPERATURE2)) + String("\n"));
}

void loop() {
  // put your main code here, to run repeatedly:
  server.handleClient();
  delay(50); // lower power consumption
  
  if (millis() > 24 * 60 * 60 * 1000) {
    Serial.println("Regular reboot");
    ESP.restart();
  }
}
