#!/usr/bin/perl -w

package graylib;

use strict;
use warnings;
use DBI;

my %DEFAULTS = (
  # graypold.pl
  dbtype                   => "Pg",
  dbhost                   => "",
  dbport                   => "",
  dbuser                   => "",
  dbpass                   => "",
  dbname                   => "",
  db_key_err               => "duplicate key value violates unique constraint",

  listen_graylist          => "/var/run/graypold_graylist.sock",
  listen_mailcount         => "/var/run/graypold_mailcount.sock",
  listen_sent              => "/var/run/graypold_sent.sock",
  listen_spf               => "/var/run/graypold_spf.sock",
  listen_spf_helo          => "/var/run/graypold_spf_helo.sock",

  # see https://metacpan.org/pod/Mail::SPF::Server#max_void_dns_lookups
  spf_max_void_dns_lookups => 2,

  owner                    => "postfix",
  rights                   => "0600",
  backlog                  => "128",
  pidfile                  => "/var/run/graypold.pid",
  user                     => "graylist",

  # graycron.pl
  lockfile                 => "/var/lock/graycron.lock",
  rrdfile_graylist         => "/var/db/graylist.rrd",
  rrdfile_mailcount        => "/var/db/mailcount.rrd",
  rrdfile_spf_helo         => "/var/db/spf_helo.rrd",
  rrdfile_spf_mfrom        => "/var/db/spf_mfrom.rrd",
  picpath                  => "/usr/local/www/data",
  delay_permit             => "3 minute",
  delay_halfopen           => "12 hour",
  delay_max                => "35 day",
);

sub new ($)
{
  my ($class) = @_;

  my $self = {
    config_file => "/usr/local/etc/graypol.conf",
    %DEFAULTS,
  };

  bless($self, $class);
}

sub version ($)
{
  12;
}

sub db_key_err ($)
{
  my ($self) = @_;

  if ($self->{dbtype} eq "SQLite") {
    $self->{db_key_err} = "UNIQUE constraint failed";
  } elsif ($self->{dbtype} eq "mysql") {
    $self->{db_key_err} = "Duplicate entry";
  } else {
    $self->{db_key_err} = $DEFAULTS{db_key_err};
  }
}

sub load_config ($)
{
  my ($self) = @_;

  open(FH, "<", $self->{config_file}) || return "$self->{config_file}: $!";

  $self->{db_key_err} = "";

  foreach (map { s/\r|\n//sgio; $_ } <FH>) {
    next if /^\s*$/;
    next if /^#/;
    next if /^;/;
    next if /^\/\//;

    next unless /^([^\s]+)\s+=\s+(.*)$/;
    my ($key, $value) = ($1, $2);
    if (!exists $DEFAULTS{$key}) {
      return "$self->{config_file}: unknown option: $key";
    }

    $self->{$key} = $value;
  }

  close(FH);

  $self->db_key_err() if !$self->{db_key_err};

  return "";
}

sub dbopen ($$$)
{
  my ($self, $handle_error, $table) = @_;
  my $is_sqlite = ($self->{dbtype} eq "SQLite");

  $self->dbclose() if $is_sqlite;

  if (!$self->{dbh}) {
    my %attr;

    if ($handle_error) {
      %attr = (
        RaiseError  => 1,
        PrintError  => 0,
        HandleError => $handle_error,
      );
    }

    my $dbname = $self->{dbname};
    $dbname .= "/" . $table . ".sqlite" if $is_sqlite;

    my $dsn = "DBI:$self->{dbtype}:dbname=$dbname";
    $dsn .= ";host=$self->{dbhost}" if $self->{dbhost};
    $dsn .= ";port=$self->{dbport}" if $self->{dbport};

    $self->{dbh} = DBI->connect($dsn, $self->{dbuser}, $self->{dbpass},
                                \%attr);
  }

  return $self->{dbh};
}

sub dbclose ($)
{
  my ($self) = @_;

  if ($self->{dbh}) {
    $self->{dbh}->disconnect();
    $self->{dbh} = undef;
  }
}

sub dbinterval ($$)
{
  my ($self, $base, $add) = @_;

  if ($self->{dbtype} eq "SQLite") {
    return "datetime($base, '$add')";
  } elsif ($self->{dbtype} eq "mysql") {
    return "$base + interval $add";
  } else {
    return "$base + interval '$add'";
  }
}

1;
