#!/usr/local/bin/perl -w

# modules
use strict;
use warnings;
use RRDs;
use Fcntl qw(:flock);
use FindBin;
use lib "$FindBin::Bin/../lib";
use graylib;


# helper for counting
sub dbcount ($$)
{
  my ($config, $clause) = @_;

  my $sql = "SELECT COUNT(*) FROM graylist";
  $sql .= " WHERE $clause" if $clause;
  my @row = $config->{dbh}->selectrow_array($sql);

  return (@row ? $row[0] : undef);
}

my %actions;

# called every minute
$actions{permit} = sub ($) {
  my ($config) = @_;
  my $sql = "UPDATE graylist SET ts_ok=current_timestamp " .
            "WHERE ts_ok IS NULL AND " .
            "current_timestamp >= " .
            $config->dbinterval("ts_entry", $config->{delay_permit});
  $config->dbopen(undef, "graylist")->do($sql);
  $config->dbclose();
};

# called every 5 minutes
$actions{graph} = sub ($) {
  my ($config) = @_;

  if (!-e $config->{rrdfile_graylist}) {
    # hmmm, step implies crontab entry here...
    RRDs::create($config->{rrdfile_graylist}, "--start", "1000000000",
                 "--step", 300,
                 "DS:deny:GAUGE:600:0:U", "DS:halfopen:GAUGE:600:0:U",
                 "DS:permit:GAUGE:600:0:U", "RRA:MAX:0:1:210240");
    my $err = RRDs::error;
    die "create $config->{rrdfile_graylist} failed: $err" if $err;
  }
  # get numbers
  $config->dbopen(undef, "graylist");
  my $cnt_deny = dbcount($config, "ts_ok IS NULL");
  my $clause = "NOT ts_ok IS NULL AND " .
               "ts_latest < " .
               $config->dbinterval("ts_entry", $config->{delay_permit});
  my $cnt_halfopen = dbcount($config, $clause);
  my $cnt_all = dbcount($config, "");

  my $cnt_permit;
  if (defined $cnt_deny && defined $cnt_halfopen && defined $cnt_all) {
    $cnt_permit = $cnt_all - $cnt_deny - $cnt_halfopen;
    # update rrd file
    RRDs::update($config->{rrdfile_graylist}, "-t", "deny:halfopen:permit",
                 "N:$cnt_deny:$cnt_halfopen:$cnt_permit");
    my $err = RRDs::error;
    die "update $config->{rrdfile_graylist} failed: $err" if $err;
  } else {
    $cnt_deny = $cnt_halfopen = $cnt_permit = "?";
  }
  # make pictures
  foreach my $period("1h", "1d", "1w", "1m", "1y") {
    my $fn = $config->{picpath} . "/graylist" . $period . ".png";
    RRDs::graph($fn, "-s", "end-" . $period, "-l", "0", "-w", "500",
                "-t", "Graylisting per " . $period,
                "DEF:deny=$config->{rrdfile_graylist}:deny:MAX",
                "DEF:halfopen=$config->{rrdfile_graylist}:halfopen:MAX",
                "DEF:permit=$config->{rrdfile_graylist}:permit:MAX",
                "AREA:permit#00FF00:permit\\: $cnt_permit\\n",
                "STACK:halfopen#0000FF:half open\\: $cnt_halfopen\\n",
                "STACK:deny#FF0000:deny\\: $cnt_deny");
    my $err = RRDs::error;
    die "graph $fn ($config->{rrdfile_graylist}) failed: $err" if $err;
  }

  if (!-e $config->{rrdfile_mailcount}) {
    # hmmm, step implies crontab entry here...
    RRDs::create($config->{rrdfile_mailcount}, "--start", "1000000000",
                 "--step", 300,
                 "DS:mails:COUNTER:600:0:U", "DS:recipients:COUNTER:600:0:U",
                 "RRA:MIN:0:1:210240", "RRA:AVERAGE:0:1:210240",
                 "RRA:MAX:0:1:210240", "RRA:LAST:0:1:210240");
    my $err = RRDs::error;
    die "create $config->{rrdfile_mailcount} failed: $err" if $err;
  }
  # get numbers
  my $sql = "SELECT COUNT(*), SUM(count) FROM mailcount";
  my ($cnt_mails, $cnt_recipients) = $config->dbopen(undef, "mailcount")
                                            ->selectrow_array($sql);

  if (defined $cnt_mails && defined $cnt_recipients) {
    # update rrd file
    RRDs::update($config->{rrdfile_mailcount}, "-t", "mails:recipients",
                 "N:$cnt_mails:$cnt_recipients");
    my $err = RRDs::error;
    die "update $config->{rrdfile_mailcount} failed: $err" if $err;
  }
  # make pictures
  foreach my $period("1h", "1d", "1w", "1m", "1y") {
    my $fn = $config->{picpath} . "/mailcount" . $period . ".png";
    RRDs::graph($fn, "-s", "end-" . $period, "-l", "0", "-w", "500",
                "-t", "Mails per " . $period,
                "DEF:minmails=$config->{rrdfile_mailcount}:mails:MIN",
                "DEF:avgmails=$config->{rrdfile_mailcount}:mails:AVERAGE",
                "DEF:maxmails=$config->{rrdfile_mailcount}:mails:MAX",
                "DEF:lstmails=$config->{rrdfile_mailcount}:mails:LAST",
                "DEF:minrcpts=$config->{rrdfile_mailcount}:recipients:MIN",
                "DEF:avgrcpts=$config->{rrdfile_mailcount}:recipients:AVERAGE",
                "DEF:maxrcpts=$config->{rrdfile_mailcount}:recipients:MAX",
                "DEF:lstrcpts=$config->{rrdfile_mailcount}:recipients:LAST",
                "CDEF:varmails=maxmails,minmails,-",
                "CDEF:varrcpts=maxrcpts,minrcpts,-",
                "AREA:minmails",
                "STACK:varmails#bbbbbb",
                "LINE1:avgmails#00ff00:Mails     ",
                "GPRINT:minmails:MIN:Min\\:%6.1lf %s",
                "GPRINT:avgmails:AVERAGE:Avg\\:%6.1lf %s",
                "GPRINT:maxmails:MAX:Max\\:%6.1lf %s",
                "GPRINT:lstmails:LAST:Cur\\:%6.1lf %s\\n",
                "AREA:minrcpts",
                "STACK:varrcpts#ccccff",
                "LINE1:avgrcpts#0000ff:Recipients",
                "GPRINT:minrcpts:MIN:Min\\:%6.1lf %s",
                "GPRINT:avgrcpts:AVERAGE:Avg\\:%6.1lf %s",
                "GPRINT:maxrcpts:MAX:Max\\:%6.1lf %s",
                "GPRINT:lstrcpts:LAST:Cur\\:%6.1lf %s\\n");
    my $err = RRDs::error;
    die "graph $fn ($config->{rrdfile_mailcount}) failed: $err" if $err;
  }

  my %codes = (
    pass      => "00FF00",
    fail      => "FF0000",
    softfail  => "0088FF",
    neutral   => "0044FF",
    none      => "0000FF",
    error     => "FF4400",
    permerror => "FF6600",
    temperror => "FF9900",
  );

  foreach my $scope(qw(helo mfrom)) {
    my $rrdfile = $config->{"rrdfile_spf_" . $scope};
    if (!-e $rrdfile) {
      # e.g. DS:helo_fail:COUNTER:600:0:U
      my @ds = map { "DS:" . $_ . ":COUNTER:600:0:U" } keys %codes;
      # hmmm, step implies crontab entry here...
      RRDs::create($rrdfile, "--start", "1000000000", "--step", 300, @ds,
                   "RRA:MIN:0:1:210240", "RRA:AVERAGE:0:1:210240",
                   "RRA:MAX:0:1:210240", "RRA:LAST:0:1:210240");
      my $err = RRDs::error;
      die "create $rrdfile failed: $err" if $err;
    }
    # get numbers
    my (@keys, @vals);
    my %values = map { $_ => 0 } keys %codes;
    $sql = "SELECT COUNT(code), code FROM spf WHERE scope=? GROUP BY code";
    my $sth = $config->dbopen(undef, "spf")->prepare($sql);
    $sth->execute($scope);
    while (my ($cnt, $code) = $sth->fetchrow_array()) {
      if (defined $codes{$code}) {
        push @keys, $code;
        push @vals, $cnt;
        $values{$code} = $cnt;
      }
    }
    $sth->finish();

    if (@keys) {
      # update rrd file
      RRDs::update($rrdfile, "-t", join(":", @keys), "N:" . join(":", @vals));
      my $err = RRDs::error;
      die "update $rrdfile failed: $err" if $err;
    }
    # make pictures
    foreach my $period("1h", "1d", "1w", "1m", "1y") {
      my (@defs, @graphs);
      while (my ($code, $color) = each %codes) {
        push @defs, "DEF:" . $code . "=" . $rrdfile . ":" . $code . ":MAX";
        push @graphs, (@graphs ? "STACK:" : "AREA:") . $code . "#" . $color .
                      ":" . $code . "\\: " . $values{$code} . "\\n";
      }
        
      my $fn = $config->{picpath} . "/spf_" . $scope . $period . ".png";
      RRDs::graph($fn, "-s", "end-" . $period, "-l", "0", "-w", "500",
                  "-t", "SPF " . $scope . " per " . $period,
                  @defs, @graphs);
      my $err = RRDs::error;
      die "graph $fn ($rrdfile) failed: $err" if $err;
    }
  }

  $config->dbclose();
};

# called once an hour
$actions{clean} = sub ($) {
  my ($config) = @_;
  my $sql = "DELETE FROM graylist WHERE NOT ts_ok IS NULL AND " .
            $config->dbinterval("ts_entry", $config->{delay_permit}) .
            " > ts_latest AND " .
            $config->dbinterval("ts_entry", $config->{delay_halfopen}) .
            " <= current_timestamp AND " .
            "NOT ip = '::'";
  $config->dbopen(undef, "graylist")->do($sql);
  $config->dbclose();
};

# called once a day
$actions{scrub} = sub ($) {
  my ($config) = @_;
  my $sql = "DELETE FROM graylist WHERE " .
            $config->dbinterval("ts_latest", $config->{delay_max}) .
            " <= current_timestamp";
  $config->dbopen(undef, "graylist")->do($sql);
  $config->dbclose();
};


### MAIN ###
# get action from commandline
my $action = shift;
if (!$action || !exists $actions{$action}) {
  die "usage: $0 <" . join(" | ", keys %actions), ">";
}

# load defaults and generic helpers
my $config = graylib->new();

# load config
my $err = $config->load_config();
warn $err if $err;

# avoid multiple instances
my $lockfile = "$config->{lockfile}.$action";
open(FH, ">", $lockfile) || die "$lockfile: $!";
flock(FH, LOCK_EX | LOCK_NB) || exit 0;

# open database and do something
$actions{$action}->($config);

# byebye
flock(FH, LOCK_UN) || die "$lockfile: $!";
close(FH) || die "$lockfile: $!";
unlink($lockfile);
