#!/usr/bin/perl -w

# modules
use strict;
use warnings;
use IO::Socket;
use Sys::Syslog;
use POSIX;
use FindBin qw($Bin);
use Sys::Hostname;
use Mail::SPF;

my $HOSTNAME = Sys::Hostname::hostname();

# load defaults and generic helpers
our %config;
require "$Bin/../lib/graylib.pl";

# socket has to be global so that byebye() can inform postfix
my $socket;

# helper functions
sub normalize_ip ($)
{
  my $ip = shift;

  if ($ip =~ /:/) {
    # normalize ipv6 address to /64
    my @parts = split(/::/, $ip);
    my @left = ($#parts > -1 ? split(/:/, $parts[0]) : ());
    my @right = ($#parts > 0 ? split(/:/, $parts[1]) : ());

    my $missing = 8 - ($#left + 1) - ($#right + 1);
    for (my $i = 0; $i < $missing; $i++) {
      push @left, "0";
    }

    $ip = join(":", (@left, @right)[0..3]) . "::";
  } else {
    # normalize ipv4 address to /24
    substr($ip, rindex($ip, ".") + 1) = "0";
  }

  return $ip;
}

sub choose_action ($@)
{
  my ($action, $reason) = @_;

  $action = "dunno" if !$action;

  if ($action eq "defer") {
    return "action=defer_if_permit Please try again in several minutes.\n\n";
  }
  if ($action eq "prepend") {
    return "action=prepend $reason\n\n";
  }
  if ($action eq "reject") {
    return "action=reject" . ($reason ? " $reason" : "") . "\n\n";
  }
  return "action=dunno\n\n";
}

# fatal error
sub byebye ($@)
{
  my ($errstr, $dbh) = @_;

  print $socket choose_action("defer") if $socket;
  if ($dbh) {
    foreach my $sth(grep { defined } @{$dbh->{ChildHandles}}) {
      $sth->finish() if (ref($sth) eq "DBI::st");
      $sth = undef;
    }
    $dbh->disconnect();
  }

  Sys::Syslog::syslog("err", "%s", $errstr);
  Sys::Syslog::closelog();

  exit(1);
}

# return uid and gid for a given username
sub get_uid_gid ($)
{
  my ($user) = @_;

  my ($login, $pass, $uid, $gid) = getpwnam($user);
  byebye("$user: no such user") if (!defined $uid || !defined $gid);
  return ($uid, $gid);
}

# create a unix or tcp listen socket
sub listen_socket ($)
{
  my ($listen) = @_;
  my $socket;

  if ($listen =~ /^\//) {
    # unix socket
    unlink($listen);
    $socket = IO::Socket::UNIX->new(
      Local  => $listen,
      Type   => IO::Socket::SOCK_STREAM,
      Listen => $config{backlog}) || byebye("socket(): $listen: $!");
    my ($uid, $gid) = get_uid_gid($config{owner});
    chown($uid, $gid, $listen) || byebye("chown(): $listen: $!");
    chmod(oct($config{rights}), $listen) || byebye("chmod(): $listen: $!");
  } else {
    # inet socket
    $socket = IO::Socket::INET->new(
      LocalAddr  => $listen,
      Proto      => "tcp",
      ReuseAddr  => 1,
      Type       => IO::Socket::SOCK_STREAM,
      Listen     => $config{backlog}) ||
      byebye("socket(): $listen: $!");
  }

  return $socket;
}

# generic graylisting callbacks
sub _graypol_queries ($) {
  my ($ts_ok) = @_;
  my $primkey = "WHERE ip=? AND sender=? AND recipient=?";

  return (
    "select" => "SELECT ts_ok FROM graylist $primkey",
    "update" => "UPDATE graylist SET count=count+?, ts_latest=now() $primkey",
    "insert" => "INSERT INTO graylist (hostname, ip, username, sender, " .
                "recipient, count, instance, ts_entry, ts_latest, ts_ok) " .
                "VALUES (?, ?, ?, ?, ?, ?, ?, now(), now(), $ts_ok)",
  );
}

sub _graypol_check ($$$$$$) {
  my ($sth, $params, $ip, $from, $to, $do_insert) = @_;

  my $action = "defer";

  # check for entry in graylist
  $sth->{select}->execute($ip, $from, $to);

  if ($sth->{select}->rows == 1) {
    # found an entry
    my ($ts_ok) = $sth->{select}->fetchrow_array();
    # update timestamp
    $sth->{update}->execute(($ts_ok ? 0 : 1), $ip, $from, $to);
    $action = "dunno" if $ts_ok;
  } elsif ($do_insert) {
    # did not found an entry -> create it
    local $sth->{insert}->{HandleError} = sub {
      my ($errstr, $dbh) = @_;
      if ($errstr =~ /$config{db_key_err}/) {
        $sth->{update}->execute(1, $ip, $from, $to);
      } else {
        byebye($errstr, $dbh);
      }
    };
    $sth->{insert}->execute($HOSTNAME, $ip, $params->{sasl_username}, $from,
                            $to, 1, $params->{instance});
  }

  return $action;
}

# graylisting callbacks
sub graypol_queries () {
  return _graypol_queries("NULL");
}

sub graypol_check ($$) {
  my ($sth, $params) = @_;

  my $action = "dunno";

  if (($params->{request} eq "smtpd_access_policy") &&
      ($params->{protocol_state} eq "RCPT")) {

    my $from = lc($params->{sender});
    my $to = lc($params->{recipient});

    $action = _graypol_check($sth, $params,
                             normalize_ip(lc($params->{client_address})),
                             $from, $to, 1) ||
              _graypol_check($sth, $params, "::", "",    $to, 0) ||
              _graypol_check($sth, $params, "::", $from, $to, 0);
  }

  return $action;
}

# mailcount callbacks
sub mailcount_queries () {
  return (
    "insert" => "INSERT INTO mailcount (hostname, ip, username, sender, " .
                "recipient, count, instance, ts_entry, queue_id, size) " .
                "VALUES (?, ?, ?, ?, ?, ?, ?, now(), ?, ?)",
  );
}

sub mailcount_check ($$$) {
  my ($sth, $private, $params) = @_;

  return "dunno" if ($params->{request} ne "smtpd_access_policy");

  if (!$$private) {
    $$private = {};
  }
  my $instances = $$private;

  my $from = lc($params->{sender});
  my $to = lc($params->{recipient});

  if ($params->{protocol_state} eq "RCPT") {
    # just collect all recipients
    my $instance = $params->{instance};
    push @{$instances->{$instance}}, $to;
  } elsif ($params->{protocol_state} eq "END-OF-MESSAGE") {
    my $instance = $params->{instance};
    my $recipients;
    if (defined $instances->{$instance}) {
      $recipients = join(",", @{$instances->{$instance}});
      delete $instances->{$instance};
    } else {
      # we didn't get the recipients during the RCPT phase
      # maybe this mail has just one recipient, then postfix will tell it to
      # us at the end of message
      $recipients = $to;
    }
    $sth->{insert}->execute($HOSTNAME, $params->{client_address},
                            $params->{sasl_username}, $from, $recipients,
                            $params->{recipient_count}, $params->{instance},
                            $params->{queue_id}, $params->{size});
    Sys::Syslog::syslog("info", "instance=%s client=%s sasl_username=%s " .
                        "queue_id=%s size=%i nrcpts=%i from=%s to=%s",
                        $params->{instance}, $params->{client_address},
                        $params->{sasl_username}, $params->{queue_id},
                        $params->{size}, $params->{recipient_count},
                        $from, $recipients);
  }

  return "dunno";
}

# sent callbacks to bypass graylisting if there's been outgoing mail
sub sent_queries () {
  return _graypol_queries("now()");
}

sub sent_check ($$) {
  my ($sth, $params) = @_;

  if (($params->{request} eq "smtpd_access_policy") &&
      ($params->{protocol_state} eq "RCPT") &&
      $params->{sasl_username}) {
    my $from = lc($params->{sender});
    my $to = lc($params->{recipient});
    _graypol_check($sth, $params, "::", $to, $from, 1);
  }

  return "dunno";
}

# spf callbacks
sub spf_queries () {
  return (
    "insert" => "INSERT INTO spf (hostname, ip, scope, heloname, sender, " .
                "code, reason, instance, ts_entry) " .
                "VALUES (?, ?, ?, ?, ?, ?, ?, ?, now())",
  );
}

sub spf_check ($$$) {
  my ($sth, $private, $params) = @_;

  return "dunno" if ($params->{request} ne "smtpd_access_policy");
  foreach (qw(sender client_address)) {
    return "dunno" if !$params->{$_};
  }

  if (!$$private) {
    $$private = Mail::SPF::Server->new();
  }
  my $spf_srv = $$private;

  my $from = lc($params->{sender});

  my $req = Mail::SPF::Request->new(
    scope         => "mfrom",
    identity      => $params->{sender},
    ip_address    => $params->{client_address},
    helo_identity => $params->{helo_name}
  );
  my $res = $spf_srv->process($req);

  $sth->{insert}->execute($HOSTNAME, $params->{client_address}, "mfrom",
                          $params->{helo_name}, $from, $res->code, $res->text,
                          $params->{instance});
  Sys::Syslog::syslog("info", "instance=%s client=%s scope=mfrom " .
                      "helo=%s from=%s code=%s reason=%s",
                      $params->{instance}, $params->{client_address},
                      $params->{helo_name}, $from, $res->code, $res->text);

  if (($res->code eq "fail") || ($res->code eq "permerror")) {
    return ("reject", $res->local_explanation);
  } else {
    return ("prepend", $res->received_spf_header);
  }
}

sub spf_helo_check ($$$) {
  my ($sth, $private, $params) = @_;

  return "dunno" if ($params->{request} ne "smtpd_access_policy");
  foreach (qw(helo_name client_address)) {
    return "dunno" if !$params->{$_};
  }

  if (!$$private) {
    $$private = Mail::SPF::Server->new();
  }
  my $spf_srv = $$private;

  my $req = Mail::SPF::Request->new(
    scope      => "helo",
    identity   => $params->{helo_name},
    ip_address => $params->{client_address}
  );
  my $res = $spf_srv->process($req);

  $sth->{insert}->execute($HOSTNAME, $params->{client_address}, "helo",
                          $params->{helo_name}, undef, $res->code, $res->text,
                          $params->{instance});
  Sys::Syslog::syslog("info", "instance=%s client=%s scope=helo " .
                      "helo=%s code=%s reason=%s",
                      $params->{instance}, $params->{client_address},
                      $params->{helo_name}, $res->code, $res->text);

  if (($res->code eq "fail") || ($res->code eq "permerror")) {
    return ("reject", $res->local_explanation);
  } else {
    return ("prepend", $res->received_spf_header);
  }
}


### main ###
my $running;

### autoflush, signal handlers
$| = 1;
foreach (keys %SIG) {
  $SIG{$_} = "IGNORE";
}
$SIG{TERM} = sub {
  $running = 0;
};
$SIG{INT} = sub {
  $running = 0;
};
$SIG{CHLD} = sub { };

### daemonize
my $pid = fork();
die "fork(): $!" if ($pid < 0);
exit(0) if $pid;

### open syslog
Sys::Syslog::openlog("graypold", "pid", "daemon");
close(STDIN);
close(STDOUT);
close(STDERR);
POSIX::setsid();

### load configfile over defaults
my $err = load_config();
Sys::Syslog::syslog("warning", "%s", "$err") if $err;

### create listen sockets and get their file descriptors
my %listen_sockets;
foreach (qw(graypol mailcount sent spf spf_helo)) {
  $listen_sockets{$_}->{socket} = listen_socket($config{"listen_" . $_});
  $listen_sockets{$_}->{fileno} = fileno($listen_sockets{$_}->{socket});
}

### unprivileged user
my ($unpriv_uid, $unpriv_gid) = get_uid_gid($config{user});

### save pid
if (open(FH, "<", $config{pidfile})) {
  $pid = <FH>;
  close(FH);
  if (kill(0, $pid)) {
    byebye("already running (pid $pid)!");
  } else {
    Sys::Syslog::syslog("warning", "removing stale pid file %s",
                        $config{pidfile});
  }
}
open(FH, ">", $config{pidfile}) || byebye("open(): $config{pidfile}: $!");
print FH $$;
close(FH);
if (!chown($unpriv_uid, $unpriv_gid, $config{pidfile})) {
  byebye("chown(): $config{pidfile}: $!");
}

byebye("setgid(): $!") if !POSIX::setgid($unpriv_gid);
byebye("setuid(): $!") if !POSIX::setuid($unpriv_uid);

Sys::Syslog::syslog("info", "starting...");

### main loop
for ($running = 1; $running; ) {
  do { } while (waitpid(-1, POSIX::WNOHANG) > 0);
  $socket = undef;

  # wait for activity on the listening sockets
  my $rin = "";
  foreach (keys %listen_sockets) {
    vec($rin, $listen_sockets{$_}->{fileno}, 1) = 1;
  }
  my $nfound = select(my $rout=$rin, undef, undef, undef);
  if ($nfound < 0) {
    next if $!{EINTR};
    byebye("select(): $!");
  }

  foreach my $name(keys %listen_sockets) {
    next if (vec($rout, $listen_sockets{$name}->{fileno}, 1) != 1);

    $socket = $listen_sockets{$name}->{socket}->accept();
    next if !$socket;

    $pid = fork();
    byebye("fork(): $!") if ($pid < 0);
    next if $pid; # parent continues with next listening socket

    # child
    $0 .= " [" . $name . "]";
    $SIG{TERM} = "DEFAULT";
    $SIG{INT}  = "DEFAULT";
    $SIG{CHLD} = "DEFAULT";
    foreach (keys %listen_sockets) {
      close($listen_sockets{$_}->{socket});
    }

    # child reloads config file
    $err = load_config();
    Sys::Syslog::syslog("warning", "%s", "$err") if $err;

    # connect to db
    my $dbh = dbopen(\&byebye);

    # prepare statements
    my %queries;
    %queries = graypol_queries() if ($name eq "graypol");
    %queries = mailcount_queries() if ($name eq "mailcount");
    %queries = sent_queries() if ($name eq "sent");
    %queries = spf_queries() if ($name eq "spf");
    %queries = spf_queries() if ($name eq "spf_helo");

    my %sth;
    while (my ($stmt_name, $query) = each %queries) {
      $sth{$stmt_name} = $dbh->prepare($query);
    }

    # client loop
    my (%params, $private);
    while (<$socket>) {
      s/\r|\n//sgio;
      if (!/^$/o) {
        # save parameter
        my ($key, $value) = split(/=/, $_, 2);
        $params{$key} = $value;
        next;
      }
      # empty line, end of parameters
      my $action = "dunno";
      my $reason;
      if ($name eq "graypol") {
        ($action, $reason) = graypol_check(\%sth, \%params);
      }
      if ($name eq "mailcount") {
        ($action, $reason) = mailcount_check(\%sth, \$private, \%params);
      }
      if ($name eq "sent") {
        ($action, $reason) = sent_check(\%sth, \%params);
      }
      if ($name eq "spf") {
        ($action, $reason) = spf_check(\%sth, \$private, \%params);
      }
      if ($name eq "spf_helo") {
        ($action, $reason) = spf_helo_check(\%sth, \$private, \%params);
      }
      print $socket choose_action($action, $reason);
    }

    # child exits
    foreach my $sth(keys %sth) {
      $sth->finish();
    }
    %sth = undef;
    $dbh->disconnect();
    Sys::Syslog::closelog();
    exit(0);
  }
}

foreach (keys %listen_sockets) {
  close($listen_sockets{$_}->{socket});
}
unlink($config{pidfile});
Sys::Syslog::syslog("info", "%s", "exiting...");
Sys::Syslog::closelog();
