#!/usr/bin/perl -w

use strict;
use warnings;
use IO::Socket;
use DBI;
use Sys::Syslog;
use POSIX;
use FindBin qw($Bin);

our %config;
require "$Bin/../lib/graylib.pl";

my $listen_socket;

### signal handler, autoflush
$| = 1;
$SIG{CHLD} = "IGNORE";
$SIG{TERM} = sub {
  Sys::Syslog::syslog("info", "shutting down...");
  close($listen_socket);
  unlink($config{pidfile});
  exit(0);
};

### daemonize
my $pid = fork();
die "fork(): $!" if ($pid < 0);
exit(0) if $pid;

### open syslog
Sys::Syslog::setlogsock("unix");
Sys::Syslog::openlog("graypold", "pid", "daemon");
close(STDIN);
close(STDOUT);
close(STDERR);
POSIX::setsid();

# load configfile over defaults
my $err = &load_config();
Sys::Syslog::syslog("warning", "%s", "$err") if $err;

### save pid
if (open(FH, "$config{pidfile}")) {
  $pid = <FH>;
  close(FH);
  if (kill($pid, 0)) {
    &byebye("already running (pid $pid)!");
  } else {
    Sys::Syslog::syslog("warning", "%s", "removing stale pid file $config{pidfile}");
  }
}
open(FH, ">$config{pidfile}") || &byebye("$config{pidfile}: $!");
print FH $$;
close(FH);

# open socket
if ($config{listen} =~ /^\//) {
  # unix socket
  unlink($config{listen});
  $listen_socket = new IO::Socket::UNIX(
    Local  => $config{listen},
    Type   => IO::Socket::SOCK_STREAM,
    Listen => $config{backlog}) || &byebye("socket: $config{listen}: $!");
  my ($login, $pass, $uid, $gid) = getpwnam($config{owner});
  &byebye("$config{owner}: $!") if (!defined $uid || !defined $gid);
  chown($uid, $gid, $config{listen}) || &byebye("chown: $config{listen}: $!");
  chmod(oct($config{rights}), $config{listen}) ||
    &byebye("chmod: $config{listen}: $!");
} else {
  # inet socket
  $listen_socket = new IO::Socket::INET(
    LocalAddr  => $config{listen},
    LocalPort  => $config{port},
    Proto      => "tcp",
    ReuseAddr  => 1,
    Type       => IO::Socket::SOCK_STREAM,
    Listen     => $config{backlog}) ||
    &byebye("socket: $config{listen}:$config{port}: $!");
}

Sys::Syslog::syslog("info", "starting...");

# main loop
my ($socket, $dbh, %sth, %session);
while ($socket = $listen_socket->accept()) {
  $pid = fork();
  &handle_error("fork(): $!") if ($pid < 0);
  if ($pid) {
    close($socket);
    do {
    } while (waitpid(-1, &POSIX::WNOHANG) > 0);
    next;
  }

  $SIG{TERM} = "DEFAULT";
  close($listen_socket);

  # child reloads config file
  $err = &load_config();
  Sys::Syslog::syslog("warning", "%s", "$err") if $err;

  # connect to db
  $dbh = &dbopen(\&handle_error);

  # prepare statements
  my $primkey = "WHERE $config{db_field_ip}=? AND $config{db_field_from}=? AND ";
  $primkey .= "$config{db_field_to}=?";

  my $sql = "SELECT $config{db_field_ok} FROM $config{db_tbl_gray} $primkey";
  $sth{select} = $dbh->prepare($sql);

  $sql = "UPDATE $config{db_tbl_gray} SET $config{db_field_ts_latest}=now(), ";
  $sql .= "$config{db_field_count}=$config{db_field_count} + ? $primkey";
  $sth{update} = $dbh->prepare($sql);

  $sql = "INSERT INTO $config{db_tbl_gray} ($config{db_field_ip}, ";
  $sql .= "$config{db_field_from}, $config{db_field_to}, ";
  $sql .= "$config{db_field_ts_entry}, $config{db_field_ts_latest}, ";
  $sql .= "$config{db_field_ts_ok}) VALUES (?, ?, ?, now(), now(), now())";
  $sth{insert} = $dbh->prepare($sql);

  if ($config{dbtype} eq "Pg") {
    # blackwhite list
    $sql = "SELECT $config{db_field_ok} FROM $config{db_tbl_bw} WHERE ";
    $sql .= "$config{db_field_ip}>>=? AND $config{db_field_from}=? AND ";
    $sql .= "$config{db_field_to}=? ORDER BY $config{db_field_ip} DESC LIMIT 1";
    $sth{select_bw} = $dbh->prepare($sql);
  } elsif ($config{dbtype} eq "mysql") {
    # el-cheapo blackwhite list (mysql)
    $sql = "SELECT $config{db_field_ok} FROM $config{db_tbl_bw} WHERE ";
    $sql .= "LENGTH(?)>0 AND $config{db_field_from}=? AND ";
    $sql .= "$config{db_field_to}=? LIMIT 1";
    $sth{select_bw} = $dbh->prepare($sql);
  }

  $sql = "INSERT INTO $config{db_tbl_mailcount} ($config{db_field_count}, ";
  $sql .= "$config{db_field_ip}, $config{db_field_from}, ";
  $sql .= "$config{db_field_to}, $config{db_field_size}, ";
  $sql .= "$config{db_field_queue_id}, $config{db_field_instance}, ";
  $sql .= "$config{db_field_ts_entry}) ";
  $sql .= "VALUES (?, ?, ?, ?, ?, ?, ?, now())";
  $sth{mailcount} = $dbh->prepare($sql);

  ### client loop
  my %params;
  while (<$socket>) {
    s/\r|\n//sgio;
    if (/^$/o) {
      ### empty line, end of parameters
      print $socket &check_policy(%params);
    } else {
      ### save parameter
      my ($key, $value) = split(/=/, $_, 2);
      $params{$key} = $value;
    }
  }

  # child exits
  foreach my $sth(keys %sth) {
    $sth->finish();
  }
  %sth = undef;
  $dbh->disconnect();
  Sys::Syslog::closelog();
  exit(0);
}

sub check_policy ()
{
  my %params = @_;
  my $ok = 1;

  ### ensure that we are called at the correct stage
  if ($params{request} eq "smtpd_access_policy") {
    if ($params{protocol_state} eq "RCPT") {
      $ok = &graylisting(%params);
      if ($ok) {
        push @{$session{$params{instance}}}, $params{recipient};
        Sys::Syslog::syslog("info", "instance=%s client=%s from=%s to=%s",
                            $params{instance}, $params{client_address},
                            $params{sender}, $params{recipient});
      }
    }
    if ($params{protocol_state} eq "END-OF-MESSAGE") {
      my $instance = $params{instance};
      my $recipients = "";
      if (defined $session{$instance}) {
        $recipients = join(",", @{$session{$instance}});
        delete $session{$instance};
      }
      $sth{mailcount}->execute($params{recipient_count},
                               $params{client_address}, $params{sender},
                               $recipients, $params{size}, $params{queue_id},
                               $params{instance});
      Sys::Syslog::syslog("info", "instance=%s client=%s queue_id=%s " .
                          "size=%i nrcpts=%i from=%s to=%s",
                          $params{instance}, $params{client_address},
                          $params{queue_id}, $params{size},
                          $params{recipient_count}, $params{sender},
                          $recipients);
    }
  }

  return &choose_action($ok);
}

sub graylisting ()
{
  my %params = @_;

  ### the primary key
  my $ip = &normalize_ip(lc($params{client_address}));
  my $from = lc($params{sender});
  my $to = lc($params{recipient});

  if (exists $sth{select_bw}) {
    ### check for entry in blackwhitelist
    my ($from_user, $from_dom) = split(/\@/, $from, 2);
    my ($to_user, $to_dom) = split(/\@/, $to, 2);

    foreach my $f($from, $from_dom, "") {
      foreach my $t($to, $to_dom, "") {
        $sth{select_bw}->execute($ip, $f, $t);
        if ($sth{select_bw}->rows == 1) {
          my $row = $sth{select_bw}->fetchrow_hashref();
          return $row->{$config{db_field_ok}};
        }
      }
    }
  }

  ### check for entry in graylist
  my $ok = 1;
  $sth{select}->execute($ip, $from, $to);
  if ($sth{select}->rows == 1) {
    ### found an entry
    my $row = $sth{select}->fetchrow_hashref();
    $ok = $row->{$config{db_field_ok}};
    ### update timestamp(s)
    $sth{update}->execute(($ok ? 0 : 1), $ip, $from, $to);
  } else {
    $ok = 0;
    ### did not found an entry -> create it
    $sth{insert}->{RaiseError} = 0;
    $sth{insert}->{HandleError} = undef;
    $sth{insert}->execute($ip, $from, $to);
    if ($dbh->err && ($dbh->errstr !~ /$config{db_key_err}/o)) {
      &handle_error($dbh->errstr);
    }
    $sth{insert}->{HandleError} = \&handle_error;
    $sth{insert}->{RaiseError} = 1;
  }

  return $ok;
}

sub normalize_ip ()
{
  my $ip = shift;

  if ($ip =~ /:/) {
    # normalize ipv6 address to /64
    my @parts = split(/::/, $ip);
    my @left = ($#parts > -1 ? split(/:/, $parts[0]) : ());
    my @right = ($#parts > 0 ? split(/:/, $parts[1]) : ());

    my $missing = 8 - ($#left + 1) - ($#right + 1);
    for (my $i = 0; $i < $missing; $i++) {
      push @left, "0";
    }

    $ip = join(":", (@left, @right)[0..3]) . "::";
  } else {
    # normalize ipv4 address to /24
    substr($ip, rindex($ip, ".") + 1) = "0";
  }

  return $ip;
}

sub choose_action ()
{
  return "action=dunno\n\n" if $_[0];
  return "action=defer_if_permit Please try again in several minutes.\n\n";
}

sub handle_error ()
{
  print $socket &choose_action(1);
  foreach my $sth(keys %sth) {
    $sth->finish();
  }
  %sth = undef;
  $dbh->disconnect() if $dbh;
  &byebye($_[0]);
}

# fatal error
sub byebye ()
{
  my $err = shift;
  Sys::Syslog::syslog("err", "%s", $err);
  Sys::Syslog::closelog();
  exit(1);
}
