#!/usr/bin/perl

use IO::Socket;
use DBI;
use Sys::Syslog ":DEFAULT", "setlogsock";
use POSIX;
require "/usr/local/lib/graylib.pl";

=for comment

1. create tables:

1.1. PostgreSQL:

CREATE TABLE graylist (
  "ok"        BOOLEAN                     NOT NULL DEFAULT false,
  "ip"        INET                        NOT NULL,
  "from"      TEXT                        NOT NULL,
  "to"        TEXT                        NOT NULL,
  "ts_entry"  TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT now(),
  "ts_latest" TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT now(),
  "ts_ok"     TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT now(),
  "count"     BIGINT                      NOT NULL DEFAULT 1,
  UNIQUE ("ip", "from", "to"),
  PRIMARY KEY ("ip", "from", "to")
);

CREATE TABLE blackwhitelist (
  "ok"   BOOLEAN NOT NULL DEFAULT false,
  "ip"   INET    NOT NULL,
  "from" TEXT    NOT NULL,
  "to"   TEXT    NOT NULL,
  UNIQUE ("ip", "from", "to"),
  PRIMARY KEY ("ip", "from", "to")
);

1.2. MySQL:

CREATE TABLE graylist (
  ok        BOOL            NOT NULL DEFAULT 0,
  ip        LONGTEXT        NOT NULL,
  src       LONGTEXT        NOT NULL,
  dst       LONGTEXT        NOT NULL,
  ts_entry  DATETIME        NOT NULL,
  ts_latest DATETIME        NOT NULL,
  ts_ok     DATETIME        NOT NULL,
  count     BIGINT UNSIGNED NOT NULL DEFAULT 1,
  UNIQUE (ip(133), src(133), dst(133)),
  PRIMARY KEY (ip(133), src(133), dst(133))
);

CREATE TABLE blackwhitelist (
  ok   BOOL      NOT NULL DEFAULT 0,
  src  LONGTEXT  NOT NULL,
  dst  LONGTEXT  NOT NULL,
  UNIQUE (src(133), dst(133)),
  PRIMARY KEY (src(133), dst(133))
);

CREATE INDEX graylist_ok_idx ON graylist(ok);
CREATE INDEX graylist_ts_entry_idx ON graylist(ts_entry);
CREATE INDEX graylist_ts_latest_idx ON graylist(ts_latest);
CREATE INDEX graylist_ts_ok_idx ON graylist(ts_ok);

=cut

### signal handler, autoflush
$SIG{CHLD} = IGNORE;
$SIG{TERM} = sub {
  syslog("info", "%s", "shutting down...");
  close($listen_socket);
  unlink($config{pidfile});
  exit 0;
};
$| = 1;

### daemonize
$pid = fork();
die "fork(): $!" if ($pid < 0);
exit 0 if $pid;

### open syslog
setlogsock("unix");
openlog("graypold", "pid", "daemon");
close(STDIN);
close(STDOUT);
close(STDERR);
setsid();

# load configfile over defaults
$err = &load_config();
syslog("warning", "%s", "$err") if $err;

### save pid
if (open(FH, "$config{pidfile}")) {
  $pid = <FH>;
  close(FH);
  if (kill($pid, 0)) {
    &byebye("already running (pid $pid)!");
  }
  else {
    syslog("warning", "%s", "removing stale pid file $config{pidfile}");
  }
}
open(FH, ">$config{pidfile}") or &byebye("$config{pidfile}: $!");
print FH $$;
close(FH);

# open socket
if ($config{listen} =~ /^\//) {
  # unix socket
  unlink $config{listen};
  $listen_socket = new IO::Socket::UNIX(
    Local  => $config{listen},
    Type   => SOCK_STREAM,
    Listen => $config{backlog}) or &byebye("socket: $config{listen}: $!");
  ($login, $pass, $uid, $gid, @rest) = getpwnam($config{owner})
    or &byebye("$config{owner}: $!");
  chown $uid, $gid, $config{listen} or &byebye("chown: $config{listen}: $!");
  chmod oct($config{rights}), $config{listen}
    or &byebye("chmod: $config{listen}: $!");
}
else {
  # inet socket
  $listen_socket = new IO::Socket::INET(
    LocalAddr  => $config{listen},
    LocalPort  => $config{port},
    Proto      => "tcp",
    ReuseAddr  => 1,
    Type       => SOCK_STREAM,
    Listen     => $config{backlog})
  or &byebye("socket: $config{listen}:$config{port}: $!");
}

syslog("info", "%s", "starting...");

# main loop
while ($socket = $listen_socket->accept()) {
  $pid = fork();
  &handle_error("fork(): $!") if ($pid < 0);
  if ($pid) {
    close($socket);
    do {
    } while (waitpid(-1, &WNOHANG) > 0);
    next;
  } 

  # child reloads config file
  $err = &load_config();
  syslog("warning", "%s", "$err") if $err;

  # connect to db
  $dbh = &dbopen(\&handle_error);

  # prepare statements
  $primkey = "WHERE $config{db_field_ip}=? AND $config{db_field_from}=? AND ";
  $primkey .= "$config{db_field_to}=?";

  $sql = "SELECT $config{db_field_ok} FROM $config{db_tbl_gray} $primkey";
  $sth{select} = $dbh->prepare($sql);

  $sql = "UPDATE $config{db_tbl_gray} SET $config{db_field_ts_latest}=now(), ";
  $sql .= "$config{db_field_count}=$config{db_field_count} + ? $primkey";
  $sth{update} = $dbh->prepare($sql);

  $sql = "INSERT INTO $config{db_tbl_gray} ($config{db_field_ip}, ";
  $sql .= "$config{db_field_from}, $config{db_field_to}, ";
  $sql .= "$config{db_field_ts_entry}, $config{db_field_ts_latest}, ";
  $sql .= "$config{db_field_ts_ok}) VALUES (?, ?, ?, now(), now(), now())";
  $sth{insert} = $dbh->prepare($sql);

  if ($config{dbtype} eq "Pg") {
    # blackwhite list
    $sql = "SELECT $config{db_field_ok} FROM $config{db_tbl_bw} WHERE ";
    $sql .= "$config{db_field_ip}>>=? AND $config{db_field_from}=? AND ";
    $sql .= "$config{db_field_to}=? ORDER BY $config{db_field_ip} DESC LIMIT 1";
    $sth{select_bw} = $dbh->prepare($sql);
  }
  elsif ($config{dbtype} eq "mysql") {
    # el-cheapo blackwhite list (mysql)
    $sql = "SELECT $config{db_field_ok} FROM $config{db_tbl_bw} WHERE ";
    $sql .= "LENGTH(?)>0 AND $config{db_field_from}=? AND ";
    $sql .= "$config{db_field_to}=? LIMIT 1";
    $sth{select_bw} = $dbh->prepare($sql);
  }

  ### client loop
  while (<$socket>) {
    s/\r|\n//sgio;
    if (/^$/o) {
      ### empty line, end of parameters
      print $socket &check_policy();
      %params = ();
    }
    else {
      ### save parameter
      ($key, $value) = split(/=/, $_, 2);
      $params{$key} = $value;
    }
  }

  # child exits
  %sth = undef;
  $dbh->disconnect();
  closelog();
  exit 0;
}

sub check_policy ()
{
  ### ensure that we are called at the correct stage
  return &choose_action(1) if (($params{request} ne "smtpd_access_policy") ||
                               ($params{protocol_state} ne "RCPT"));

  ### the primary key
  $ip = lc($params{client_address});
  substr($ip, rindex($ip, ".") + 1) = "0";
  $from = lc($params{sender});
  $to = lc($params{recipient});

  if (exists $sth{select_bw}) {
    ### check for entry in blackwhitelist
    ($from_user, $from_dom) = split(/\@/, $from, 2);
    ($to_user, $to_dom) = split(/\@/, $to, 2);

    foreach $f($from, $from_dom, "") {
      foreach $t($to, $to_dom, "") {
        $sth{select_bw}->execute($ip, $f, $t);
        if ($sth{select_bw}->rows == 1) {
          $row = $sth{select_bw}->fetchrow_hashref();
          return &choose_action($row->{$config{db_field_ok}});
        }
      }
    }
  }

  ### check for entry in graylist
  $sth{select}->execute($ip, $from, $to);
  if ($sth{select}->rows == 1) {
    ### found an entry
    $row = $sth{select}->fetchrow_hashref();
    $ok = $row->{$config{db_field_ok}};
    ### update timestamp(s)
    $sth{update}->execute(($ok ? 0 : 1), $ip, $from, $to);
  }
  else {
    $ok = 0;
    ### did not found an entry -> create it
    $sth{insert}->{RaiseError} = 0;
    $sth{insert}->{HandleError} = undef;
    $sth{insert}->execute($ip, $from, $to);
    if ($dbh->err && ($dbh->errstr !~ /$config{db_key_err}/o)) {
      &handle_error($dbh->errstr);
    }
    $sth{insert}->{HandleError} = \&handle_error;
    $sth{insert}->{RaiseError} = 1;
  }

  return &choose_action($ok);
}

sub choose_action ()
{
  return "action=dunno\n\n" if $_[0];
  return "action=defer_if_permit Please try again in several minutes.\n\n";
}

sub handle_error ()
{
  print $socket &choose_action(1);
  %sth = undef;
  $dbh->disconnect() if $dbh;
  &byebye($_[0]);
}

# fatal error
sub byebye ()
{
  $err = shift;
  syslog("err", "%s", $err);
  closelog();
  exit 1;
}
