#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <err.h>
#include <syslog.h>
#include <pwd.h>
#include <grp.h>
#include <sys/time.h>
#include <fcntl.h>
#include <sys/file.h>
#include <alsa/asoundlib.h>

#define VERSION "0.4"
#define CHROOT_DIR "/var/empty"
#define DEFAULT_SOUNDCARD "default"
#define DEFAULT_CONTROL "Master"
#define DEFAULT_DEVICE "/dev/input/by-id/usb-Griffin_Technology__Inc._" \
  "Griffin_PowerMate-event-if00"

#define ERR(fmt, args ...) do { \
  if (debug) err(1, fmt, ## args); \
  else { \
    syslog(LOG_ERR, fmt ": %s\n", ## args, strerror(errno)); \
    exit(1); \
  } \
} while (0)

#define ERRX(fmt, args ...) do { \
  if (debug) errx(1, fmt, ## args); \
  else { \
    syslog(LOG_ERR, fmt "\n", ## args); \
    exit(1); \
  } \
} while (0)

#define ERRS(fmt, args ...) do { \
  if (debug) errx(1, fmt ": %s", ## args, snd_strerror(i)); \
  else { \
    syslog(LOG_ERR, fmt ": %s\n", ## args, snd_strerror(i)); \
    exit(1); \
  } \
} while (0)

#define WARNS(fmt, args ...) do { \
  if (debug) fprintf(stderr, fmt ": %s\n", ## args, snd_strerror(i)); \
  else syslog(LOG_WARNING, fmt ": %s\n", ## args, snd_strerror(i)); \
} while (0)

struct input_event {
    struct timeval time;
    unsigned short type;
    unsigned short code;
    int value;
};

static int changed = 1;

static int elem_callback(snd_mixer_elem_t *elem, unsigned int mask)
{
  elem = elem; mask = mask;
  changed = 1;
  return 0;
}

static char *device = DEFAULT_DEVICE;
static int debug = 0;
static int fd;
static long volume;
static long min_volume;
static long max_volume;
static int knob_vol;

static void set_brightness(int show)
{
  struct input_event ev;
  int brightness;

  /* light intensity isn't linear */
  brightness = (volume - min_volume) * (volume - min_volume) * 255 /
               ((max_volume - min_volume) * (max_volume - min_volume));

  if (brightness < 0) brightness = 0;
  if (brightness > 255) brightness = 255;

  ev.time.tv_sec = 0;
  ev.time.tv_usec = 0;
  ev.type = 4;
  ev.code = 1;
  ev.value = brightness * show;

  if (debug)
    printf("brightness=%i show=%i value=%i\n",
           brightness, show, ev.value);

  if (write(fd, &ev, sizeof(ev)) != sizeof(ev))
    ERR("cannot write to knob device %s", device);
}

static void usage()
{
  puts(
"pomavold V" VERSION " - POwerMAte VOLume Daemon\n\n"
"usage: pomavold [-i <device>] [-s <soundcard>] [-c <control>] [-u <user>]\n"
"                [-w <timeout>] [-d]\n"
"       pomavold -h\n\n"
"  -i <device>       use <device> as knob wheel; default:\n"
"    " DEFAULT_DEVICE "\n"
"  -s <soundcard>    use <soundcard> as soundcard; default: "
DEFAULT_SOUNDCARD "\n"
"  -c <control>      use <control> as volume control; default: "
DEFAULT_CONTROL "\n"
"  -u <user>         run as <user> and chroot to " CHROOT_DIR "\n"
"  -w <timeout>      wait <timeout> seconds before accessing devices\n"
"  -d                run in foreground, debug to stdout, don't log to "
"syslog\n"
"  -h                this help ;-)\n\n"
"examples:\n\n"
"run in foreground as the current user:\n\n"
"  pomavold -i /dev/input/by-id/*Griffin*Technology*PowerMate* -d\n\n"
"to start automatically when the device is plugged in\n"
"add the following to /etc/udev/rules.d/99-local.rules and\n"
"do a `udevadm trigger`:\n\n"
"  DEVPATH==\"/devices/*/usb*/input/input*/event*\",\\\n"
"  ATTRS{product}==\"Griffin PowerMate\",\\\n"
"  RUN=\"/usr/local/libexec/pomavold -u nobody -i %N\"\n"
);
}

int main(int argc, char **argv)
{
  char *user = NULL;
  char *soundcard = DEFAULT_SOUNDCARD;
  char *control = DEFAULT_CONTROL;
  int timeout = 0;

  snd_mixer_t *mixer;
  snd_mixer_selem_id_t *id;
  snd_mixer_elem_t *elem;
  int active;
  int old_active;
  int clear = 0;
  int is_mono;
  long volume_right;

  struct pollfd *pfds = NULL;
  int num_pfds = 0;
  unsigned short *revents = NULL;
  int num_events;

  struct input_event ev;

  struct passwd *pw = NULL;
  pid_t pid;
  int i;

  /* command line */
  while ((i = getopt(argc, argv, "i:s:c:u:w:dh")) != -1) {
    switch (i) {
      case 'i':
        device = optarg;
        break;
      case 's':
        soundcard = optarg;
        break;
      case 'c':
        control = optarg;
        break;
      case 'u':
        user = optarg;
        break;
      case 'w':
        timeout = atoi(optarg);
        break;
      case 'd':
        debug = 1;
        break;
      case 'h':
        usage();
        return 0;
      default:
        fputs("use -h for help\n", stderr);
        return 1;
    }
  }

  if (!debug) {
    /* daemonize */
    pid = fork();
    if (pid < 0) err(1, "fork()");
    if (pid > 0) return 0;

    if (chdir("/") < 0) err(1, "chdir() to / failed");
    if (setsid() < 0) err(1, "setsid()");

    if (pid < 0) err(1, "fork()");
    if (pid > 0) return 0;

    close(0); close(1); close(2);
  }

  /* wait for devices to appear */
  if (timeout > 0) sleep(timeout);

  if (!debug) openlog("pomavold", LOG_NDELAY | LOG_PID, LOG_DAEMON);
  if (user && !(pw = getpwnam(user))) ERRX("no such user: %s", user);

  /* open mixer */
  if ((i = snd_mixer_open(&mixer, 0)))
    ERRS("snd_mixer_open()");
  if ((i = snd_mixer_attach(mixer, soundcard)))
    ERRS("snd_mixer_attach()");
  if ((i = snd_mixer_selem_register(mixer, NULL, NULL)))
    ERRS("snd_mixer_selem_register()");
  if ((i = snd_mixer_load(mixer)))
    ERRS("snd_mixer_load()");

  /* open selected volume control */
  snd_mixer_selem_id_alloca(&id);
  snd_mixer_selem_id_set_index(id, 0);
  snd_mixer_selem_id_set_name(id, control);
  if ((elem = snd_mixer_find_selem(mixer, id)) == NULL)
    ERRX("snd_mixer_find_selem(): no such control: %s", control);

  i = snd_mixer_selem_get_playback_volume_range(elem, &min_volume,
                                                &max_volume);
  if (i) ERRS("snd_mixer_selem_get_playback_volume_range()");
  if (max_volume <= min_volume)
    ERRX("%s/%s: volume insanity: minvol=%li >= maxvol=%li\n",
         soundcard, control, min_volume, max_volume);

  is_mono = snd_mixer_selem_is_playback_mono(elem);

  if (debug)
    printf("%s/%s: minvol=%li maxvol=%li mono=%i\n",
           soundcard, control, min_volume, max_volume, is_mono);

  snd_mixer_elem_set_callback(elem, elem_callback);

  /* open & lock knob wheel */
  if ((fd = open(device, O_RDWR)) < 0)
    ERR("cannot open %s for reading and writing", device);

  if (flock(fd, LOCK_EX|LOCK_NB) != 0)
    ERR("cannot lock %s exclusively", device);

  if (user) {
    /* drop privileges, chroot */
    if (initgroups(user, pw->pw_gid) < 0) ERR("initgroups()");
    if (chroot(CHROOT_DIR) < 0) ERR("chroot() to " CHROOT_DIR " failed");
    if (setgid(pw->pw_gid) < 0) ERR("setgid()");
    if (setuid(pw->pw_uid) < 0) ERR("setuid()");
  }

  if (debug)
    printf("starting: device=%s soundcard=%s/%s\n",
           device, soundcard, control);
  else
    syslog(LOG_INFO, "starting: device=%s soundcard=%s/%s\n",
                     device, soundcard, control);

  /* main loop */
  for (;;) {
    if (changed) {
      /* some external program changed the volume */
      /* adjust brightness of knob wheel          */
      changed = 0;
      old_active = active;

      /* get current volume */
      i = snd_mixer_selem_get_playback_volume(elem, SND_MIXER_SCHN_MONO,
                                              &volume);
      if (i) ERRS("snd_mixer_selem_get_playback_volume()");

      if (!is_mono) {
        i = snd_mixer_selem_get_playback_volume(elem,
                                                SND_MIXER_SCHN_FRONT_RIGHT,
                                                &volume_right);
        if (i) ERRS("snd_mixer_selem_get_playback_volume()");
        if (volume_right > volume) volume = volume_right;
      }

      knob_vol = (volume * 100) / (max_volume - min_volume);

      /* get current active state (active == not muted) */
      i = snd_mixer_selem_get_playback_switch(elem, SND_MIXER_SCHN_MONO,
                                              &active);
      if (i) WARNS("snd_mixer_selem_get_playback_switch()");

      if (debug)
        printf("changed: volume=%li active=%i knob_vol=%i\n",
               volume, active, knob_vol);

      if (old_active && !active) set_brightness(0); // externally muted
      else {
        set_brightness(1);
        clear = !active;
      }
    }

    i = snd_mixer_poll_descriptors_count(mixer) + 1;
    if (i != num_pfds) {
      /* make room for mixer descriptors */
      num_pfds = i;

      if ((pfds = realloc(pfds, num_pfds * sizeof(pfds[0]))) == NULL)
        ERRX("no memory for %i pollfd(s)", num_pfds);

      if ((revents = realloc(revents, num_pfds * sizeof(revents[0]))) == NULL)
        ERRX("no memory for %i revent(s)", num_pfds);
    }

    /* load mixer descriptors */
    if ((i = snd_mixer_poll_descriptors(mixer, &pfds[1], num_pfds - 1)) < 0)
      ERRS("snd_mixer_poll_descriptors()");

    pfds[0].fd = fd;
    pfds[0].events = POLLIN;

    /* wait for some events */
    num_events = poll(pfds, num_pfds, (clear ? 1000 : -1));

    if (!num_events && clear) {
      /* set_brightness was called earlier, sound is muted */
      /* so turn off light */
      set_brightness(0);
      clear = 0;
    }

    if (pfds[0].revents & (POLLERR | POLLHUP | POLLNVAL))
      ERRX("poll error on knob device %s", device);

    if (pfds[0].revents & POLLIN) {
      /* someone touched the knob wheel */
      num_events--;

      if (read(fd, &ev, sizeof(ev)) != sizeof(ev))
        ERRX("cannot read from knob device %s", device);

      if (debug)
        printf("type=%hu code=%hu value=%i\n", ev.type, ev.code, ev.value);

      if ((ev.type == 2) && (ev.code == 7)) {
        /* knob rotated */
        knob_vol += ev.value;
        if (knob_vol < 0) knob_vol = 0;
        if (knob_vol > 100) knob_vol = 100;

        volume = min_volume + (knob_vol * (max_volume - min_volume)) / 100;
        if (volume < min_volume) volume = min_volume;
        if (volume > max_volume) volume = max_volume;

        if ((i = snd_mixer_selem_set_playback_volume_all(elem, volume)))
          ERRS("snd_mixer_selem_set_playback_volume_all()");

        if (debug)
          printf("rotated: knob_vol=%i volume=%li\n", knob_vol, volume);

        set_brightness(1);
        clear = !active;
      }

      if ((ev.type == 1) && (ev.code == 256) && (ev.value == 1)) {
        /* knob pressed */
        active = !active;

        i = snd_mixer_selem_set_playback_switch(elem, SND_MIXER_SCHN_MONO,
                                                active);
        if (i) WARNS("snd_mixer_selem_set_playback_switch()");

        if (debug)
          printf("pressed: active=%i\n", active);

        set_brightness(active);
      }
    }

    if (num_events > 0) {
      /* some external program changed the volume */
      /* consume events and trigger callback      */
      if ((i = snd_mixer_poll_descriptors_revents(mixer, &pfds[1],
                                                  num_pfds - 1, revents)))
        ERRS("snd_mixer_poll_descriptors_revents()");

      for (i = 1; i < num_pfds; i++)
        if (pfds[i].revents & (POLLERR | POLLHUP | POLLNVAL))
          ERRX("poll error on mixer");

      snd_mixer_handle_events(mixer);
    }
  }

  /* not reached */
  return 0;
}
