/*
 * $Header: /export/cvs/www.ogris.de/mount/mount.c,v 1.2 2002/11/10 22:55:45 fjo Exp $
 */

#include <string.h>
#include <stdlib.h>
#include <mntent.h>
#include <limits.h>
#include <sys/mount.h>
#include <fcntl.h>
#include <errno.h>

#define VERSION "0.01 (2002-08-16)"

#define _PATH_MNTTAB	"/etc/fstab"
#ifdef _TMPFS
#define _PATH_MOUNTED	"/var/run/mtab"  /* tmpfs will be mounted on /var/run */
#endif

/* fstab */
#define MNTOPT_AUTO	"auto"
#define MNTOPT_NOUSER	"nouser"
#define MNTOPT_USER	"user"

/* command line */
#define MNTOPT_REMOUNT	"remount"

/* fstab & command line */
#define MNTOPT_NOATIME	"noatime"
#define MNTOPT_ATIME	"atime"
#define MNTOPT_ASYNC	"async"
#define MNTOPT_SYNC	"sync"
#define MNTOPT_NODEV	"nodev"
#define MNTOPT_DEV	"dev"
#define MNTOPT_NOEXEC	"noexec"
#define MNTOPT_EXEC	"exec"

static void print ( int count, char *arg, ... )
{
  char **tmp = &arg;
  for (; count; count--, tmp++) write(1, *tmp, strlen(*tmp));
}

static void die ( int count, char *arg, ... )
{
  char **tmp = &arg;
  write(2, "mount: ", 7);
  for (; count; count--, tmp++) write(2, *tmp, strlen(*tmp));
  write(2,"\n",1);
  exit(1);
}

static void fprint ( int fd, int count, char *arg, ... )
{
  char **tmp = &arg;
  for (; count; count--, tmp++) write(fd, *tmp, strlen(*tmp));
}

unsigned int fmt_ulong(char *dest,unsigned long i) {
  register unsigned long len,tmp,len2;
  for (len=1, tmp=i; tmp>9; ++len) tmp/=10;
  if (dest)
    for (tmp=i, dest+=len, len2=len+1; --len2; tmp/=10)
      *--dest = (tmp%10)+'0';
  return len;
}


static void usage ()
{
  print(1, "not implemented yet\n");
  exit(1);
}

static void version ()
{
  print(3, "fjo's mount ", VERSION, "\n");
  exit(1);
}

static char test_param (const char *haystack, const char needle,
                       unsigned int *param, unsigned int success)
{
  if (strchr(haystack, needle)) {
    *param = success;
    return 1;
  }
  return 0;
}

static char set_opts (const char *haystack, const char *needle,
                      unsigned int *options, unsigned int success)
{
  size_t len = strlen(needle);
  char *c = strstr(haystack, needle); 
  if (c != NULL) {
    *options = success;
    if (*(c+len) == ',') len++;
    memmove(c, c+len, strlen(haystack)-len+1);    /* +1 for the trailing '\n' */
    return 1;
  }
  return 0;
}

/*
 * returns -1 for I/O error
 * returns 0 for eof
 * returns count of bytes read by default
 */
int read_line (int fd, char *buf, int size)
{
  int ret, count;
  for (count=0; count<size; count++, buf++) {
    ret = read(fd, buf, 1);
    if (ret<0) return -1;
    if (!ret) return 0;
    if (*buf=='\n') return ++count;
  }
  return ++count;
}
  
/*
 * returns -1 for an empty line
 * returns 0 for a syntax error
 * returns 1 for a valid data line
 */
static signed char parse_mntent (struct mntent *m, char *buf, int size)
{
  register int pos;
  int num=-1;
  char space=1;

  for (pos=0; pos<size; pos++, buf++) {
    switch (*buf) {
       case ' ':
       case '\t': *buf=0; space=1; break;
       case '#':
       case '\n': *buf=0;
                  if (num==-1) return -1;
                  if (num<5) return 0;
                  return 1;
                  break;
       default:
         if (space) {
           space=0;
           ++num;
           switch (num) {
             case 0: m->mnt_fsname = buf; break;
             case 1: m->mnt_dir = buf; break;
             case 2: m->mnt_type = buf; break;
             case 3: m->mnt_opts = buf; break;
             case 4:
             case 5: break;
           }
         }
    }
  }
  if (num==-1) return -1;
  if (num<5) return 0;
  return 1;
}

static int open_or_die (char *pathname, int flags)
{
  int fd = open(pathname, flags);
  if (fd == -1) die(2, "can not open ", pathname);
  return fd;
}

static void close_or_die (int fd, char *pathname)
{
  if (close(fd) == -1) die(2, "can not close ", pathname);
}

static char parse_mntfile (char *mntfile, struct mntent *m, char* path)
{
  unsigned int i, lines = 0;
  char buf[PATH_MAX+1], *c;
  int fd = open_or_die(mntfile, O_RDONLY);

  for (;;) {
    i = read_line(fd, &buf[0], PATH_MAX);
    if (i==-1) die(2, "can not read ", mntfile);
    if (!i) {
      close_or_die(fd,mntfile);
      return 0;
    }
    buf[i]=0;
    lines++;
    if (i && !parse_mntent(m, &buf[0], i)) {
      if ((c = alloca(fmt_ulong(0, lines))) == NULL)
        die(1, "can not allocate memory");
      fmt_ulong(c, lines);
      die(4, "syntax error in ", mntfile, " on line ", c);
    }
    if (!strcmp(path, m->mnt_fsname) || !strcmp(path, m->mnt_dir))
      return 1;
  }
}

static void dump_mntfile (char *mntfile) {
  struct mntent m;
  unsigned int i, lines = 0;
  char buf[PATH_MAX+1], *c;
  int fd = open_or_die(mntfile, O_RDONLY);

  for (;;) {
    i = read_line(fd, &buf[0], PATH_MAX);
    if (i==-1) die(2, "can not read ", mntfile);
    if (!i) {
      close_or_die(fd,mntfile);
      return;
    }
    buf[i]=0;
    lines++;
    if (i && !parse_mntent(&m, &buf[0], i)) {
      if ((c = alloca(fmt_ulong(0, lines))) == NULL)
        die(1, "can not allocate memory");
      fmt_ulong(c, lines);
      die(4, "syntax error in ", mntfile, " on line ", c);
    }
    print(8, m.mnt_fsname, " on ", m.mnt_dir, " type ", m.mnt_type,
             " (", m.mnt_opts, ")\n");
  };
}

int main (int argc, char *argv[])
{
  struct mntent m, tmp;
  unsigned int i, fstype = 0, device = 0, dir = 0, options = 0,
               flags = 0, verbose = 0, all = 0, Fork = 0, fake = 0,
               nomtab = 0;
  char suid = (getuid()!=geteuid());

  if (argc == 1) {
    dump_mntfile(_PATH_MOUNTED);
    return 0;
  }

  for (i=1; i<argc; i++) {
    if (!strncmp(argv[i], "-", 1)) {
      if (strchr(argv[i], 'h')) usage();
      if (strchr(argv[i], 'V')) version();

      test_param(argv[i], 'v', &verbose, 1);
      test_param(argv[i], 'a', &all, 1);
      test_param(argv[i], 'F', &Fork, 1);
      test_param(argv[i], 'f', &fake, 1);
      test_param(argv[i], 'n', &nomtab, 1);
      test_param(argv[i], 'r', &flags, flags | MS_RDONLY);
      test_param(argv[i], 'w', &flags, flags & !MS_RDONLY);

      if (strchr(argv[i], 't')) {
        if (++i == argc) die(1, "-t needs fstype");
        fstype = i;
        continue;
      }

      if (strchr(argv[i], 'o')) {
        options = ++i;
        if (options == argc) die(1, "-o needs options");
        set_opts(argv[options], MNTOPT_DEFAULTS, &flags, 0);
        set_opts(argv[options], MNTOPT_RW, &flags, flags & !MS_RDONLY);
        set_opts(argv[options], MNTOPT_RO, &flags, flags | MS_RDONLY);
        set_opts(argv[options], MNTOPT_REMOUNT, &flags, flags | MS_REMOUNT);

        set_opts(argv[options], MNTOPT_NOATIME, &flags, flags | MS_NOATIME);
        set_opts(argv[options], MNTOPT_ATIME, &flags, flags & !MS_NOATIME);

        set_opts(argv[options], MNTOPT_NODEV, &flags, flags | MS_NODEV);
        set_opts(argv[options], MNTOPT_DEV, &flags, flags & !MS_NODEV);

        set_opts(argv[options], MNTOPT_NOEXEC, &flags, flags | MS_NOEXEC);
        set_opts(argv[options], MNTOPT_EXEC, &flags, flags & !MS_NOEXEC);

        set_opts(argv[options], MNTOPT_ASYNC, &flags, flags & !MS_SYNCHRONOUS);
        set_opts(argv[options], MNTOPT_SYNC, &flags, flags | MS_SYNCHRONOUS);

        set_opts(argv[options], MNTOPT_NOSUID, &flags, flags | MS_NOSUID);
        set_opts(argv[options], MNTOPT_SUID, &flags, flags & !MS_NOSUID);
      }
      continue;
    }

    if (device) {
      if (dir) die(2, "useless parameter: ", argv[i]);
      dir=i;
    }
    else
      device=i;
  }

  if (!device) usage();

  if (realpath(argv[device], argv[device]) == NULL)
    die(2, "no such file or directory: ", argv[device]);
  if (dir) {
    if (realpath(argv[dir], argv[dir]) == NULL)
      die(2 ,"no such file or directory: ", argv[dir]);
  }
  else if (!parse_mntfile(_PATH_MNTTAB, &m, argv[device]))
    die(3, argv[device], " not found in ", _PATH_MNTTAB);

  if ( (suid) &&
       ((strstr(m.mnt_opts, MNTOPT_NOUSER) != NULL) ||
        (strstr(m.mnt_opts, MNTOPT_USER) == NULL)) )
    die(1,"only root can do that");

  if (parse_mntfile(_PATH_MOUNTED, &tmp, argv[device]))
    die(4, "according to " _PATH_MOUNTED ", ", tmp.mnt_fsname,
        " is already mounted on ", tmp.mnt_dir);

  parse_mntfile(_PATH_MNTTAB, &m, argv[device]);

  if (verbose) print(7 ,"mounting ", (dir?argv[device]:m.mnt_fsname),
                     " on ", (dir?argv[dir]:m.mnt_dir), " type ",
                     (fstype?argv[fstype]:m.mnt_type), " ()\n");
  if (fake)
    print(1, "just kidding\n");
  else {
    if ( (mount((dir?argv[device]:m.mnt_fsname),
                (dir?argv[dir]:m.mnt_dir),
                (fstype?argv[fstype]:m.mnt_type),
                flags, "")) && (errno == EROFS) ) {
      fprint(2, 2, (dir?argv[device]:m.mnt_fsname),
             " is write-protected, mounting read-only\n");
      flags |= MS_RDONLY;
      if (mount((dir?argv[device]:m.mnt_fsname),
                (dir?argv[dir]:m.mnt_dir),
                (fstype?argv[fstype]:m.mnt_type),
                flags,
                ""))
        die(2, "mount failed: ", strerror(errno));
    } else die(2, "mount failed: ", strerror(errno));
    if (!nomtab) {
      int fd = open_or_die(_PATH_MOUNTED, O_WRONLY|O_CREAT|O_APPEND);
      fprint(fd, 8, (dir?argv[device]:m.mnt_fsname), " ",
                    (dir?argv[dir]:m.mnt_dir), " ",
                    (fstype?argv[fstype]:m.mnt_type), " ",
                    (options?argv[options]:m.mnt_opts), " 0 0\n");
      close_or_die(fd, _PATH_MOUNTED);
    }
  }
  return(0);
}
