#include <unistd.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/wait.h>
#include <resolv.h>
#include <netdb.h>
#include <string.h>
#include <errno.h>
#include <malloc.h>
#include <fcntl.h>
#include <pwd.h>

#define TIMEOUT 10   /* seconds */
#ifndef _PATH_TMP
#  define _PATH_TMP "/tmp/"
#endif
#define CONFIG "/etc/lpdpipe.conf"
#define BUFFER_SIZE 4096
#define USAGE "lpdpipe - prints files on a remote lpd server\n\n" \
"lpdpipe [-c <#n>] [-f <configfile>] [-h] [-l] [-q <queue>] [file ...]\n\n" \
"  -c      print each file n times (default: 1)\n" \
"  -f      use configfile as configfile (default: /etc/lpdpipe.conf)\n" \
"  -h      print this help screen\n" \
"  -l      do not print anything, display status of the queue\n" \
"  -q      use queue for printing or displaying status\n" \
"          (default: first queue mentioned in configfile)\n" \
"  file    print these documents; if not specified, read file from stdin\n"

void print (const int fd, const char *s)
{
  if (s) write(fd, s, strlen(s));
}

void die (const char *s1, const char *s2)
{
  print(2, s1);
  if (s2) {
    write(2, ": ", 2);
    print(2, s2);
  }
  write(2, "\n", 1);
  exit(1);
}

void hint (const char *opt)
{
  die(opt, "requires an additional option");
}

void nomem ()
{
  die("no memory!", NULL);
}

unsigned long readfd (int fd, char **c, unsigned long max)
{
  static ssize_t tmp;
  static unsigned long len;

  alarm(TIMEOUT);
  for (len = 0;;) {
    *c = realloc(*c, len + BUFFER_SIZE);
    if (!*c) nomem();
    tmp = read(fd, *c + len, BUFFER_SIZE);
    if (tmp < 0) die("read()", strerror(errno));
    if (!tmp) break;
    len += tmp;
    if (max) if (len >= max) break;
  }
  alarm(0);
  return len;
}

unsigned long stralloc_cpy (char** dst, const char* src)
{
  static unsigned long len;

  if (!dst || !src) return 0;
  len = strlen(src) + 1;
  *dst = realloc(*dst, len);
  if (!*dst) nomem();
  memcpy(*dst, src, len);
  return len - 1;
}

unsigned long stralloc_cat (char** dst, const char* src)
{
  static unsigned long len1, len2;

  if (!dst || !src) return 0;
  if (!*dst) return 0;
  len1 = strlen(*dst);
  len2 = strlen(src) + 1;
  *dst = realloc(*dst, len1 + len2);
  if (!*dst) nomem();
  memcpy(*dst + len1, src, len2);
  return len1 + len2 - 1;
}

char *ultoa (const unsigned long l)
{
  static char buf[32], *c;
  static unsigned long tmp;
  
  tmp = l;
  c = &buf[0];
  do {
    tmp /= 10;
    ++c;
  } while (tmp);
  tmp = l;
  *(c--) = 0;
  do {
    *(c--) = '0' + tmp % 10;
    tmp /= 10;
  } while (tmp);
  return &buf[0];
}

void lprcmd (const int sock, const char *cmd, const char *param)
{
  static unsigned long len;
  static char *buf = NULL;

  stralloc_cpy(&buf, cmd);
  stralloc_cat(&buf, param);
  len = stralloc_cat(&buf, "\n");
  write(sock, buf, len);
}

int seperator (const char c)
{
  return (c == ' ' || c == '\t' || c == '=' || c == ':');
}

int main (int argc, char **argv)
{
  struct passwd *pw;
  struct hostent *he; 
  struct sockaddr_in sin;
  struct stat st;
  unsigned long ctrl_len, data_len, pid_len;
  pid_t procid;
  int sock, i, fh, pfd0[2], pfd1[2], copies = 1, files = 0, listonly = 0, cnt;
  char *host = NULL, *queue = NULL, *fn = NULL, *tmpdir = NULL, *filter = NULL,
    *config = CONFIG, *args[3], **filenames = NULL, *reqqueue = NULL,
    hostname[BUFFER_SIZE], *c = NULL, *ctrl = NULL, *ctrl_hdr = NULL,
    *user = NULL, *data = NULL, *data_hdr = NULL, *pid = NULL, job[4],
    *left, *right, *middle, *input = NULL, *remote_queue = NULL;

#ifdef STRICT_RFC1179
  int port;
#endif

  /*
   * parse command line options
   */
  for (i = 1; i < argc; ++i) {
    if (*argv[i] == '-') {
      switch (*(argv[i] + 1)) {
        case 'h':
          die("usage", USAGE);

        case 'l':
          listonly = 1;
          break;

        case 'c':
          if (!argv[++i]) hint("-c");
          copies = atoi(argv[i]);
          if (!copies) return 0;
          break;

        case 'f':
          if (!argv[++i]) hint("-f");
          config = argv[i];
          break;

        case 'q':
          if (!argv[++i]) hint("-q");
          reqqueue = argv[i];
          break;

        default:
          die("unknown option", argv[i]);
      }
    }
    else {
      filenames = realloc(filenames, (files + 1) * sizeof(char*));
      if (!filenames) nomem();
      filenames[files++] = argv[i];
    }
  }

  /*
   * parse config file
   */
  fh = open(config, O_RDONLY);
  if (fh < 0) die(config, strerror(errno));
  if (fstat(fh, &st)) die(config, strerror(errno));
  input = mmap(0, st.st_size, PROT_READ | PROT_WRITE, MAP_PRIVATE, fh, 0);
  if (input == MAP_FAILED) die("mmap()", strerror(errno));

  for (left = input;;) {
    right = strchr(left, '\r');
    if (!right) right = strchr(left, '\n');
    if (!right) {
      *((char*) input + st.st_size - 1) = '\0';
      die("incomplete line", left);
    }
    *right = '\0';
    for (middle = left; !seperator(*middle); ++middle)
      if (middle >= right) die("invalid line", left);
    for (; seperator(*middle); ++middle)
      if (middle >= right) die("invalid line", left);

    switch (*left) {
      case 'q':
        host = NULL;
        filter = NULL;
        remote_queue = NULL;
        queue = middle;
        break;

      case 'h':
        host = middle;
        break;

      case 'f':
        filter = middle;
        break;

      case 'r':
        remote_queue = middle;
        break;

      default:
        die("unknown configuration option", left);
    }
    if (queue && host && filter) {
      if (!reqqueue) break;
      if (!strcmp(queue, reqqueue)) break;
    }

    for (left = ++right; *left == '\r' || *left == '\n'; ++left)
      ;;

    if (left >= input + st.st_size) break;
  }

  if (!queue) die(config, "no queues found");
  if (reqqueue)
    if (strcmp(reqqueue, queue)) die(reqqueue, "no such queue");

  if (!host) die(queue, "missing host address");
  if (!filter) die(queue, "missing filter");

  if (!remote_queue) remote_queue = queue;

  if (gethostname(&hostname[0], BUFFER_SIZE - 1))
    die("gethostname()", strerror(errno));

  he = gethostbyname(host);
  if (!he) die("unknown host", host);

  user = getenv("USER");
  if (!user) {
    pw = getpwuid(getuid());
    user = pw->pw_name;
  }

  for (data_len = 0, input = NULL;;) {
    if (!listonly) {
      if (files) {
        fn = filenames[files - 1];
        fh = open(fn, O_RDONLY);
        if (fh < 0) die(fn, strerror(errno));
        if (fstat(fh, &st)) die(fn, strerror(errno));
        input = mmap(0, st.st_size, PROT_READ, MAP_PRIVATE, fh, 0);
        if (input == MAP_FAILED) die("mmap()", strerror(errno));
      }
      else st.st_size = readfd(0, &input, 0);

      if (pipe(pfd0) || pipe(pfd1)) die("pipe()", strerror(errno));
      procid = fork();
      if (procid < 0) die("fork()", strerror(errno));
      if (!procid) {
        close(pfd0[1]);
        close(pfd1[0]);
        stralloc_cpy(&tmpdir, _PATH_TMP);
        stralloc_cat(&tmpdir, queue);
        umask(0);
        mkdir(tmpdir, 01777);
        stralloc_cat(&tmpdir, "/");
        stralloc_cat(&tmpdir, (user)?(user):(ultoa(getpid())));
        if (dup2(pfd0[0], 0) < 0) die("dup()", strerror(errno));
        if (dup2(pfd1[1], 1) < 0) die("dup()", strerror(errno));
        args[0] = filter;
        args[1] = tmpdir;
        args[2] = NULL;
        execve(filter, args, (char*[]) { "CONTROL=1", NULL });
        die("execve()", strerror(errno));
      }
      close(pfd0[0]);
      close(pfd1[1]);
      write(pfd0[1], input, st.st_size);
      close(pfd0[1]);
      if (files) {
        if (munmap(input, st.st_size)) die("munmap()", strerror(errno));
        close(fh);
      }

      data_len = readfd(pfd1[0], &data, 0);
      if (waitpid(procid, &i, 0) != procid) die("waitpid()", strerror(errno));
      if (WIFEXITED(i)) {
        if (WEXITSTATUS(i)) die(filter, "returned error status");
      }
      else die(filter, "exited abnormally");
      close(pfd1[0]);

      pid_len = stralloc_cpy(&pid, ultoa(procid));
      if (pid_len >= 3) memmove(job, pid + pid_len - 3, 4);
      else {
        memset(job, '0', 3);
        memmove(&job[3 - pid_len], pid, pid_len);
      }
      job[3] = '\0';

      stralloc_cpy(&ctrl, "H");
      stralloc_cat(&ctrl, &hostname[0]);

      if (user) {
        stralloc_cat(&ctrl, "\nM");
        stralloc_cat(&ctrl, user);
      }

      stralloc_cat(&ctrl, "\nN");

      if (files) stralloc_cat(&ctrl, fn);
      else stralloc_cat(&ctrl, "<stdin>");

      stralloc_cat(&ctrl, "\nP");
      stralloc_cat(&ctrl, (user)?(user):(ultoa(getpid())));
      stralloc_cat(&ctrl, "\nldfA");
      stralloc_cat(&ctrl, job);
      stralloc_cat(&ctrl, &hostname[0]);
      ctrl_len = stralloc_cat(&ctrl, "\n\n");
      *(ctrl + ctrl_len - 1) = '\0';

      stralloc_cpy(&ctrl_hdr, ultoa(ctrl_len - 1));
      stralloc_cat(&ctrl_hdr, " cfA");
      stralloc_cat(&ctrl_hdr, job);
      stralloc_cat(&ctrl_hdr, &hostname[0]);

      stralloc_cpy(&data_hdr, ultoa(data_len));
      stralloc_cat(&data_hdr, " dfA");
      stralloc_cat(&data_hdr, job);
      stralloc_cat(&data_hdr, &hostname[0]);
    }

    for (cnt = copies; cnt; --cnt) {
      sock = socket(AF_INET, SOCK_STREAM, 0);
      if (!sock) die("socket()", strerror(errno));

      memset(&sin, 0, sizeof(sin));

#ifdef STRICT_RFC1179
      for (port = 721; port <= 731; ++port) {
        sin.sin_port = htons(port);
        if (!bind(sock, (struct sockaddr*) &sin, sizeof(sin))) break;
      }
      if (port > 731) die("bind()", strerror(errno));
#endif

      memset(&sin, 0, sizeof(sin));
      memcpy((char*) &sin.sin_addr, he->h_addr, he->h_length);
      sin.sin_port = htons(515);
      sin.sin_family = PF_INET;

      if (connect(sock, (struct sockaddr*) &sin, sizeof(sin)))
        die(host, strerror(errno));

      if (listonly) {
        lprcmd(sock, "\4", remote_queue);
        ctrl_len = readfd(sock, &c, 0);
        write(1, c, ctrl_len);
        close(sock);
        return 0;
      }

      lprcmd(sock, "\2", remote_queue);
      readfd(sock, &c, 1);
      if (*c) die(remote_queue, "does not like receive printer job");

      lprcmd(sock, "\2", ctrl_hdr);
      readfd(sock, &c, 1);
      if (*c) die(remote_queue, "does not like receive control file");

      write(sock, ctrl, ctrl_len);
      readfd(sock, &c, 1);
      if (*c) die(remote_queue, "does not like control file");

      lprcmd(sock, "\3", data_hdr);
      readfd(sock, &c, 1);
      if (*c) die(remote_queue, "does not like receive data file");

      write(sock, data, data_len);
      write(sock, "\0", 1);
      readfd(sock, &c, 1);
      if (*c) die(remote_queue, "does not like data file");

      close(sock);
    }

    --files;
    if (files <= 0) break;
  }

  return 0;
}
