import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

import javax.crypto.*;
import javax.crypto.spec.*;

class Job {
  static enum MODE { IMPORT, EXPORT };
  MODE mode;
  File file;
  String name;
  SecretKeySpec sks;
  boolean has_password;

  Job (MODE m, File f, String n, String p)
  {
    mode = m;
    try {
      file = new File(f.getCanonicalPath());
    } catch (Exception e) {
      file = f;
    }
    name = n;

    if (p.length() > 0) {
      sks = new SecretKeySpec(AES_align(p).substring(0, 16).getBytes(), "AES");
      has_password = true;
    }
    else
      has_password = false;
  }

  static String AES_align (String s)
  {
    return s.concat("                ".substring(0, 16 - (s.length() % 16)));
  }

  MODE getMode () { return mode; }
  File getFile () { return file; }
  String getName () { return name; }
  Boolean hasPassword () { return has_password; }

  Cipher getCipher () throws Exception
  {
    Cipher cipher = Cipher.getInstance("AES");
    cipher.init((mode == MODE.IMPORT ? Cipher.ENCRYPT_MODE : Cipher.DECRYPT_MODE), sks);
    return cipher;
  }

  public String toString ()
  {
    switch (mode) {
      case IMPORT:
        return "Import " + file.getAbsolutePath();
      case EXPORT:
        return "Export " + name + " to " + file.getAbsolutePath();
    }
    return "";
  }
}

class SafeTaskListModelRemove extends Thread {
  DefaultListModel taskListModel;
  Job job;

  SafeTaskListModelRemove (DefaultListModel m, Job j) {
    taskListModel = m;
    job = j;
  }

  public void run () {
    taskListModel.removeElement(job);
  }
}

class lechunk implements ActionListener, Runnable, ListSelectionListener, FilenameFilter
{
  static final int CHUNK_SIZE = 512 * 1024 * 1024;
  static final File REPOSITORY_DIR = new File(".lechunk");
  static enum TASKLIST_MODE { PUSH, POP };

  JFrame mainFrame;
  JProgressBar progressBar;
  JButton cancelButton, exportButton, deleteButton;
  JList taskList, repositoryList;
  JPasswordField pwdField;
  DefaultListModel taskListModel, repositoryListModel;
  Thread worker;
  long currentSize, bytesSoFar, startTime;
  JFileChooser importChooser, exportChooser;
  boolean brokenJFileChooser;

  lechunk()
  {
    brokenJFileChooser = System.getProperty("os.name").equals("Mac OS X");
    exportChooser = new JFileChooser();
    if (brokenJFileChooser) {
      JPanel jp = new JPanel();
      jp.setLayout(new BoxLayout(jp, BoxLayout.Y_AXIS));
      String[] msg = {
        "You are running a",
        "broken operating system.",
        "Do not select the",
        "destination directory.",
        "Change into it!"
      };
      for (int i = 0; i < msg.length; i++) {
        JLabel jl = new JLabel(msg[i]);
        jl.setAlignmentX(Component.CENTER_ALIGNMENT);
        jp.add(jl);
      }
      exportChooser.setAccessory(jp);
    }
    exportChooser.setMultiSelectionEnabled(false);
    exportChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

    importChooser = new JFileChooser();
    importChooser.setMultiSelectionEnabled(true);
    importChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);

    JPanel mainPanel = new JPanel();
    mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.LINE_AXIS));
    JButton button = new JButton("Import");
    button.setActionCommand("Import");
    button.addActionListener(this);
    button.setAlignmentY(Component.CENTER_ALIGNMENT);
    button.setAlignmentX(Component.CENTER_ALIGNMENT);
    mainPanel.add(button);
    exportButton = new JButton("Export");
    exportButton.setActionCommand("Export");
    exportButton.addActionListener(this);
    exportButton.setAlignmentY(Component.CENTER_ALIGNMENT);
    exportButton.setAlignmentX(Component.CENTER_ALIGNMENT);
    exportButton.setEnabled(false);
    mainPanel.add(exportButton);
    deleteButton = new JButton("Delete");
    deleteButton.setActionCommand("Delete");
    deleteButton.addActionListener(this);
    deleteButton.setAlignmentY(Component.CENTER_ALIGNMENT);
    deleteButton.setAlignmentX(Component.CENTER_ALIGNMENT);
    deleteButton.setEnabled(false);
    mainPanel.add(deleteButton);
    mainPanel.setAlignmentY(Component.CENTER_ALIGNMENT);
    mainPanel.setAlignmentX(Component.CENTER_ALIGNMENT);

    repositoryListModel = new DefaultListModel();
    repositoryList = new JList(repositoryListModel);
    repositoryList.setLayoutOrientation(JList.VERTICAL_WRAP);
    repositoryList.addListSelectionListener(this);

    JPanel panel = new JPanel();
    panel.setLayout(new BoxLayout(panel, BoxLayout.PAGE_AXIS));
    panel.setBorder(new TitledBorder("Repository"));
    panel.add(new JScrollPane(repositoryList));
    panel.add(Box.createVerticalStrut(10));
    panel.add(mainPanel);
    panel.add(Box.createVerticalStrut(10));

    progressBar = new JProgressBar(0, 0);
    progressBar.setValue(0);
    progressBar.setStringPainted(false);
    progressBar.setAlignmentY(Component.CENTER_ALIGNMENT);
    progressBar.setAlignmentX(Component.CENTER_ALIGNMENT);
    panel.add(progressBar);

    panel.setAlignmentY(Component.CENTER_ALIGNMENT);
    panel.setAlignmentX(Component.CENTER_ALIGNMENT);

    JPanel panel2 = new JPanel();
    panel2.setLayout(new BoxLayout(panel2, BoxLayout.PAGE_AXIS));
    panel2.setBorder(new TitledBorder("Password"));
    pwdField = new JPasswordField(1);
    pwdField.setAlignmentY(Component.CENTER_ALIGNMENT);
    pwdField.setAlignmentX(Component.CENTER_ALIGNMENT);
    pwdField.setMinimumSize(new Dimension(50, 20));
    pwdField.setMaximumSize(new Dimension(500, 20));
    pwdField.setPreferredSize(new Dimension(100, 20));
    panel2.add(pwdField);
    panel2.setAlignmentY(Component.CENTER_ALIGNMENT);
    panel2.setAlignmentX(Component.CENTER_ALIGNMENT);

    JPanel panel3 = new JPanel();
    panel3.setLayout(new BoxLayout(panel3, BoxLayout.PAGE_AXIS));
    panel3.add(panel);
    panel3.add(Box.createVerticalStrut(15));
    panel3.add(panel2);

    mainPanel = new JPanel();
    mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.LINE_AXIS));
    mainPanel.add(panel3);

    taskListModel = new DefaultListModel();
    taskList = new JList(taskListModel);
    taskList.addListSelectionListener(this);
    panel = new JPanel();
    panel.setLayout(new BoxLayout(panel, BoxLayout.PAGE_AXIS));
    panel.setBorder(new TitledBorder("Tasks"));
    panel.add(new JScrollPane(taskList));
    panel.add(Box.createVerticalStrut(10));
    cancelButton = new JButton("Cancel");
    cancelButton.setActionCommand("Cancel");
    cancelButton.addActionListener(this);
    cancelButton.setAlignmentY(Component.CENTER_ALIGNMENT);
    cancelButton.setAlignmentX(Component.CENTER_ALIGNMENT);
    cancelButton.setEnabled(false);
    panel.add(cancelButton);
    panel.setAlignmentY(Component.CENTER_ALIGNMENT);
    panel.setAlignmentX(Component.CENTER_ALIGNMENT);

    mainPanel.add(panel);

    mainFrame = new JFrame("leChunk") {
      protected void processWindowEvent (WindowEvent e) {
        super.processWindowEvent(e);
        if (e.getID() == e.WINDOW_CLOSING) System.exit(0);
      }
    };

    mainFrame.setIconImage(new ImageIcon(lechunk.class.getResource("logo.png")).getImage());
    mainFrame.getContentPane().add(mainPanel);
    mainFrame.pack();
    mainFrame.setVisible(true);
    mainFrame.setEnabled(true);

    worker = null;
    updateRepositoryList();
  }

  void wrkDirCleanup (File wrkDir)
  {
    File dstFiles[] = wrkDir.listFiles();
    for (int i = 0; i < dstFiles.length; ++i) dstFiles[i].delete();
  }

  void wrkDirRemove (File wrkDir)
  {
    wrkDirCleanup(wrkDir);
    wrkDir.delete();
    new File(wrkDir.getAbsolutePath() + ".md5").delete();
    new File(wrkDir.getAbsolutePath() + ".size").delete();
    REPOSITORY_DIR.delete();
  }

  boolean importChunk (FileInputStream src, OutputStream dst, String name) throws IOException
  {
    byte buffer[] = new byte[65536];
    long readTotal;
    int readBytes;

    for (readTotal = 0; readTotal < CHUNK_SIZE; readTotal += readBytes) {
      readBytes = src.read(buffer);
      if (readBytes == -1) return false;
      dst.write(buffer, 0, readBytes);
      bytesSoFar += readBytes;
      progressBar.setValue((int)(10000 * bytesSoFar/currentSize));
      progressBar.setString("Importing " + name + ": " +
                            (int)(100 * bytesSoFar/currentSize) +
                            "%");
      long now = Calendar.getInstance().getTimeInMillis() / 1000;
      if (now > startTime) {
        long speed = (bytesSoFar / 1024) / (now - startTime);
        progressBar.setToolTipText(new Long(speed).toString() + " kB/s");
      }
    }

    return true;
  }

  boolean importOverwrite (String name)
  {
    return (JOptionPane.showOptionDialog(null, name + " already exists in " +
                                         "the repository. Overwrite?", name,
                                         JOptionPane.YES_NO_OPTION,
                                         JOptionPane.QUESTION_MESSAGE, null,
                                         null, null) == JOptionPane.YES_OPTION);
  }

  void importFile (Job job)
  {
    int chunk = 0;
    boolean srcHasSome;
    File srcFile = job.getFile();
    String name = " " + srcFile.getName();

    File wrkDir = new File(REPOSITORY_DIR, name);

    File md5File = new File(REPOSITORY_DIR, name + ".md5");
    if (md5File.exists()) {
      if (importOverwrite(name)) wrkDirRemove(wrkDir);
      else return;
    }

    wrkDir.mkdirs();

    progressBar.setValue(0);
    progressBar.setStringPainted(true);

    try {
      OutputStream dst;
      FileInputStream src = new FileInputStream(srcFile);
      bytesSoFar = 0;
      currentSize = srcFile.length();
      startTime = Calendar.getInstance().getTimeInMillis() / 1000;
      byte bsize[] = Job.AES_align(new Long(currentSize).toString()).getBytes("US-ASCII");
      progressBar.setMaximum(10000);
      do {
        dst = new FileOutputStream(new File(wrkDir, new Formatter().format("%016d", chunk).toString()));
        if (job.hasPassword()) dst = new CipherOutputStream(dst, job.getCipher());
        srcHasSome = importChunk(src, dst, name);
        dst.close();
        ++chunk;
      } while (srcHasSome);
      src.close();
      FileOutputStream dst2 = new FileOutputStream(new File(REPOSITORY_DIR, name + ".md5"));
      dst2.write(bsize);
      dst2.close();
      dst = new FileOutputStream(new File(REPOSITORY_DIR, name + ".size"));
      if (job.hasPassword()) dst = new CipherOutputStream(dst, job.getCipher());
      dst.write(bsize);
      dst.close();
    } catch (Exception e) {
      JOptionPane.showMessageDialog(null, e.toString(), name, JOptionPane.ERROR_MESSAGE);
      wrkDirRemove(wrkDir);
    }

    progressBar.setStringPainted(false);
    progressBar.setValue(0);
    progressBar.setMaximum(0);
  }

  void exportChunk (InputStream src, FileOutputStream dst, String name) throws Exception
  {
    byte buffer[] = new byte[65536];
    long readTotal = 0;
    int readBytes;

    while ((readBytes = src.read(buffer)) != -1) {
      dst.write(buffer, 0, readBytes);
      readTotal += readBytes;
      bytesSoFar += readBytes;
      progressBar.setValue((int)(10000 * bytesSoFar/currentSize));
      progressBar.setString("Exporting " + name + ": " +
                            (int)(100 * bytesSoFar/currentSize) +
                            "%");
      long now = Calendar.getInstance().getTimeInMillis() / 1000;
      if (now > startTime) {
        long speed = (bytesSoFar / 1024) / (now - startTime);
        progressBar.setToolTipText(new Long(speed).toString() + " kB/s");
      }
    }

    src.close();
  }

  boolean exportOverwrite (File dir, String file)
  {
    return (JOptionPane.showOptionDialog(null, file + " already exists in " +
                                         dir.getAbsolutePath() + ". Overwrite?",
                                         file, JOptionPane.YES_NO_OPTION,
                                         JOptionPane.QUESTION_MESSAGE, null,
                                         null, null) == JOptionPane.YES_OPTION);
  }

  void exportFile (Job job)
  {
    File dir = job.getFile();
    String file = job.getName();

    File dstFile = new File(dir, file.substring(1, file.length()));
    if (dstFile.exists()) if (!exportOverwrite(dir, file)) return;

    try {
      File f = new File(REPOSITORY_DIR, file + ".md5");
      FileInputStream src2 = new FileInputStream(f);
      startTime = Calendar.getInstance().getTimeInMillis() / 1000;
      byte tmp2[] = new byte[src2.available()];
      byte tmp[] = new byte[src2.available()];
      src2.read(tmp2);
      src2.close();
      InputStream src = new FileInputStream(new File(REPOSITORY_DIR, file + ".size"));
      if (job.hasPassword()) src = new CipherInputStream(src, job.getCipher());
      src.read(tmp);
      src.close();
      if (!new String(tmp).equals(new String(tmp2))) {
        JOptionPane.showMessageDialog(null, "Wrong password!", file, JOptionPane.ERROR_MESSAGE);
        return;
      }
      bytesSoFar = 0;
      currentSize = new Long(new String(tmp).trim()).longValue();
      progressBar.setValue(0);
      progressBar.setStringPainted(true);
      progressBar.setMaximum(10000);
      FileOutputStream dst = new FileOutputStream(dstFile);
      File srcFiles[] = new File(REPOSITORY_DIR, file).listFiles();
      for (int i = 0; i < srcFiles.length; ++i) {
        src = new FileInputStream(srcFiles[i]);
        if (job.hasPassword()) src = new CipherInputStream(src, job.getCipher());
        exportChunk(src, dst, file);
      }
      dst.close();
    } catch (Exception e) {
      JOptionPane.showMessageDialog(null, e.toString(), file, JOptionPane.ERROR_MESSAGE);
      dstFile.delete();
    }

    progressBar.setStringPainted(false);
    progressBar.setValue(0);
    progressBar.setMaximum(0);
  }

  public void run ()
  {
    Job job;

    while ((job = updateTaskList(TASKLIST_MODE.POP, null)) != null) {
      switch (job.getMode()) {
        case IMPORT:
          importFile(job);
          EventQueue.invokeLater(new Runnable() {
            public void run() {
              updateRepositoryList();
            }
          });
          break;
        case EXPORT:
          exportFile(job);
          break;
      }
    }
  }

  synchronized Job updateTaskList (TASKLIST_MODE mode, Job job)
  {
    switch (mode) {
      case PUSH:
        taskListModel.addElement(job);
        if (worker == null) {
          worker = new Thread(this);
          worker.start();
        }
        break;
      case POP:
        if (taskListModel.isEmpty()) worker = null;
        else {
          job = (Job) taskListModel.firstElement();
          try{
            EventQueue.invokeAndWait(new SafeTaskListModelRemove(taskListModel, job));
          } catch (Exception e) {
            System.out.println(e.toString());
          }
        }
        break;
    }
    return job;
  }

  void actionImport ()
  {
    if (importChooser.showOpenDialog(mainFrame) == JFileChooser.APPROVE_OPTION) {
      File files[] = importChooser.getSelectedFiles();
      for (int i = 0; i < files.length; ++i)
        updateTaskList(TASKLIST_MODE.PUSH, new Job(Job.MODE.IMPORT, files[i], null, new String(pwdField.getPassword())));
    }
  }

  void actionExport ()
  {
    if (exportChooser.showSaveDialog(mainFrame) == JFileChooser.APPROVE_OPTION) {
      File dir = (brokenJFileChooser ? exportChooser.getCurrentDirectory() : exportChooser.getSelectedFile());
      String files[] = getSelectedRepositoryFiles();
      for (int i = 0; i < files.length; ++i)
        updateTaskList(TASKLIST_MODE.PUSH, new Job(Job.MODE.EXPORT, dir, files[i], new String(pwdField.getPassword())));
    }
  }

  String[] getSelectedRepositoryFiles ()
  {
    int idx[] = repositoryList.getSelectedIndices();
    String files[] = new String[idx.length];

    for (int i = 0; i < idx.length; ++i)
      files[i] = (String) repositoryListModel.get(idx[i]);

    return files;
  }

  boolean deleteFromRepository (String files[])
  {
    String allFiles = "", dialog;
    for (int i = 0; i < files.length; ++i) allFiles += "\n" + files[i];
    if (files.length == 1)
      dialog = "Do you really want to delete this file from the repository?\n";
    else
      dialog = "Do you really want to delete these files from the repository?\n";
    return (JOptionPane.showConfirmDialog(null, dialog + allFiles,
                                          "Confirm deletion",
                                          JOptionPane.YES_NO_OPTION)
            == JOptionPane.YES_OPTION);
  }

  void actionDelete ()
  {
    String files[] = getSelectedRepositoryFiles();

    if (deleteFromRepository(files)) {
      for (int i = 0; i < files.length; ++i)
        wrkDirRemove(new File(REPOSITORY_DIR, files[i]));
    }

    updateRepositoryList();
  }

  void actionCancel ()
  {
    Object obj;
    while ((obj = taskList.getSelectedValue()) != null)
      taskListModel.removeElement(obj);
  }

  public void actionPerformed (ActionEvent e)
  {
    String s = e.getActionCommand();

    if (s.equals("Import")) actionImport();
    else if (s.equals("Export")) actionExport();
    else if (s.equals("Delete")) actionDelete();
    else if (s.equals("Cancel")) actionCancel();
  }

  public void valueChanged (ListSelectionEvent e)
  {
    if (e.getSource() == taskList) {
      if (taskList.getSelectedIndex() == -1) cancelButton.setEnabled(false);
      else cancelButton.setEnabled(true);
    }
    if (e.getSource() == repositoryList) {
      if (repositoryList.getSelectedIndex() == -1) {
        exportButton.setEnabled(false);
        deleteButton.setEnabled(false);
      }
      else {
        exportButton.setEnabled(true);
        deleteButton.setEnabled(true);
      }
    }
  }

  void updateRepositoryList ()
  {
    String files[] = REPOSITORY_DIR.list(this);
    repositoryListModel.clear();
    if (files != null) for (int i = 0; i < files.length; ++i)
      repositoryListModel.addElement(files[i]);
  }

  public boolean accept (File dir, String name)
  {
    File testDir = new File(dir, name);
    File firstChunk = new File(testDir, "0000000000000000");
    File md5File = new File(dir, name + ".md5");
    File sizeFile = new File(dir, name + ".size");
    return (" ".equals(name.substring(0, 1)) && testDir.isDirectory() && firstChunk.isFile() && md5File.isFile() && sizeFile.isFile());
  }

  public static void main (String[] argv)
  {
    lechunk mainApp = new lechunk();
  }
}
