import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

class Job {
  static enum MODE { IMPORT, EXPORT };
  MODE mode;
  File file;
  String name;

  Job (MODE m, File f, String n)
  {
    mode = m;
    file = f;
    name = n;
  }

  MODE getMode () { return mode; }
  File getFile () { return file; }
  String getName () { return name; }

  public String toString ()
  {
    switch (mode) {
      case IMPORT:
        return "Import " + file.getAbsolutePath();
      case EXPORT:
        return "Export " + name + " to " + file.getAbsolutePath();
    }
    return "";
  }
}

class SafeTaskListModelRemove extends Thread {
  DefaultListModel taskListModel;
  Job job;

  SafeTaskListModelRemove (DefaultListModel m, Job j) {
    taskListModel = m;
    job = j;
  }

  public void run () {
    taskListModel.removeElement(job);
  }
}

class lechunk implements ActionListener, Runnable, ListSelectionListener, FilenameFilter
{
  static final int CHUNK_SIZE = 512 * 1024 * 1024;
  static final File REPOSITORY_DIR = new File(".lechunk");
  static enum TASKLIST_MODE { PUSH, POP };

  JFrame mainFrame;
  JProgressBar progressBar;
  JButton cancelButton, exportButton, deleteButton;
  JList taskList, repositoryList;
  DefaultListModel taskListModel, repositoryListModel;
  Thread worker;

  lechunk()
  {
    JPanel mainPanel = new JPanel();
    mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.LINE_AXIS));
    JButton button = new JButton("Import");
    button.setActionCommand("Import");
    button.addActionListener(this);
    button.setAlignmentY(Component.CENTER_ALIGNMENT);
    button.setAlignmentX(Component.CENTER_ALIGNMENT);
    mainPanel.add(button);
    exportButton = new JButton("Export");
    exportButton.setActionCommand("Export");
    exportButton.addActionListener(this);
    exportButton.setAlignmentY(Component.CENTER_ALIGNMENT);
    exportButton.setAlignmentX(Component.CENTER_ALIGNMENT);
    exportButton.setEnabled(false);
    mainPanel.add(exportButton);
    deleteButton = new JButton("Delete");
    deleteButton.setActionCommand("Delete");
    deleteButton.addActionListener(this);
    deleteButton.setAlignmentY(Component.CENTER_ALIGNMENT);
    deleteButton.setAlignmentX(Component.CENTER_ALIGNMENT);
    deleteButton.setEnabled(false);
    mainPanel.add(deleteButton);
    mainPanel.setAlignmentY(Component.CENTER_ALIGNMENT);
    mainPanel.setAlignmentX(Component.CENTER_ALIGNMENT);

    repositoryListModel = new DefaultListModel();
    repositoryList = new JList(repositoryListModel);
    repositoryList.setLayoutOrientation(JList.VERTICAL_WRAP);
    repositoryList.addListSelectionListener(this);

    JPanel panel = new JPanel();
    panel.setLayout(new BoxLayout(panel, BoxLayout.PAGE_AXIS));
    panel.setBorder(new TitledBorder("Repository"));
    panel.add(new JScrollPane(repositoryList));
    panel.add(Box.createVerticalStrut(10));
    panel.add(mainPanel);
    panel.add(Box.createVerticalStrut(10));

    progressBar = new JProgressBar(0, 0);
    progressBar.setValue(0);
    progressBar.setStringPainted(false);
    progressBar.setAlignmentY(Component.CENTER_ALIGNMENT);
    progressBar.setAlignmentX(Component.CENTER_ALIGNMENT);
    panel.add(progressBar);
    panel.setAlignmentY(Component.CENTER_ALIGNMENT);
    panel.setAlignmentX(Component.CENTER_ALIGNMENT);

    mainPanel = new JPanel();
    mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.LINE_AXIS));
    mainPanel.add(panel);

    taskListModel = new DefaultListModel();
    taskList = new JList(taskListModel);
    taskList.addListSelectionListener(this);
    panel = new JPanel();
    panel.setLayout(new BoxLayout(panel, BoxLayout.PAGE_AXIS));
    panel.setBorder(new TitledBorder("Tasks"));
    panel.add(new JScrollPane(taskList));
    panel.add(Box.createVerticalStrut(10));
    cancelButton = new JButton("Cancel");
    cancelButton.setActionCommand("Cancel");
    cancelButton.addActionListener(this);
    cancelButton.setAlignmentY(Component.CENTER_ALIGNMENT);
    cancelButton.setAlignmentX(Component.CENTER_ALIGNMENT);
    cancelButton.setEnabled(false);
    panel.add(cancelButton);
    panel.setAlignmentY(Component.CENTER_ALIGNMENT);
    panel.setAlignmentX(Component.CENTER_ALIGNMENT);

    mainPanel.add(panel);

    mainFrame = new JFrame("leChunk") {
      protected void processWindowEvent (WindowEvent e) {
        super.processWindowEvent(e);
        if (e.getID() == e.WINDOW_CLOSING) System.exit(0);
      }
    };

    mainFrame.setIconImage(new ImageIcon(lechunk.class.getResource("logo.png")).getImage());
    mainFrame.getContentPane().add(mainPanel);
    mainFrame.pack();
    mainFrame.setVisible(true);
    mainFrame.setEnabled(true);

    worker = null;
    updateRepositoryList();
  }

  void wrkDirCleanup (File wrkDir)
  {
    File dstFiles[] = wrkDir.listFiles();
    for (int i = 0; i < dstFiles.length; ++i) dstFiles[i].delete();
  }

  void wrkDirRemove (File wrkDir)
  {
    wrkDirCleanup(wrkDir);
    wrkDir.delete();
    new File(wrkDir.getAbsolutePath() + ".md5").delete();
    REPOSITORY_DIR.delete();
  }

  boolean importChunk (FileInputStream src, FileOutputStream dst, String name) throws IOException
  {
    byte buffer[] = new byte[65536];
    int readTotal, readBytes;
    long startTime = Calendar.getInstance().getTimeInMillis() / 1000;

    for (readTotal = 0; readTotal < CHUNK_SIZE; readTotal += readBytes) {
      readBytes = src.read(buffer);
      if (readBytes == -1) return false;
      dst.write(buffer, 0, readBytes);
      progressBar.setValue(progressBar.getValue() + readBytes);
      progressBar.setString("Importing " + name + ": " +
                            (int) (100 * progressBar.getPercentComplete()) +
                            "%");
      long now = Calendar.getInstance().getTimeInMillis() / 1000;
      if (now > startTime) {
        long speed = (readTotal / 1024) / (now - startTime);
        progressBar.setToolTipText(new Long(speed).toString() + " kB/s");
      }
    }

    return true;
  }

  boolean importOverwrite (String name)
  {
    return (JOptionPane.showOptionDialog(null, name + " already exists in " +
                                         "the repository. Overwrite?", name,
                                         JOptionPane.YES_NO_OPTION,
                                         JOptionPane.QUESTION_MESSAGE, null,
                                         null, null) == JOptionPane.YES_OPTION);
  }

  void importFile (File srcFile)
  {
    int chunk = 0;
    boolean srcHasSome;
    String name = srcFile.getName();

    File wrkDir = new File(REPOSITORY_DIR, name);

    File md5File = new File(REPOSITORY_DIR, name + ".md5");
    if (md5File.exists()) {
      if (importOverwrite(name)) wrkDirRemove(wrkDir);
      else return;
    }

    wrkDir.mkdirs();

    progressBar.setValue(0);
    progressBar.setStringPainted(true);

    try {
      FileOutputStream dst;
      FileInputStream src = new FileInputStream(srcFile);
      int size = src.available();
      progressBar.setMaximum(size);
      do {
        dst = new FileOutputStream(new File(wrkDir, new Formatter().format("%016d", chunk).toString()));
        srcHasSome = importChunk(src, dst, name);
        dst.close();
        ++chunk;
      } while (srcHasSome);
      src.close();
      dst = new FileOutputStream(new File(REPOSITORY_DIR, name + ".md5"));
      dst.write(new Integer(size).toString().getBytes("US-ASCII"));
      dst.close();
    } catch (IOException e) {
      JOptionPane.showMessageDialog(null, e.toString(), name, JOptionPane.ERROR_MESSAGE);
      wrkDirRemove(wrkDir);
    }

    progressBar.setStringPainted(false);
    progressBar.setValue(0);
    progressBar.setMaximum(0);
  }

  void exportChunk (File srcFile, FileOutputStream dst, String name) throws IOException
  {
    byte buffer[] = new byte[65536];
    int readTotal = 0, readBytes;
    long startTime = Calendar.getInstance().getTimeInMillis() / 1000;

    FileInputStream src = new FileInputStream(srcFile);

    while ((readBytes = src.read(buffer)) != -1) {
      dst.write(buffer, 0, readBytes);
      readTotal += readBytes;
      progressBar.setValue(progressBar.getValue() + readBytes);
      progressBar.setString("Exporting " + name + ": " +
                            (int) (100 * progressBar.getPercentComplete()) +
                            "%");
      long now = Calendar.getInstance().getTimeInMillis() / 1000;
      if (now > startTime) {
        long speed = (readTotal / 1024) / (now - startTime);
        progressBar.setToolTipText(new Long(speed).toString() + " kB/s");
      }
    }

    src.close();
  }

  boolean exportOverwrite (File dir, String file)
  {
    return (JOptionPane.showOptionDialog(null, file + " already exists in " +
                                         dir.getAbsolutePath() + ". Overwrite?",
                                         file, JOptionPane.YES_NO_OPTION,
                                         JOptionPane.QUESTION_MESSAGE, null,
                                         null, null) == JOptionPane.YES_OPTION);
  }

  void exportFile (File dir, String file)
  {
    File dstFile = new File(dir, file);
    if (dstFile.exists()) if (!exportOverwrite(dir, file)) return;

    progressBar.setValue(0);
    progressBar.setStringPainted(true);

    try {
      FileInputStream src = new FileInputStream(new File(REPOSITORY_DIR, file + ".md5"));
      byte tmp[] = new byte[src.available()];
      src.read(tmp);
      src.close();
      progressBar.setMaximum(new Integer(new String(tmp, "US-ASCII")).intValue());
      FileOutputStream dst = new FileOutputStream(dstFile);
      File srcFiles[] = new File(REPOSITORY_DIR, file).listFiles();
      for (int i = 0; i < srcFiles.length; ++i) exportChunk(srcFiles[i], dst, file);
      dst.close();
    } catch (IOException e) {
      JOptionPane.showMessageDialog(null, e.toString(), file, JOptionPane.ERROR_MESSAGE);
      dstFile.delete();
    }

    progressBar.setStringPainted(false);
    progressBar.setValue(0);
    progressBar.setMaximum(0);
  }

  public void run ()
  {
    Job job;

    while ((job = updateTaskList(TASKLIST_MODE.POP, null)) != null) {
      switch (job.getMode()) {
        case IMPORT:
          importFile(job.getFile());
          EventQueue.invokeLater(new Runnable() {
            public void run() {
              updateRepositoryList();
            }
          });
          break;
        case EXPORT:
          exportFile(job.getFile(), job.getName());
          break;
      }
    }
  }

  synchronized Job updateTaskList (TASKLIST_MODE mode, Job job)
  {
    switch (mode) {
      case PUSH:
        taskListModel.addElement(job);
        if (worker == null) {
          worker = new Thread(this);
          worker.start();
        }
        break;
      case POP:
        if (taskListModel.isEmpty()) worker = null;
        else {
          job = (Job) taskListModel.firstElement();
          try{
            EventQueue.invokeAndWait(new SafeTaskListModelRemove(taskListModel, job));
          } catch (Exception e) {
            System.out.println(e.toString());
          }
        }
        break;
    }
    return job;
  }

  void actionImport ()
  {
    JFileChooser chooser = new JFileChooser();
    chooser.setMultiSelectionEnabled(true);
    chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
    if (chooser.showOpenDialog(mainFrame) == JFileChooser.APPROVE_OPTION) {
      File files[] = chooser.getSelectedFiles();
      for (int i = 0; i < files.length; ++i)
        updateTaskList(TASKLIST_MODE.PUSH, new Job(Job.MODE.IMPORT, files[i], null));
    }
  }

  void actionExport ()
  {
    JFileChooser chooser = new JFileChooser();
    chooser.setMultiSelectionEnabled(false);
    chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
    if (chooser.showSaveDialog(mainFrame) == JFileChooser.APPROVE_OPTION) {
      File dir = chooser.getSelectedFile();
      String files[] = getSelectedRepositoryFiles();
      for (int i = 0; i < files.length; ++i)
        updateTaskList(TASKLIST_MODE.PUSH, new Job(Job.MODE.EXPORT, dir, files[i]));
    }
  }

  String[] getSelectedRepositoryFiles ()
  {
    int idx[] = repositoryList.getSelectedIndices();
    String files[] = new String[idx.length];

    for (int i = 0; i < idx.length; ++i)
      files[i] = (String) repositoryListModel.get(idx[i]);

    return files;
  }

  boolean deleteFromRepository (String files[])
  {
    String allFiles = "", dialog;
    for (int i = 0; i < files.length; ++i) allFiles += "\n" + files[i];
    if (files.length == 1)
      dialog = "Do you really want to delete this file from the repository?\n";
    else
      dialog = "Do you really want to delete these files from the repository?\n";
    return (JOptionPane.showConfirmDialog(null, dialog + allFiles,
                                          "Confirm deletion",
                                          JOptionPane.YES_NO_OPTION)
            == JOptionPane.YES_OPTION);
  }

  void actionDelete ()
  {
    String files[] = getSelectedRepositoryFiles();

    if (deleteFromRepository(files)) {
      for (int i = 0; i < files.length; ++i)
        wrkDirRemove(new File(REPOSITORY_DIR, files[i]));
    }

    updateRepositoryList();
  }

  void actionCancel ()
  {
    Object obj;
    while ((obj = taskList.getSelectedValue()) != null)
      taskListModel.removeElement(obj);
  }

  public void actionPerformed (ActionEvent e)
  {
    String s = e.getActionCommand();

    if (s.equals("Import")) actionImport();
    else if (s.equals("Export")) actionExport();
    else if (s.equals("Delete")) actionDelete();
    else if (s.equals("Cancel")) actionCancel();
  }

  public void valueChanged (ListSelectionEvent e)
  {
    if (e.getSource() == taskList) {
      if (taskList.getSelectedIndex() == -1) cancelButton.setEnabled(false);
      else cancelButton.setEnabled(true);
    }
    if (e.getSource() == repositoryList) {
      if (repositoryList.getSelectedIndex() == -1) {
        exportButton.setEnabled(false);
        deleteButton.setEnabled(false);
      }
      else {
        exportButton.setEnabled(true);
        deleteButton.setEnabled(true);
      }
    }
  }

  void updateRepositoryList ()
  {
    String files[] = REPOSITORY_DIR.list(this);
    repositoryListModel.clear();
    if (files != null) for (int i = 0; i < files.length; ++i)
      repositoryListModel.addElement(files[i]);
  }

  public boolean accept (File dir, String name)
  {
    File testDir = new File(dir, name);
    File md5File = new File(dir, name + ".md5");
    return (testDir.isDirectory() && md5File.exists());
  }

  public static void main (String[] argv)
  {
    lechunk mainApp = new lechunk();
  }
}
