#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/wait.h>
#include <malloc.h>

#define STDIN 0
#define STDOUT 1
#define STDERR 2
#define READ 0
#define WRITE 1
#define BUFFER_SIZE 4096 * 4096

#define USAGE "rpostfilter </path/to/sm> <token> <filename> [<headline to erase> ...]\nrpostfilter - inntools " VERSION "\n"

void die (const char *s1, const char *s2)
{
	write(STDERR, s1, strlen(s1));
	write(STDERR, ": ", 2);
	write(STDERR, s2, strlen(s2));
	write(STDERR, "\n", 1);
	exit(1);
}

void *realloc_or_die (void *ptr, const size_t len)
{
	ptr = realloc(ptr, len);
	if (!ptr) die("realloc()", strerror(errno));
	return ptr;
}

int main (int argc, char **argv, char **envp)
{
	ssize_t ret;
	size_t len;
	pid_t pid, child;
	int status, fd[2];
	char *sm, *token, *outfn, *msg, *bodysep, *left, *right, *tmp;

	if (argc < 4) die("usage", USAGE);

	if (close(STDIN)) die("close()", strerror(errno));
	if (pipe(fd)) die("pipe()", strerror(errno));

	sm = argv[1];
	token = argv[2];
	outfn = argv[3];

	pid = fork();

	if (pid < 0) die("fork()", strerror(errno));

	/* child */
	if (!pid) {
		if (close(fd[READ])) die("close()", strerror(errno));
		if ((dup2(fd[WRITE], STDERR) == -1) ||
		    (dup2(fd[WRITE], STDOUT) == -1))
			die("dup2()", strerror(errno));

		outfn = NULL;
		execve(sm, argv + sizeof(char*), envp);
		exit(errno);
	}

	/* parent */
	if (close(fd[WRITE])) die("close()", strerror(errno));

	child = wait(&status);
	if (child < 0) die("waitpid()", strerror(errno));
	if (pid != child) die("waitpid()", "bogus pid claims to be our child");
	if (WEXITSTATUS(status)) die(sm, strerror(WEXITSTATUS(status)));

	for (msg = NULL, len = 0;;) {
		msg = realloc_or_die(msg, len + BUFFER_SIZE);
		ret = read(fd[READ], msg + len, BUFFER_SIZE);
		if (ret < 0) {
			if (errno == EINTR) continue;
			die("read()", strerror(errno));
		}
		if (!ret) break;
		len += ret;
	}
	msg = realloc_or_die(msg, len + 1);
	*(msg + len) = '\0';
	++len;

	bodysep = strstr(msg, "\r\n\r\n");
	if (!bodysep)
		die(token, "invalid message (no body found)");

	for (tmp = NULL; argc > 4; --argc) {
		if (*(argv[argc - 1] + strlen(argv[argc - 1]) - 1) != ':') {
			tmp = realloc_or_die(tmp, strlen(argv[argc - 1]) + 2);
			strcpy(tmp, argv[argc - 1]);
			strcat(tmp, ":");
		}
		else tmp = argv[argc - 1];

		left = strstr(msg, tmp);
		if (!left || (left > bodysep)) continue;

		right = strchr(left, '\n');
		if (!right || (right > bodysep + 1)) continue;
		++right;
		if ((left == msg) || (*(left - 1) == '\n')) {
			memmove(left, right, len - (right - msg));
			len -= right - left;
		}
	}

	write(STDOUT, msg, len);
	free(msg);
	return status;
}
