#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <resolv.h>
#include <netdb.h>
#include <fcntl.h>
#include <errno.h>
#include <dirent.h>
#include <signal.h>

#define STDIN 0
#define STDOUT 1
#define STDERR 2

#define TCP "tcp"
#define NNTP "nntp"

#define BUFFER_SIZE 8192

#define TIMEOUT 30

ssize_t server_len;
char *server;

void _write (int fd, const char *s)
{
	write(fd, s, strlen(s));
}

void print (const char *s)
{
	_write(STDOUT, s);
}

void warn (const char *s1, const char *s2)
{
	write(STDERR, "WARNING: ", 9);
	_write(STDERR, s1);
	write(STDERR, ": ", 2);
	_write(STDERR, s2);
	write(STDERR, "\n", 1);
}

void die (const char *s1, const char *s2)
{
	write(STDERR, "ERROR: ", 7);
	_write(STDERR, s1);
	write(STDERR, ": ", 2);
	_write(STDERR, s2);
	write(STDERR, "\n", 1);
	exit(1);
}

void alrm_handler (int i)
{
	die("read()", "timeout");
}

ssize_t _read (const int fd, void *buf, const size_t count)
{
	static struct sigaction act, oldact;
	static ssize_t ret;

	act.sa_handler = alrm_handler;
	if (sigaction(SIGALRM, &act, &oldact)) return -1;

	alarm(TIMEOUT);
	ret = read(fd, buf, count);
	alarm(0);
	if (sigaction(SIGALRM, &oldact, NULL)) return -1;

	return ret;
}

void usage ()
{
	print("usage: repost <dir> <server>\nrepost - inntools " VERSION "\n");
	exit(2);
}

void rm (const char *fn)
{
	if (unlink(fn)) warn(fn, strerror(errno));
}

void repost (const char *fn, off_t size)
{
	static ssize_t ret;
	static int fh, sock;
	static char *p = NULL, *mid = NULL, *tmp, *right, buf[BUFFER_SIZE];

	p = realloc(p, size + 1);
	if (!p) {
		warn("realloc()", "no memory");
		return;
	}

	fh = open(fn, O_RDONLY);
	if (!fh) {
		warn(fn, strerror(errno));
		return;
	}

	ret = read(fh, p, size);
	close(fh);
	if (ret != size) {
		warn(fn, "could not read whole file");
		return;
	}

	/*
	 * replace any \n by \r\n
	 */

	*(p + size) = '\0';
	for (tmp = strchr(p, '\n'); tmp; tmp = strchr(tmp, '\n')) {
		if (*(tmp - 1) != '\r') ++size;
		++tmp;
	}

	p = realloc(p, size + 1);
	if (!p) {
		warn("realloc()", "no memory");
		return;
	}

	for (tmp = strchr(p, '\n'); tmp; tmp = strchr(tmp, '\n')) {
		if (*(tmp - 1) != '\r') {
			memmove(tmp + 1, tmp, size - (tmp - p));
			*tmp = '\r';
		}
		++tmp;
	}

	/*
	 * remove old server from Path: line
	 */

	for (tmp = p;;) {
		tmp = strstr(tmp, "Path: ");
		if (!tmp) {
			warn(fn, "no Path: line");
			return;
		}
		if ((tmp == p) || (*(tmp - 1) == '\n')) break;
		++tmp;
	}

	tmp += 6;

	if (!strncmp(tmp, server, server_len)) {
		size -= server_len + 1;
		memmove(tmp, tmp + server_len + 1, size - (tmp - p));
	}
	else warn(fn, "server not in Path: line");

	/*
	 * grep Message-ID
	 */

	for (tmp = p;;) {
		tmp = strstr(tmp, "Message-ID: ");
		if (!tmp) {
			warn(fn, "no Message-ID: line");
			return;
		}
		if ((tmp == p) || (*(tmp - 1) == '\n')) break;
		++tmp;
	}
	tmp += 12;
	while (*tmp == ' ') ++tmp;

	right = strchr(tmp, '\r');
	if (!right) {
		warn(fn, "Message-ID: line does not end with carriage return");
		return;
	}

	*right = '\0';
	mid = realloc(mid, (right - tmp) + 1);
	if (!mid) {
		warn("realloc()", "no memory");
		return;
	}
	strcpy(mid, tmp);
	*right = '\r';

	/*
	 * delete Xref: line if it shows the old server 
	 */

	for (tmp = p;;) {
		tmp = strstr(tmp, "Xref: ");
		if (!tmp || (tmp == p) || (*(tmp - 1) == '\n')) break;
		++tmp;
	}
		
	if (tmp) {
		if (!strncmp((tmp + 6), server, server_len)) {
			right = strchr(tmp, '\n');
			if (!right) {
				warn(fn,"Xref: line does not end with newline");
				return;
			}
			++right;
			size -= right - tmp;
			memmove(tmp, right, size - (tmp - p));
		}
		else warn(fn, "server not in Xref: line");
	}
	else warn(fn, "no Xref: line");

	print(fn);
	write(STDOUT, ": ", 2);
	print(mid);
	write(STDOUT, ": ", 2);

#ifndef DEBUG
	write(sock, "IHAVE ", 6);
	write(1, "IHAVE ", 6);
	_write(sock, mid);
	_write(1, mid);
	write(sock, "\r\n", 2);
	write(1, "\r\n", 2);

	ret = _read(sock, buf, BUFFER_SIZE);
	if (ret < 0) die("read()", strerror(errno));
	buf[ret] = '\0';

	if (strncmp(buf, "335", 3)) {
		print(buf);
		rm(fn);
		return;
	}

	write(sock, p, size);
	write(1, p, size);
	write(sock, ".\r\n", 3);
	write(1, ".\r\n", 3);

	ret = _read(sock, buf, BUFFER_SIZE);
	if (ret < 0) die("read()", strerror(errno));
	buf[ret] = '\0';

	if (strncmp(buf, "235", 3)) {
		print(buf);
		return;
	}
#endif

	rm(fn);
	write(STDOUT, "OK\n", 3);
}

void traverse (char *rootdir)
{
	DIR *dir;
	static struct dirent *de;
	static struct stat st;
	char newdir[BUFFER_SIZE], *c = &newdir[0];

	dir = opendir(rootdir);
	if (!dir) {
		warn(rootdir, strerror(errno));
		return;
	}

	strcpy(c, rootdir);
	strcat(c, "/");
	c += strlen(c);

	while ((de = readdir(dir))) {
		if (!strcmp(de->d_name, ".") || !strcmp(de->d_name, ".."))
			continue;

		memmove(c, de->d_name, de->d_reclen);
		*(c + de->d_reclen) = '\0';

		if (stat(newdir, &st)) {
			warn(newdir, strerror(errno));
			closedir(dir);
			return;
		}

		if (!(st.st_mode & S_IRUSR)) {
			warn(newdir, "no read permissions");
			closedir(dir);
			return;
		}

		if (S_ISDIR(st.st_mode)) {
			if (st.st_mode & S_IXUSR) traverse(newdir);
			else warn(newdir, "no execute permissions");
		}
		else if (S_ISREG(st.st_mode)) {
			if (st.st_size) repost(newdir, st.st_size);
			else warn(newdir, "zero length");
		}

		else warn(newdir, "is a bogus file type");
	}

	closedir(dir);
}

int main (int argc, char **argv)
{
	struct hostent *he;
	struct protoent *pe;
	struct servent *se;
	struct sockaddr_in sin;
	int sock;
	char *rootdir, buf[BUFFER_SIZE];

	if (argc != 3) usage();

	rootdir = argv[1];
	server = argv[2];
	server_len = strlen(server);

	if (!(he = gethostbyname(server)))
		die(server, hstrerror(h_errno));

	if (!(pe = getprotobyname(TCP)))
		die(TCP, "unknown protocol");

	if (!(se = getservbyname(NNTP, pe->p_name)))
		die(NNTP, "unknown service");

	memcpy((char*) &sin.sin_addr, he->h_addr, he->h_length);
	sin.sin_port = se->s_port;
	sin.sin_family = PF_INET;

#ifndef DEBUG
	sock = socket(AF_INET, SOCK_STREAM, 0);
	if (sock < 0) die("socket(): ", strerror(errno));

	if (connect(sock, (struct sockaddr*) &sin, sizeof(sin)))
		die("connect(): ", strerror(errno));

	/*
	 * initial greeting from server
	 */

	if (_read(sock, &buf[0], BUFFER_SIZE) < 0)
		die("read()", strerror(errno));
#endif

	traverse(rootdir);

#ifndef DEBUG
	write(sock, "QUIT\r\n", 6);
	if (_read(sock, &buf[0], BUFFER_SIZE) < 0)
		die("read()", strerror(errno));

	close(sock);
#endif

	return 0;
}
