#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>

#define STDIN 0
#define STDOUT 1
#define STDERR 2

#define BUFFER_SIZE 4096

struct path {
  char *name, *desc;
  struct path *sub, *next;
};

struct path *quicksort (struct path *p) {
  struct path *tmp, *pre, *new, *left;

  if (!p) return NULL;

  for (tmp = p->next, pre = p, left = NULL; tmp;) {
    if (strcmp(tmp->name, p->name) < 0) {
      new = tmp;
      tmp = tmp->next;
      pre->next = tmp;
      new->next = left;
      left = new;
    }
    else {
      pre = tmp;
      tmp = tmp->next;
    }
  }

  if (p->next) p->next = quicksort(p->next);

  if (left) {
    if (left->next) left = quicksort(left);
    for (tmp = left; tmp->next; tmp = tmp->next)
      ;;

    tmp->next = p;
    p = left;
  }

  return p;
}

void warn (const char *s1, const char *s2)
{
  write(STDERR, s1, strlen(s1));
  write(STDERR, ": ", 2);
  write(STDERR, s2, strlen(s2));
  write(STDERR, "\n", 1);
}

void die (const char *s1, const char *s2)
{
  warn(s1, s2);
  exit(1);
}

char *realloc_or_die (char *p, ssize_t len)
{
  p = realloc(p, len);
  if (!p) die("realloc()", "no memory!");
  return p;
}

struct path *seen (struct path *groups, char *name, char *desc)
{
  struct path *tmp = NULL;
  char *c;

  if (!*name) return groups;

  c = strchr(name, '.');
  if (c) *c++ = 0;

  for (tmp = groups; tmp; tmp = tmp->next)
    if (!strcmp(name, tmp->name)) break;

  if (!tmp) {
    tmp = (struct path*) realloc_or_die(NULL, sizeof(struct path));
    tmp->name = name;
    if (!c) tmp->desc = desc;
    tmp->next = groups;
    tmp->sub = NULL;
    groups = tmp;
  }

  if (c) tmp->sub = seen(tmp->sub, c, desc);
  return groups;
}

void dump (struct path *groups, char *root)
{
  if (!groups) return;

  if (groups->sub) {
    char *c;

    if (root) {
      c = realloc_or_die(NULL, strlen(root) + strlen(groups->name) + 2);
      strcpy(c, root);
      strcat(c, ".");
      strcat(c, groups->name);
    }
    else {
      c = realloc_or_die(NULL, strlen(groups->name) + 1);
      strcpy(c, groups->name);
    }
    dump(quicksort(groups->sub), c);
    free(c);
  }
  else {
    if (root) {
      write(STDOUT, root, strlen(root));
      write(STDOUT, ".", 1);
    }
    write(STDOUT, groups->name, strlen(groups->name));
    write(STDOUT, "\t", 1);
    write(STDOUT, groups->desc, strlen(groups->desc));
    write(STDOUT, "\n", 1);
  }

  if (groups->next) dump(groups->next, root);
}

int main (int argc, char **argv)
{
  struct path *groups = NULL;
  ssize_t ret, len;
  char *list, *left, *middle, *right;

  if (argc != 1) die("usage", "mergelist - inntools " VERSION);

  for (list = NULL, len = 0;;) {
    list = realloc_or_die(list, len + BUFFER_SIZE);
    ret = read(STDIN, list + len, BUFFER_SIZE);
    if (ret < 0) die("read()", strerror(errno));
    if (!ret) break;
    len += ret;
  }

  *(list + len) = '\0';

  for (left = list;;) {
    middle = strchr(left, '\t');
    right = strchr(left, '\n');
    if (!middle || !right) break;
    *right = '\0';
    if (right < middle) warn("no [TAB] seperation", left);
    else {
      *middle = '\0';
      groups = seen(groups, left, middle + 1);
    }

    left = right + 1;
    if (left >= list + len) break;
  }

  dump(quicksort(groups), NULL);

  return 0;
}
