#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <err.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <netinet/tcp.h>
#include <net/if.h>
#include <net/pfvar.h>
#include <capsicum_helpers.h>
#include "fw.h"

#define PF_ANCHOR_NAME "ftp6proxy"

struct pf_data {
  int fd;
  struct pfioc_trans pft;
  struct pfioc_trans_e pfte;
  struct pfioc_pooladdr pfp;
  struct pfioc_rule pfr;
};

const char *fw_ident ()
{
  return "pf";
}

void *fw_init ()
{
  int save_errno;
  struct pf_data *pfd;
  cap_rights_t rights;
  unsigned long allowed_ioctls[] = { DIOCXBEGIN, DIOCXCOMMIT, DIOCXROLLBACK,
                                     DIOCADDRULE, DIOCBEGINADDRS };

  pfd = malloc(sizeof(*pfd));
  if (pfd == NULL) goto FW_INIT_ERR0;

  pfd->fd = open("/dev/pf", O_RDWR);
  if (pfd->fd < 0) goto FW_INIT_ERR1;

  cap_rights_init(&rights, CAP_IOCTL);
  if (caph_rights_limit(pfd->fd, &rights) != 0)
    goto FW_INIT_ERR2;

  if (caph_ioctls_limit(pfd->fd, allowed_ioctls,
                        sizeof(allowed_ioctls)/sizeof(allowed_ioctls[0])) != 0)
    goto FW_INIT_ERR2;

  return pfd;

FW_INIT_ERR2:
  save_errno = errno;
  close(pfd->fd);
  errno = save_errno;

FW_INIT_ERR1:
  save_errno = errno;
  free(pfd);
  errno = save_errno;

FW_INIT_ERR0:
  return NULL;
}

int fw_close (void *data)
{
  int fd;

  fd = ((struct pf_data*) data)->fd;
  free(data);
  return (close(fd) == -1 ? -1 : 0);
}

static int pf_prepare (struct pf_data *pfd, uint32_t rulenum,
                       char anchor[MAXPATHLEN])
{
  bzero(&pfd->pft, sizeof(pfd->pft));
  bzero(&pfd->pfte, sizeof(pfd->pfte));

  pfd->pft.size = 1;
  pfd->pft.esize = sizeof(pfd->pfte);
  pfd->pft.array = &pfd->pfte;

  pfd->pfte.rs_num = PF_RULESET_FILTER;
  snprintf(pfd->pfte.anchor, MAXPATHLEN, "%s/%d.%d",
           PF_ANCHOR_NAME, getpid(), rulenum);

  if (anchor != NULL)
    strncpy(anchor, pfd->pfte.anchor, sizeof(MAXPATHLEN));

  return (ioctl(pfd->fd, DIOCXBEGIN, &pfd->pft) == -1 ? -1 : 0);
}

static int pf_commit (struct pf_data *pfd)
{
  return (ioctl(pfd->fd, DIOCXCOMMIT, &pfd->pft) == -1 ? -1 : 0);
}

int fw_add_rule (void *data, uint32_t rulenum, struct rule *rule)
{
  int save_errno;
  struct pf_data *pfd = (struct pf_data*) data;
  char anchor[MAXPATHLEN];

  if (pf_prepare(pfd, rulenum, anchor) == -1)
    return -1;

  bzero(&pfd->pfp, sizeof(pfd->pfp));
  strlcpy(pfd->pfp.anchor, anchor, MAXPATHLEN);

  if (ioctl(pfd->fd, DIOCBEGINADDRS, &pfd->pfp) == -1)
    goto FW_ADD_RULE_FAIL;

  bzero(&pfd->pfr, sizeof(pfd->pfr));
  strlcpy(pfd->pfr.anchor, anchor, MAXPATHLEN);
  pfd->pfr.ticket = pfd->pfte.ticket;
  pfd->pfr.pool_ticket = pfd->pfp.ticket;

  pfd->pfr.rule.action = PF_PASS;
  pfd->pfr.rule.rtableid = -1;
  pfd->pfr.rule.quick = 1;
  pfd->pfr.rule.log = 0;
  if (rule->pf_queue != NULL)
    strlcpy(pfd->pfr.rule.qname, rule->pf_queue, sizeof(pfd->pfr.rule.qname));
  pfd->pfr.rule.keep_state = 1;
  pfd->pfr.rule.flags = TH_SYN;
  pfd->pfr.rule.flagset = TH_SYN|TH_ACK|TH_FIN|TH_RST;
  pfd->pfr.rule.max_states = 2; /* client -> server _and_ vice versa */
  pfd->pfr.rule.af = (rule->is_ipv6 ? AF_INET6 : AF_INET);
  pfd->pfr.rule.proto = IPPROTO_TCP;
  pfd->pfr.rule.src.port[0] = htons(rule->src_port_low);
  if (rule->src_port_low == rule->src_port_high)
    pfd->pfr.rule.src.port_op = PF_OP_EQ;
  else {
    pfd->pfr.rule.src.port_op = PF_OP_RRG;
    pfd->pfr.rule.src.port[1] = htons(rule->src_port_high);
  }
  pfd->pfr.rule.dst.port[0] = htons(rule->dst_port_low);
  if (rule->dst_port_low == rule->dst_port_high)
    pfd->pfr.rule.dst.port_op = PF_OP_EQ;
  else {
    pfd->pfr.rule.dst.port_op = PF_OP_RRG;
    pfd->pfr.rule.dst.port[1] = htons(rule->dst_port_high);
  }
  pfd->pfr.rule.src.addr.type = PF_ADDR_ADDRMASK;
  pfd->pfr.rule.dst.addr.type = PF_ADDR_ADDRMASK;
  if (rule->is_ipv6) {
    memcpy(&pfd->pfr.rule.src.addr.v.a.addr.v6, rule->src_ip, 16);
    memset(&pfd->pfr.rule.src.addr.v.a.mask.addr8, 255, 16);
    memcpy(&pfd->pfr.rule.dst.addr.v.a.addr.v6, rule->dst_ip, 16);
    memset(&pfd->pfr.rule.dst.addr.v.a.mask.addr8, 255, 16);
  } else {
    memcpy(&pfd->pfr.rule.src.addr.v.a.addr.v4, rule->src_ip, 4);
    memset(&pfd->pfr.rule.src.addr.v.a.mask.addr8, 255, 4);
    memcpy(&pfd->pfr.rule.dst.addr.v.a.addr.v4, rule->dst_ip, 4);
    memset(&pfd->pfr.rule.dst.addr.v.a.mask.addr8, 255, 4);
  }

  if (ioctl(pfd->fd, DIOCADDRULE, &pfd->pfr) == -1)
    goto FW_ADD_RULE_FAIL;

  if (pf_commit(pfd) == -1)
    goto FW_ADD_RULE_FAIL;

  return 0;

FW_ADD_RULE_FAIL:
  save_errno = errno;
  ioctl(pfd->fd, DIOCXROLLBACK, &pfd->pft);
  errno = save_errno;

  return -1;
}

int fw_del_rule (void *data, uint32_t rulenum)
{
  struct pf_data *pfd = (struct pf_data*) data;
  int ret;

  ret = pf_prepare(pfd, rulenum, NULL);
  return (pf_commit(pfd) || ret);
}
