#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mount.h>
#include <sys/syscall.h>
#include <sys/wait.h>

#define ROOT_DIR "/root"
#define NEW_ROOT_DIR "/tmp"

static void byebye ()
{
	fputs("\n\033[0;31m*** Fatal error, halting "
	      "(try Ctrl+Alt+Del) ***\033[0m\n", stderr);

	while (1) sleep(60);
}

static void fatal (char *usererr)
{
	fprintf(stderr, "%s: %s", usererr, strerror(errno));
	byebye();
}

static void fatalx (char *usererr)
{
	fputs(usererr, stderr);
	byebye();
}

static void lvm (char* argv[])
{
	pid_t pid;
	int i;

	pid = fork();
	if (pid < 0) fatal("cannot fork()");
	if (pid == 0) {
		execve("/sbin/lvm", argv, (char*[]) { NULL });
		fatal("cannot execute lvm"); 
	}

	while (wait(&i) != pid)
		;;
}

static void fmount (char *dev, char *path, char *fstype, int flags,
                    char *opts, char *err)
{
	if (mount(dev, path, fstype, flags, opts)) fatal(err);
}

static void funmount (char *path, char *err)
{
	if (umount(path)) fatal(err);
}

static char *getfilesystems (FILE **f)
{
	static char buffer[1024];

	if (!*f) {
		*f = fopen("/proc/filesystems", "r");
		if (!*f) fatal("opening /proc/filesystems failed");
	}

	while (fgets(buffer, sizeof(buffer), *f)) {
		if ((buffer[0] == '\t') &&
		    (buffer[strlen(buffer) - 1] == '\n')) {
			buffer[strlen(buffer) - 1] = '\0';
			return &buffer[1];
		}
	}

	return NULL;
}

int main (int argc __attribute__((unused)),
          char **argv __attribute__((unused)), char **envp)
{
	FILE *f = NULL;
	char *root, *rootflags, *rootfstype;
	int i, fh;

	/* some optical sugar :-) */
	puts("\033[0;36m*** Felix' LVM initrd ***\033[0m");

	/* mount ramdisk */
	fmount("/", "/", NULL, MS_REMOUNT, NULL, "remounting / failed");
	fmount("none", "/proc", "proc", 0, NULL, "mounting /proc failed");
	fmount("none", "/sys", "sysfs", 0, NULL, "mounting /sys failed");

	/* activate lvm */
	lvm((char*[]) { "vgscan", "--ignorelockingfailure", "--mknodes", "-P",
	                NULL });
	lvm((char*[]) { "vgchange", "--ignorelockingfailure", "-a", "y", "-P",
	                NULL });
	lvm((char*[]) { "vgmknodes", "--ignorelockingfailure", NULL });

	/* get options passed by boot loader to kernel image */
	root = getenv("lvmroot");
	if (!root)
		fatalx("got no lvmroot (try passing lvmroot=... to kernel "
		       "image in grub.cfg)");

	rootflags = getenv("lvmrootflags");
	rootfstype = getenv("lvmrootfstype");

	/* mount root */
	if (!rootfstype) rootfstype = getfilesystems(&f);
	while (rootfstype) {
		if (!mount(root, ROOT_DIR, rootfstype, MS_RDONLY, rootflags))
			break;
		fprintf(stderr, "  %s is not a %s: %s\n", root, rootfstype,
		                                          strerror(errno));
		rootfstype = getfilesystems(&f);
	}
	if (f) fclose(f);
	if (!rootfstype)
		fatalx("found no suitable root filesystem type");
	
	/* unmount everything inside ramdisk */
	funmount("/sys", "unmounting /sys failed");
	funmount("/proc", "unmounting /proc failed");

	/* swap root */
	if (syscall(SYS_pivot_root, ROOT_DIR, ROOT_DIR NEW_ROOT_DIR))
		fatal("pivoting " ROOT_DIR " and " ROOT_DIR NEW_ROOT_DIR
		      " failed");

	/* release everything opened in ramdisk */
	if (chdir("/")) fatal("changing directory to / failed");
	for (i = 0; i <= 2; ++i) {
		fh = open("/dev/console", (i ? O_WRONLY : O_RDONLY));
		if (fh == -1)
			fatal((i ? "opening /dev/console for writing failed" :
			           "opening /dev/console for reading failed"));
		if (dup2(fh, i) == -1)
			fatal((i ? "duping /dev/console for writing failed" :
			           "duping /dev/console for reading failed"));
		close(fh);
	}

	umount(NEW_ROOT_DIR "/dev");

	/* report success */
	puts("\033[0;36m*** READY ***\033[0m");

	/* unmount initrd in new shell called inside new root */
	/* and pass control to real init */
	execve("/bin/sh", (char*[]) { "sh", "-c", "umount " NEW_ROOT_DIR "; "
	       "exec /sbin/init; exec /bin/init; exec /etc/init; "
	       "exec /root/init; exec /init; exec init", NULL }, envp);
	fatal("init");

	/* not reached */
	return 1;
}
