#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mount.h>
#include <sys/syscall.h>
#include <sys/wait.h>

static void _fatal (char *usererr, char *syserr)
{
	static const char *msg = "\n\033[0;31m*** Fatal error, halting "
				 "(try Ctrl+Alt+Del) ***\033[0m\n";
	write(2, usererr, strlen(usererr));
	if (syserr) {
		write(2, ": ", 2);
		write(2, syserr, strlen(syserr));
	}
	write(2, msg, strlen(msg));

	while (1) sleep(60);
}

static void fatal (char *usererr)
{
	_fatal(usererr, strerror(errno));
}

static void fatal2 (char *usererr)
{
	_fatal(usererr, NULL);
}

static void lvm (char* argv[])
{
	pid_t pid;
	int i;

	pid = fork();
	if (pid < 0) fatal("cannot fork()");
	if (pid == 0) {
		execve("/sbin/lvm", argv, (char*[]) { NULL });
		fatal("cannot execute lvm"); 
	}

	while (wait(&i) != pid)
		;;
}

static void fmount (char *dev, char *path, char *fstype, int flags,
                    char *opts, char *err)
{
	if (mount(dev, path, fstype, flags, opts)) fatal(err);
}

static void funmount (char *path, char *err)
{
	if (umount(path)) fatal(err);
}

static char *getfilesystems (FILE **f)
{
	static char buffer[1024];

	if (!*f) {
		*f = fopen("/proc/filesystems", "r");
		if (!*f) fatal("opening /proc/filesystems failed");
	}

	while (fgets(buffer, sizeof(buffer), *f)) {
		if ((buffer[0] == '\t') &&
		    (buffer[strlen(buffer) - 1] == '\n')) {
			buffer[strlen(buffer) - 1] = '\0';
			return &buffer[1];
		}
	}

	return NULL;
}

int main (int argc, char **argv, char **envp)
{
	static const char *msg = "\033[0;36m*** Felix' LVM initrd ***\033[0m\n";
	static const char *msg2 = "\033[0;36m*** READY ***\033[0m\n";
	FILE *f = NULL;
	char *root, *rootflags, *rootfstype, *err;
	int i, fh;

	/* make compiler happy */
	argc = argc;
	argv = argv;

	/* some optical sugar :-) */
	write(1, msg, strlen(msg));

	/* mount ramdisk */
	fmount("/", "/", NULL, MS_REMOUNT, NULL, "remounting / failed");
	fmount("none", "/proc", "proc", 0, NULL, "mounting /proc failed");
	fmount("none", "/sys", "sysfs", 0, NULL, "mounting /sys failed");

	/* activate lvm */
	lvm((char*[]) { "vgscan", "--ignorelockingfailure", "--mknodes", "-P",
			NULL });
	lvm((char*[]) { "vgchange", "--ignorelockingfailure", "-a", "y", "-P",
			NULL });

	/* get options passed by boot loader to kernel image */
	root = getenv("lvmroot");
	if (!root) fatal2("got no lvmroot (try passing lvmroot=... to kernel "
			  "image in grub.cfg)");
	rootflags = getenv("lvmrootflags");
	rootfstype = getenv("lvmrootfstype");

	/* mount root */
	if (!rootfstype) rootfstype = getfilesystems(&f);
	while (rootfstype) {
		if (!mount(root, "/root", rootfstype, MS_RDONLY, rootflags))
			break;
		err = strerror(errno);
		write(2, "  ", 2);
		write(2, root, strlen(root));
		write(2, " is not a ", 10);
		write(2, rootfstype, strlen(rootfstype));
		write(2, ": ", 2);
		write(2, err, strlen(err));
		write(2, "\n", 1);
		rootfstype = getfilesystems(&f);
	}
	if (f) fclose(f);
	if (!rootfstype)
		fatal2("found no suitable root filesystem type");
	
	/* unmount everything inside ramdisk */
	funmount("/sys", "unmounting /sys failed");
	funmount("/proc", "unmounting /proc failed");

	/* swap root */
	if (syscall(SYS_pivot_root, "/root", "/root/tmp"))
		fatal("pivoting /root and /root/tmp failed");

	/* release everything opened in ramdisk */
	if (chdir("/")) fatal("changing directory to / failed");
	for (i = 0; i <= 2; ++i) {
		fh = open("/dev/console", (i ? O_WRONLY : O_RDONLY));
		if (fh == -1)
			fatal((i ? "opening /dev/console for writing failed" :
				   "opening /dev/console for reading failed"));
		if (dup2(fh, i) == -1)
			fatal((i ? "duping /dev/console for writing failed" :
				   "duping /dev/console for reading failed"));
		close(fh);
	}

	/* report success */
	write(1, msg2, strlen(msg2));

	/* unmount initrd in new shell called inside new root */
	/* and pass control to real init */
	execve("/bin/sh", (char*[]) { "sh", "-c", "umount /tmp; "
	       "exec init; exec /sbin/init; exec /bin/init; exec /etc/init; "
	       "exec /root/init; exec /init", NULL }, envp);
	fatal("init");

	/* not reached */
	return 1;
}
