#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <dirent.h>
#include <sys/stat.h>
#include "filo.h"

// for ioctl's
struct hd_geometry {
  unsigned char heads;
  unsigned char sectors;
  unsigned short cylinders;
  unsigned long start;
};

struct floppy_struct {
  unsigned int size,           /* nr of sectors total */
               sect,           /* sectors per track */
               head,           /* nr of heads */
               track,          /* nr of tracks */
               stretch;        /* !=0 means double track steps */
  unsigned char gap,           /* gap1 size */
                rate,          /* data rate. |= 0x40 for perpendicular */
                spec1,         /* stepping rate, head unload time */
                fmt_gap;       /* gap2 size */
  const char *name;            /* used only for predefined formats */
};

struct geometry {
  struct hd_geometry hd_geo;                     // geometry of kernel partition
  unsigned char biosdrive;                       // bios# of that partition
  char is_hd;                                    // is it a hard drive?
  int spb;                                       // sectors/block of partition
};

void _write (int fd, char *s)
{
  write(fd, s, strlen(s));
}

void die (char *s1, char *s2, char *s3, char *s4)
{
  _write(STDERR, s1);
  _write(STDERR, s2);
  _write(STDERR, s3);
  _write(STDERR, s4);
  _write(STDERR, "\nType filo without parameters for a short briefing.\n");
  exit(1);
}

void print (char *s)
{
  _write(STDOUT, s);
}

void usage ()
{
  print("usage: <boot-device> <loader> <kernel>[:biosdrive][;(=floppy)] [commandline]\ne.g.:\n"
        "filo /dev/hda /boot/filo.b /boot/vmlinuz:1 "
        "\"root=/dev/sda1 rootflags=data=journal console=ttyS0,19200,n,8 console=tty1\"\n");
  exit(1);
}

void strchr_del(char* haystack, char needle)
{
  static char *c;
  while ((c = strrchr(haystack, needle))) memmove(c, c+1, strlen(c+1));
}

int bmap2chs (int bmap, struct geometry *geo, int additional_sector)
{
  static int sector, head;

  bmap *= geo->spb;
  bmap += geo->hd_geo.start + additional_sector; // bmap = lba-sector

  sector = bmap % geo->hd_geo.sectors + 1; // sectors start at 1
  bmap /= geo->hd_geo.sectors;
  head = bmap % geo->hd_geo.heads;
  bmap /= geo->hd_geo.heads; // now bmap = cylinder

  // x86 is big-endian
  return (
          ((head & 255) << 24) +                 // dh = head
          ((geo->biosdrive & 255) << 16) +       // dl = drive
          ((bmap & 255) << 8) +                  // ch = low 8 bits of cylinder
          (sector & 63) +                        // cl = sector (0-5) and...
          ((bmap & 768) >> 2 )                   //    ...high bits of cyl (6-7)
         );
}

char* find_dev (char* path, dev_t major_minor)
{
  DIR *dir;
  static struct dirent *de;
  static struct stat st;
  static char buf[BUFFER_SIZE], *p;

  if(stat(path, &st) == -1) return NULL;

  if (S_ISDIR(st.st_mode)) {
    if (!(dir = opendir(path)))
      die("opendir(", path, "): ", strerror(errno));

    while ((de = readdir(dir))) {
      if ( (strcmp(de->d_name, ".")) &&
           (strcmp(de->d_name, "..")) &&
           (strcmp(de->d_name, "fd")) ) {
        strncpy(&buf[0], path, BUFFER_SIZE);
        strcat(&buf[0], "/");
        strncat(&buf[0], de->d_name, BUFFER_SIZE-strlen(&buf[0]));
        if (find_dev(&buf[0], major_minor)) return &buf[0];
        p = strrchr(&buf[0], '/');
        *p = 0;
      }
    }

    if (closedir(dir) == -1)
      die("closedir(", path, "): ", strerror(errno));
  }
  else if (S_ISBLK(st.st_mode)) {
    if (st.st_rdev == major_minor) return path;
  }
  return NULL;
}

void get_geo (struct geometry *geo, char* file)
{
  struct stat st;
  struct floppy_struct floppy;
  int i, ret;
  char *partition, *generic_floppy = NULL;

  if ((generic_floppy = strchr(file, ';'))) *generic_floppy = '\0';
  if ((partition = strchr(file, ':'))) {
    *partition = '\0';
    geo->biosdrive = atol(++partition);
  }
  else geo->biosdrive = 0;

  print("assuming ");
  print(file);
  print(" resides on bios drive #");
  print((geo->biosdrive)?(partition):("0"));
  print("\n");

  if (generic_floppy) {
    geo->hd_geo.sectors = 18;
    geo->hd_geo.cylinders = 80;
    geo->hd_geo.heads = 2;
    geo->hd_geo.start = 0;
    geo->is_hd = 0;
    geo->spb = 1;
    return;
  }

  if ((i = open(file, O_RDONLY)) == -1)
    die("open(", file, "): ", strerror(errno));
  if (ioctl(i, GET_BLKSIZE, &geo->spb) == -1)
    die("ioctl(", file, "): ", strerror(errno));
  if (close(i) == -1)
    die("close(", file, "): ", strerror(errno));
  geo->spb /= SECTOR_SIZE;

  if (stat(file, &st) == -1)
    die("stat(", file,"): ", strerror(errno));
  partition = find_dev(PATH_DEV, st.st_dev);
  if (!partition) die("no partition found for ", file, "\n", "");

  geo->is_hd = ((st.st_dev>>8) == 3) || ((st.st_dev>>8) == 22) ||
               ((st.st_dev>>8) == 33) || ((st.st_dev>>8) == 34) ||
               ((st.st_dev>>8) == 8);

  if ((i = open(partition, O_RDONLY)) == -1)
    die("open(", partition, "): ", strerror(errno));

  if (geo->is_hd)
    ret = ioctl(i, HDIO_GETGEO, &geo->hd_geo);
  else
    ret = ioctl(i, FDGETPRM, &floppy);
  if (ret == -1)  die("ioctl(", partition, "): ", strerror(errno));

  if (close(i) == -1) die("close(", partition, "): ", strerror(errno));

  // FIXME: determine geo->biosdrive through partition here
  if (geo->is_hd)
    geo->biosdrive += 0x80;
  else {
    geo->hd_geo.sectors = floppy.sect;
    geo->hd_geo.cylinders = floppy.track;
    geo->hd_geo.heads = floppy.head;
    geo->hd_geo.start = 0;
  }
#if 0
printf("\nDEBUG: %i %i %i %i %i %i %i\n",
  geo->hd_geo.sectors,
  geo->hd_geo.cylinders,
  geo->hd_geo.heads,
  geo->hd_geo.start,
  geo->biosdrive,
  geo->is_hd,
  geo->spb);
#endif
}

void write_int (struct geometry *geo, int bmap_fh, char* bmap_path, int value)
{
  static int ints = 0, count = 0, offs = 0, bmap = 0, chs;
  static int buffer[(SECTOR_SIZE/INT_SIZE)+1];

  if (!geo) {
    if (write(bmap_fh, &buffer[0], ints*INT_SIZE) == -1)
      die("write(", bmap_path, "): ", strerror(errno));
    return;
  }

  buffer[ints++] = value;

  if (ints == SECTOR_SIZE/INT_SIZE) {
    buffer[ints] = value;
    if (write(bmap_fh, &buffer[0], SECTOR_SIZE+INT_SIZE) == -1)
      die("write(", bmap_path, "): ", strerror(errno));

    buffer[0] = value;
    ints = 1;
    offs++;
    if ((offs >= geo->spb) || !bmap) {
      if (bmap) {
        bmap = ++count;
        offs = 0;
      }
      if (ioctl(bmap_fh, GET_BLK, &bmap) < 0)
        die("ioctl(", bmap_path, "): ", strerror(errno));
      if (!bmap) die("ioctl(", bmap_path, ") returns invalid bitmap\n", "");
    }

    chs = bmap2chs(bmap, geo, offs);

    if (lseek(bmap_fh, -2 * INT_SIZE, SEEK_CUR) == -1)
      die("lseek(", bmap_path, "): ", strerror(errno));

    if (write(bmap_fh, &chs, INT_SIZE) == -1)
      die("write(", bmap_path, "): ", strerror(errno));
  }
}

void append_bmap (struct geometry *geo, int bmap_fh, char* bmap_path,
                  int fh, char* file)
{
  int count, bmap, offs, chs;

  for (count = 0;;) {
    bmap = count;
    
    if (ioctl(fh, GET_BLK, &bmap) < 0)
      die("ioctl(", file, "): ", strerror(errno));

    if (!bmap) {
      write_int(geo, bmap_fh, bmap_path, 0);
      break;
    }

    count++;

    for (offs = 0; offs < geo->spb; offs++) {
      chs = bmap2chs(bmap, geo, offs);
      write_int(geo, bmap_fh, bmap_path, chs);
    }
  };
}

int make_bmap (struct geometry *geo, char* kernel_path, char* cmdline)
{
  char bmap_path[BUFFER_SIZE], cmdl_path[BUFFER_SIZE], buffer[CMDL_SIZE+1];
  int bmap_fh, fh;

  strncpy(&bmap_path[0], kernel_path, BUFFER_SIZE-6);
  strcat(&bmap_path[0], ".bmap");

  strncpy(&cmdl_path[0], kernel_path, BUFFER_SIZE-6);
  strcat(&cmdl_path[0], ".cmdl");

  print("creating ");
  print(bmap_path);
  print(" for ");
  print(kernel_path);
  print(" and ");
  print(cmdl_path);
  print("\n");

  // kernel
  if ((fh = open(kernel_path, O_RDONLY)) == -1)
    die("open(", kernel_path, "): ", strerror(errno));

  bmap_fh = open(bmap_path, O_WRONLY|O_CREAT|O_TRUNC, S_IRUSR|S_IWUSR);
  if (bmap_fh == -1) die("open(", bmap_path, "): ", strerror(errno));

  append_bmap(geo, bmap_fh, bmap_path, fh, kernel_path);

  if (close(fh) == -1)
    die("close(", kernel_path, "): ", strerror(errno));

  // commandline file
  if (strlen(cmdline) > CMDL_SIZE)
    print("commandline is too long\n");

  strncpy(&buffer[0], cmdline, CMDL_SIZE);
  strchr_del(&buffer[0], '\r');
  strchr_del(&buffer[0], '\n');

  if (strncmp("auto", &buffer[0], AUTO_SIZE-1)) {
    if (strlen(&buffer[0]) > CMDL_SIZE-AUTO_SIZE)
      print("prepending 'auto ' truncates commandline\n");
    else
      print("prepending 'auto ' to commandline\n");

    memmove(&buffer[AUTO_SIZE], &buffer[0], strlen(&buffer[0]));
    memcpy(&buffer[0], "auto ", AUTO_SIZE);
  }

  if ((fh = open(cmdl_path, O_WRONLY|O_CREAT|O_TRUNC, S_IRUSR|S_IWUSR)) == -1)
    die("open(", cmdl_path, "): ", strerror(errno));

  if (write(fh, &buffer[0], strlen(&buffer[0])+1) == -1)
    die("write(", cmdl_path, "): ", strerror(errno));

  append_bmap(geo, bmap_fh, bmap_path, fh, cmdl_path);
  write_int(NULL, bmap_fh, bmap_path, 0);

  print("commandline stored in ");
  print(cmdl_path);
  print(": ");
  print(&buffer[0]);
  print("\n");

  if (close(fh) == -1)
    die("close(", cmdl_path, "): ", strerror(errno));

  fh = 0;
  if (ioctl(bmap_fh, GET_BLK, &fh) < 0)
    die("ioctl(", bmap_path, "): ", strerror(errno));

  if (close(bmap_fh) == -1)
    die("close(", bmap_path, "): ", strerror(errno));

  return bmap2chs(fh, geo, 0);
}

void install_loader (char* boot, char* loader, int start_bmap, int is_fd)
{
  char buf[MBR_SIZE];
  int boot_fh, loader_fh, i;

  print("installing ");
  print(loader);
  print(" on ");
  print(boot);
  print("\n");

  if ((loader_fh = open(loader, O_RDONLY)) == -1)
    die("open(", loader, "): ", strerror(errno));

  if ((i = read(loader_fh, &buf[0], MBR_SIZE)) == -1)
    die("read(", loader, "): ", strerror(errno));

  if (close(loader_fh) == -1)
    die("close(", loader, "): ", strerror(errno));

  if ((boot_fh = open(boot, O_WRONLY)) == -1)
    die("open(", boot, "): ", strerror(errno));

  if (write(boot_fh, &buf[0], i) == -1)
    die("write(", boot, "): ", strerror(errno));

  if (lseek(boot_fh, MBR_SIZE - INT_SIZE, SEEK_SET) == -1)
    die("lssek(", boot, "): ", strerror(errno));

  if (write(boot_fh, &start_bmap, INT_SIZE) == -1)
    die("write(", boot, "): ", strerror(errno));

  if (is_fd) {
    if (lseek(boot_fh, SECTOR_SIZE - 2, SEEK_SET) == -1)
      die("lssek(", boot, "): ", strerror(errno));
    is_fd = BOOT_MAGIC;
    if (write(boot_fh, &is_fd, 2) == -1)
      die("write(", boot, "): ", strerror(errno));
  }

  if (close(boot_fh) == -1)
    die("close(", boot, "): ", strerror(errno));
}

int main (int argc, char* argv[])
{
  struct geometry geo;
  int bmap;

  if ((argc < 4) || (argc > 5)) usage();

  get_geo(&geo, argv[3]);

  if (argc == 4)
    bmap = make_bmap(&geo, argv[3], "");
  else
    bmap = make_bmap(&geo, argv[3], argv[4]);

  install_loader(argv[1], argv[2], bmap, !geo.is_hd);

  return 0;
}
