/*
#include <stdio.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <sys/uio.h>
#include <stdlib.h>
*/

#include <strings.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <netdb.h>
#include <resolv.h>

#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <pwd.h>

#define STDIN 0
#define STDOUT 1
#define STDERR 2

#define READ 0
#define WRITE 1

#define TCP "tcp"
#define HTTP "http"

#define BUFFER_SIZE 4096
#define USAGE \
"dyndns " VERSION "\n\n" \
"usage: dyndns [-d] [configfile]\n" \
"       dyndns -h\n\n" \
"  -d           points your hostname to 127.0.0.1, regardless of configfile or\n" \
"               external ip address (useful before shutting down your external\n" \
"               interface)\n" \
"  configfile   name of configfile\n" \
"  -h           this help\n\n" \
"see also http://www.ogris.de/dyndns/\n"

char *config = "/etc/dyndns.conf";
char *ifconfig = "/sbin/ifconfig";
char *interface = NULL;
char *dyndns = "members.dyndns.org";
char *update = "/nic/update";
char *syst = "dyndns";
char *wildcard = "NO";
char *backmx = "NO";
char *hostname = NULL;
char *mx = NULL;
char *ip = NULL;
char *http_proxy = NULL;
char *proxy_port = "3128";
char *user = NULL;
char *pass = NULL;
char *logfile = "/var/log/dyndns.log";
char *errfile = "/var/log/dyndns.err";
int logfd = STDOUT;
int errfd = STDERR;

struct stat st;

void print (const int fd, const char *s)
{
  write(fd, s, strlen(s));
}

void die (const char *s1, const char *s2)
{
  print(errfd, s1);
  if (s2) {
    write(errfd, ": ", 2);
    print(errfd, s2);
  }
  write(errfd, "\n", 1);
  exit(1);
}

char *readfd (int fd, char *buf)
{
  ssize_t ret;
  size_t len;

  for (len = 0;;) {
    buf = realloc(buf, len + BUFFER_SIZE);
    if (!buf) die("realloc()", strerror(errno));
    ret = read(fd, buf + len, BUFFER_SIZE);
    if (ret < 0) die("read()", strerror(errno));
    if (!ret) break;
    len += ret;
  }

  *(buf + len) = '\0';
  return buf;
}

char _b64char (char c)
{
  if ((c >= 0) && (c <= 25)) return 'A' + c;
  if ((c >= 26) && (c <=51)) return 'a' + c - 26;
  if ((c >= 52) && (c <= 61)) return '0' + c - 52;
  if (c == '+') return 62;
  return 63;
}

char *base64_encode (char *s)
{
  size_t len;
  char *ret, *c, *pos;

  for (len = 0, c = s; *c; ++c, ++len)
    ;;
  ret = malloc((len * 4) / 3 + 3 + len / 76);
  if (!ret) return NULL;

  for (c = s, pos = ret, len = 0; *c; len += 4) {
    if (len == 76) *pos++ = '\n';
    *pos++ = _b64char((*c & 0xfc) >> 2);
    *pos++ = _b64char(((*c & 0x3) << 4) | ((*(c+1) & 0xf0) >> 4));
    ++c;
    if (*c == 0) { *pos++ = '='; *pos++ = '='; break; }
    *pos++ = _b64char(((*c & 0xf) << 2) | ((*(c+1) & 0xc0) >> 6));
    ++c;
    if (*c == 0) { *pos++ = '='; break; }
    *pos++ = _b64char(*c++ & 0x3f);
  }

  *pos = '\0';
  return ret;
}

int whitespace (char c)
{
  if ((c == ' ') || (c == '\t') || (c == '=') || (c == ':') || (c == '#'))
    return 1;

  return 0;
}

char *setcfgopt (char **opt, char *start)
{
  char *right;

  right = strchr(start, '\n');
  if (right) *right = '\0';
  if (!whitespace(*start)) die("unknown configuration option", start);

  while (whitespace(*++start))
    ;;

  if (opt) *opt = start;
  if (right) return ++right;
  return (start + st.st_size);
}

void parsecfg (char *cfg)
{
  ssize_t ret;
  int fh;
  char *p, *left;

  fh = open(cfg, O_RDONLY);
  if (fh < 0) die(cfg, strerror(errno));

  if (fstat(fh, &st)) die("fstat()", strerror(errno));

  p = malloc(st.st_size + 1);
  if (!p) die("malloc()", strerror(errno));
  ret = read(fh, p, st.st_size);
  if (ret < 0) die("read()", strerror(errno));
  if (ret != (ssize_t) st.st_size) die(cfg, "can not read the whole file");
  close(fh);
  *(p + st.st_size) = '\0';

  for (left = p;;) {
    if (*left == '#') left = setcfgopt(NULL, left);

    else if (!strncmp(left, "ifconfig", 8))
      left = setcfgopt(&ifconfig, left + 8);

    else if (!strncmp(left, "interface", 9))
      left = setcfgopt(&interface, left + 9);

    else if (!strncmp(left, "dyndns", 6))
      left = setcfgopt(&dyndns, left + 6);

    else if (!strncmp(left, "update", 6))
      left = setcfgopt(&update, left + 6);

    else if (!strncmp(left, "system", 6))
      left = setcfgopt(&syst, left + 6);

    else if (!strncmp(left, "wildcard", 8))
      left = setcfgopt(&wildcard, left + 8);

    else if (!strncmp(left, "backmx", 6))
      left = setcfgopt(&backmx, left + 6);

    else if (!strncmp(left, "hostname", 8))
      left = setcfgopt(&hostname, left + 8);

    else if (!strncmp(left, "mx", 2))
      left = setcfgopt(&mx, left + 2);

    else if (!strncmp(left, "ip", 2))
      left = setcfgopt(&ip, left + 2);

    else if (!strncmp(left, "http_proxy", 10))
      left = setcfgopt(&http_proxy, left + 10);

    else if (!strncmp(left, "proxy_port", 10))
      left = setcfgopt(&proxy_port, left + 10);

    else if (!strncmp(left, "user", 4))
      left = setcfgopt(&user, left + 4);

    else if (!strncmp(left, "pass", 4))
      left = setcfgopt(&pass, left + 4);

    else if (!strncmp(left, "logfile", 7))
      left = setcfgopt(&logfile, left + 7);

    else if (!strncmp(left, "errfile", 7))
      left = setcfgopt(&errfile, left + 7);

    if (left >= p + st.st_size) break;
  }
}

void openlog (char *filename, int *oldfd)
{
  int newfd;

  if (*filename == '-') return;
  newfd = open(filename, O_CREAT | O_APPEND | O_WRONLY, S_IRUSR | S_IWUSR);
  if (newfd < 0) die(filename, strerror(errno));
  close(*oldfd);
  *oldfd = newfd;
}

int main (int argc, char **argv)
{
  struct protoent *pe;
  struct servent *se = NULL;
  struct hostent *he;
  struct sockaddr_in sin;
  struct passwd *pw;
  size_t len;
  pid_t pid;
  int sock, pfd[2], down = 0;
  char *secure, *buf, *args[3], *c;

  close(STDIN);

  for (sock = 1; sock < argc; ++sock) {
    if (!strncmp(argv[sock], "-h", 2)) {
      print(logfd, USAGE);
      return 0;
    }
    if (!strncmp(argv[sock], "-d", 2)) {
      down = 1;
      continue;
    }
    if (*argv[sock] != '-') {
      config = argv[sock];
      continue;
    }
    die(argv[sock], "unknown parameter\ntry dyndns -h for a short help");
  }

  /* read config file */
  parsecfg(config);

  /* reopen logs */
  openlog(logfile, &logfd);
  openlog(errfile, &errfd);

  /* check config */
  if (!pass) die("no password", NULL);
  if (ip && interface) die("ip and interface are mutually exclusive", NULL);
  if (!user) {
    pw = getpwuid(getuid());
    if (!pw) die("you do not exist", NULL);
    user = pw->pw_name;
  }

  /* encode user:pass using base64 */
  buf = malloc(strlen(user) + strlen(pass) + 2);
  if (!buf) die("malloc()", strerror(errno));
  strcpy(buf, user);
  strcat(buf, ":");
  strcat(buf, pass);
  secure = base64_encode(buf);

  /* determine ip address */
  if (!ip && interface) {
    if (pipe(pfd)) die("pipe()", strerror(errno));
    pid = fork();
    if (pid < 0) die("fork()", strerror(errno));
    if (!pid) {
      close(STDOUT);
      close(STDERR);
      close(pfd[READ]);
      if (dup2(pfd[WRITE], STDOUT) < 0) die("dup2()", strerror(errno));
      if (dup2(pfd[WRITE], STDERR) < 0) die("dup2()", strerror(errno));

      args[0] = ifconfig;
      args[1] = interface;
      args[2] = NULL;
      execve(ifconfig, args, NULL);
      die("execve()", strerror(errno));
    }
    close(pfd[WRITE]);
    if (wait(&pid) < 0) die("wait()", strerror(errno));
    if (!WIFEXITED(pid)) die(ifconfig, "killed");

    buf = readfd(pfd[READ], buf);

    if (WEXITSTATUS(pid)) die(ifconfig, (*buf)?("unknown error"):(buf));

    ip = strstr(buf, "inet addr");
    if (!ip) die(ifconfig, "no inet addr");
    for (ip += 10; whitespace(*ip); ++ip)
      ;;
    for (c = ip; !whitespace(*c); ++c)
      ;;
    *c = '\0';
  }

  /* determine hostname */
  if (!hostname) {
    hostname = malloc(BUFFER_SIZE);
    if (!hostname) die("malloc()", strerror(errno));
    if (gethostname(hostname, BUFFER_SIZE))
      die("gethostname()", strerror(errno));
    len = strlen(hostname);
    *(hostname + len) = '.';
    ++len;
    *(hostname + len) = '\0';
    if (getdomainname(hostname + len, BUFFER_SIZE - len - 1))
      die("getdomainname()", strerror(errno));
  }

  if (!mx) mx = hostname;
  if (down) ip = "127.0.0.1";

  /* connect to dyndns through proxy if set */
  if (http_proxy) he = gethostbyname(http_proxy);
  else he = gethostbyname(dyndns);
  if (!he) die((http_proxy)?(http_proxy):(dyndns), hstrerror(h_errno));

  pe = getprotobyname(TCP);
  if (!pe) die("no such protocol", TCP);

  if (!http_proxy) {
    se = getservbyname(HTTP, pe->p_name);
    if (!se) die("no such service", http_proxy);
  }

  memcpy((char*) &sin.sin_addr, he->h_addr, he->h_length);
  sin.sin_port = (http_proxy)?(htons(atoi(proxy_port))):(se->s_port);
  sin.sin_family = PF_INET;

  if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    die("socket()", strerror(errno));
    
  if (connect(sock, (struct sockaddr*) &sin, sizeof(sin)))
    die("connect()", strerror(errno));

  print(sock, "GET http://");
  print(sock, dyndns);
  print(sock, update);
  print(sock, "?system=");
  print(sock, syst);
  print(sock, "&hostname=");
  print(sock, hostname);
  if (ip) {
    print(sock, "&myip=");
    print(sock, ip);
  }
  print(sock, "&wildcard=");
  print(sock, wildcard);
  print(sock, "&mx=");
  print(sock, mx);
  print(sock, "&backmx=");
  print(sock, backmx);
  print(sock, " HTTP/1.1\r\nHost: ");
  print(sock, dyndns);
  print(sock, "\r\nUser-Agent: FJO's dyndns " VERSION
    "\r\nAuthorization: Basic ");
  print(sock, secure);
  print(sock, "\r\nContent-type: application/x-www-form-urlencoded\r\n"
    "Connection: close\r\n\r\n");

  free(secure);
  buf = readfd(sock, buf);
  close(sock);
  sock = 0;

  if (strstr(buf, "good")) print(logfd, "ok.\n");
  else if (strstr(buf, "nochg")) print(logfd, "no change needed.\n");
  else {
    print(errfd, buf);
    sock = 1;
  }

  close(errfd);
  close(logfd);
  free(buf);

  return sock;
}
