#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <GL/gl.h>
#include <GL/glx.h>
#include "plugin.h"

#define TITLE "Simple PCM data visualization plugin"
#define PCM_SAMPLES 512
#define PCM_CHANNELS 2
#define MERGE 16

#define WIDTH 1024
#define HEIGHT 768

/* globale Variablen */
static gint16 global_pcm_data[PCM_CHANNELS][PCM_SAMPLES];
static pthread_mutex_t global_pcm_data_mutex;
static int global_playing_flag;
static pthread_t display_thread;

/* Forward-Definition */
static void disable_myself (char *);

/* OpenGL-Zeichenroutine im eigenen Thread */
static void *display_thread_func (void *p)
{
  int i, j, channel;
  gint16 pcm_data[PCM_CHANNELS][WIDTH], max;
  gint64 value;

  /* Variablen fr X11 */
  Display *xdpy;
  Window root, win;
  XVisualInfo *xvinfo;
  GLXContext context;
  XSetWindowAttributes winattrs;
  int screen, attributes[] = { GLX_RGBA, GLX_DOUBLEBUFFER, 0 };
  XEvent event;
  Atom wm_delete_window_atom;

  /* make compiler happy */
  p = p;

  /* zum X-Server verbinden */
  if (!(xdpy = XOpenDisplay(NULL))) {
    disable_myself("can not open display");
    return NULL;
  }

  /* Bildschirm auswhlen */
  screen = DefaultScreen(xdpy);
  if (!(xvinfo = glXChooseVisual(xdpy, screen, attributes))) {
    disable_myself("can not choose visual");
    XCloseDisplay(xdpy);
    return NULL;
  }

  root = RootWindow(xdpy, screen);

  winattrs.colormap = XCreateColormap(xdpy, root, xvinfo->visual,
                                      AllocNone);
  winattrs.event_mask = StructureNotifyMask;

  /* Fenster anlegen */
  win = XCreateWindow(xdpy, root, 0, 0, WIDTH, HEIGHT, 0, xvinfo->depth,
                      InputOutput, xvinfo->visual,
                      CWColormap | CWEventMask,
                      &winattrs);

  /* Fenstername */
  XChangeProperty(xdpy, win, XA_WM_NAME, XA_STRING, 8, 0,
                  (unsigned char*) TITLE, strlen(TITLE));

  /* OpenGL-Kontext anlegen */
  context = glXCreateContext(xdpy, xvinfo, 0, True);
  glXMakeCurrent(xdpy, win, context);

  XFree(xvinfo);

  /* Handle zum Fenster schliessen */
  wm_delete_window_atom = XInternAtom(xdpy, "WM_DELETE_WINDOW", False);
  XSetWMProtocols(xdpy, win, &wm_delete_window_atom, 1);

  /* Fenster anzeigen */
  XMapWindow(xdpy, win);

  /* OpenGL initialisieren */
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0.0, (GLfloat) WIDTH, 0.0, (GLfloat) HEIGHT, 1.0, -1.0);
  glClearColor(0.0, 0.0, 0.0, 0.0);

  /* Variablen initialisieren */
  max = 1000;
  memset(pcm_data, 0, sizeof(gint16) * PCM_SAMPLES * PCM_CHANNELS);

  /* Grafik darstellen */
  while (global_playing_flag) {
    /* auf Daten warten */
    pthread_mutex_lock(&global_pcm_data_mutex);

    /* X11 Events verarbeiten */
    while (XPending(xdpy)) {
      XNextEvent(xdpy, &event);
      switch (event.type) {
        /* Fenstergrsse wurde verndert */
        case ConfigureNotify:
          glViewport(0, 0, event.xconfigure.width,
                     event.xconfigure.height);
          break;
        /* Fenster schliessen? */
        case ClientMessage:
          if ((Atom) event.xclient.data.l[0] == wm_delete_window_atom)
            disable_myself(NULL);
          break;
        default:
          break;
      }
    }

    /* alte Daten schieben */
    for (channel = 0; channel < PCM_CHANNELS; ++channel)
      memmove(&pcm_data[channel][0],
              &pcm_data[channel][PCM_SAMPLES / MERGE],
              sizeof(gint16) * (WIDTH - PCM_SAMPLES / MERGE));

    /* neue Daten zusammenfassen, in lokale Variable schreiben und
       dabei das Maximum bestimmen */
    for (channel = 0; channel < PCM_CHANNELS; ++channel)
      for (i = 0; i < PCM_SAMPLES / MERGE; ++i) {
        value = 0;
        for (j = 0; j < MERGE; ++j)
          value += global_pcm_data[channel][i * MERGE + j];
        value /= MERGE;
        if (value < 0) value *= -1;
        pcm_data[channel][WIDTH - PCM_SAMPLES / MERGE + i] = value;
        if (max < value) max = value;
      }

    /* PCM-Daten anzeigen */
    glClear(GL_COLOR_BUFFER_BIT);
    glColor3f(0.0, 0.9, 1.0);
    for (i = 0; i < WIDTH; ++i) 
      glRectf((GLfloat) i,
              HEIGHT / 2.0 + pcm_data[0][i] * HEIGHT / (2.0 * max),
              (GLfloat) (i + 1), HEIGHT / 2.0);

    glColor3f(1.0, 0.9, 1.0);
    for (i = 0; i < WIDTH; ++i) 
      glRectf((GLfloat) i, HEIGHT / 2.0,
              (GLfloat) (i + 1),
              HEIGHT / 2.0 - pcm_data[0][i] * HEIGHT / (2.0 * max));

    glXSwapBuffers(xdpy, win);
  }

  /* aufrumen, Fenster schliessen, Verbindung zum X-Server schliessen */
  glXMakeCurrent(xdpy, 0, NULL);
  glXDestroyContext(xdpy, context);
  XDestroyWindow(xdpy, win);
  XFreeColormap(xdpy, winattrs.colormap);
  XCloseDisplay(xdpy);

  return NULL;
}

static void playback_start ()
{
  global_playing_flag = 1;
  pthread_mutex_init(&global_pcm_data_mutex, NULL);
  pthread_mutex_lock(&global_pcm_data_mutex);
  pthread_create(&display_thread, NULL, display_thread_func, NULL);
}

static void playback_stop ()
{
  global_playing_flag = 0;
  pthread_mutex_unlock(&global_pcm_data_mutex);
  pthread_join(display_thread, NULL);
  pthread_mutex_destroy(&global_pcm_data_mutex);
}

static void render_pcm (gint16 pcm_data[2][PCM_SAMPLES])
{
  memcpy(global_pcm_data, pcm_data,
         sizeof(gint16) * PCM_SAMPLES * PCM_CHANNELS);
  pthread_mutex_unlock(&global_pcm_data_mutex);
}

static VisPlugin plugin = {
  .handle              = NULL,
  .filename            = NULL,
  .xmms_session        = 0,
  .description         = TITLE,
  .num_pcm_chs_wanted  = PCM_CHANNELS,
  .num_freq_chs_wanted = 0,
  .init                = NULL,
  .cleanup             = NULL,
  .about               = NULL,
  .configure           = NULL,
  .disable_plugin      = NULL,
  .playback_start      = playback_start,
  .playback_stop       = playback_stop,
  .render_pcm          = render_pcm,
  .render_freq         = NULL,
};

VisPlugin* get_vplugin_info ()
{
  return &plugin;
}

static void disable_myself (char *msg)
{
  if (msg) fputs(msg, stderr);
  if (plugin.disable_plugin) plugin.disable_plugin(&plugin);
}
