#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <GL/gl.h>
#include <GL/glx.h>
#include "plugin.h"

#define TITLE "Simple spectral visualization plugin"
#define FREQ_SAMPLES 256
#define FREQ_CHANNELS 2

#define WIDTH (2 * FREQ_SAMPLES)
#define HEIGHT 768

/* globale Variablen */
static gint16 global_freq_data[FREQ_CHANNELS][FREQ_SAMPLES];
static pthread_mutex_t global_freq_data_mutex;
static int global_playing_flag;
static pthread_t display_thread;

/* Forward-Definition */
static void disable_myself (char *);

/* OpenGL-Zeichenroutine im eigenen Thread */
static void *display_thread_func (void *p)
{
  int i, channel;
  gint16 freq_data[FREQ_CHANNELS][FREQ_SAMPLES], max;

  /* Variablen fr X11 */
  Display *xdpy;
  Window root, win;
  XVisualInfo *xvinfo;
  GLXContext context;
  XSetWindowAttributes winattrs;
  int screen, attributes[] = { GLX_RGBA, GLX_DOUBLEBUFFER, 0 };
  XEvent event;
  Atom wm_delete_window_atom;

  /* make compiler happy */
  p = p;

  /* zum X-Server verbinden */
  if (!(xdpy = XOpenDisplay(NULL))) {
    disable_myself("can not open display");
    return NULL;
  }

  /* Bildschirm auswhlen */
  screen = DefaultScreen(xdpy);
  if (!(xvinfo = glXChooseVisual(xdpy, screen, attributes))) {
    disable_myself("can not open visual");
    XCloseDisplay(xdpy);
    return NULL;
  }

  root = RootWindow(xdpy, screen);

  winattrs.colormap = XCreateColormap(xdpy, root, xvinfo->visual,
                                      AllocNone);
  winattrs.event_mask = StructureNotifyMask;

  /* Fenster anlegen */
  win = XCreateWindow(xdpy, root, 0, 0, WIDTH, HEIGHT, 0, xvinfo->depth,
                      InputOutput, xvinfo->visual,
                      CWColormap | CWEventMask,
                      &winattrs);

  /* Fenstername */
  XChangeProperty(xdpy, win, XA_WM_NAME, XA_STRING, 8, 0,
                  (unsigned char*) TITLE, strlen(TITLE));

  /* OpenGL-Kontext anlegen */
  context = glXCreateContext(xdpy, xvinfo, 0, True);
  glXMakeCurrent(xdpy, win, context);

  XFree(xvinfo);

  /* Handle zum Fenster schliessen */
  wm_delete_window_atom = XInternAtom(xdpy, "WM_DELETE_WINDOW", False);
  XSetWMProtocols(xdpy, win, &wm_delete_window_atom, 1);

  /* Fenster anzeigen */
  XMapWindow(xdpy, win);

  /* OpenGL initialisieren */
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0.0, (GLfloat) WIDTH, 0.0, (GLfloat) HEIGHT, 1.0, -1.0);
  glClearColor(0.0, 0.0, 0.0, 0.0);

  /* Variablen initialisieren */
  max = 1000;
  memset(freq_data, 0, sizeof(gint16) * FREQ_SAMPLES * FREQ_CHANNELS);

  /* Grafik darstellen */
  while (global_playing_flag) {
    /* auf Daten warten */
    pthread_mutex_lock(&global_freq_data_mutex);

    /* X11 Events verarbeiten */
    while (XPending(xdpy)) {
      XNextEvent(xdpy, &event);
      switch (event.type) {
        /* Fenstergrsse wurde verndert */
        case ConfigureNotify:
          glViewport(0, 0, event.xconfigure.width,
                     event.xconfigure.height);
          break;
        /* Fenster schliessen? */
        case ClientMessage:
          if ((Atom) event.xclient.data.l[0] == wm_delete_window_atom)
            disable_myself(NULL);
          break;
        default:
          break;
      }
    }

    /* Frequenzspektren in lokale Variable kopieren und
       dabei das Maximum bestimmen */
    for (channel = 0; channel < FREQ_CHANNELS; ++channel)
      for (i = 0; i < FREQ_SAMPLES; ++i) {
        freq_data[channel][i] = global_freq_data[channel][i];
        if (freq_data[channel][i] < 0) freq_data[channel][i] = 0;
        if (max < freq_data[channel][i]) max = freq_data[channel][i];
      }

    /* Spektren anzeigen */
    glClear(GL_COLOR_BUFFER_BIT);
    glColor3f(0.0, 0.9, 1.0);
    for (channel = 0; channel < FREQ_CHANNELS; ++channel) {
      for (i = 0; i < FREQ_SAMPLES; ++i) 
        glRectf((GLfloat) (channel * FREQ_SAMPLES + i),
                (GLfloat) (freq_data[channel][i] * HEIGHT / max),
                (GLfloat) (channel * FREQ_SAMPLES + i + 1), 0.0);

      glColor3f(1.0, 0.9, 1.0);
    }

    glXSwapBuffers(xdpy, win);
  }

  /* aufrumen, Fenster schliessen, Verbindung zum X-Server schliessen */
  glXMakeCurrent(xdpy, 0, NULL);
  glXDestroyContext(xdpy, context);
  XDestroyWindow(xdpy, win);
  XFreeColormap(xdpy, winattrs.colormap);
  XCloseDisplay(xdpy);

  return NULL;
}

static void playback_start ()
{
  global_playing_flag = 1;
  pthread_mutex_init(&global_freq_data_mutex, NULL);
  pthread_mutex_lock(&global_freq_data_mutex);
  pthread_create(&display_thread, NULL, display_thread_func, NULL);
}

static void playback_stop ()
{
  global_playing_flag = 0;
  pthread_mutex_unlock(&global_freq_data_mutex);
  pthread_join(display_thread, NULL);
  pthread_mutex_destroy(&global_freq_data_mutex);
}

static void render_freq (gint16 freq_data[2][FREQ_SAMPLES])
{
  memcpy(global_freq_data, freq_data,
         sizeof(gint16) * FREQ_SAMPLES * FREQ_CHANNELS);
  pthread_mutex_unlock(&global_freq_data_mutex);
}

static VisPlugin plugin = {
  .handle              = NULL,
  .filename            = NULL,
  .xmms_session        = 0,
  .description         = TITLE,
  .num_pcm_chs_wanted  = 0,
  .num_freq_chs_wanted = FREQ_CHANNELS,
  .init                = NULL,
  .cleanup             = NULL,
  .about               = NULL,
  .configure           = NULL,
  .disable_plugin      = NULL,
  .playback_start      = playback_start,
  .playback_stop       = playback_stop,
  .render_pcm          = NULL,
  .render_freq         = render_freq,
};

VisPlugin* get_vplugin_info ()
{
  return &plugin;
}

static void disable_myself (char *msg)
{
  if (msg) fputs(msg, stderr);
  if (plugin.disable_plugin) plugin.disable_plugin(&plugin);
}
