#include <stdio.h>
#include <math.h>
#include <GL/gl.h>
#ifdef MACOS
#  include <GLUT/glut.h>
#else
#  include <GL/glut.h>
#endif

#define TABLE_SIZE 9000

enum Listobjects {
  LIST_ROPE,
  LIST_BAR,
  LIST_CURVE,
  LIST_MAX,
};

static GLfloat diffuseLight[4] = { 1.0f, 1.0f, 1.0f, 1.0f };

static const GLfloat myWidth = 1.0f,
                     myHeight = 1.0f;

static GLuint lists[LIST_MAX];

static GLfloat* mySin_init () {
  static GLfloat table[TABLE_SIZE];
  unsigned int i;

  for (i = 0; i < TABLE_SIZE; ++i)
    table[i] = sin(M_PI * i * 0.5f / TABLE_SIZE);
  return table;
}

static GLfloat mySin (GLfloat deg) {
  static GLfloat *table;
  int degree, degree2;

  if (!table) table = mySin_init();
  degree = ((int) (deg * TABLE_SIZE / 90)) % (4 * TABLE_SIZE);
  degree2 = 1 + ((int) (deg * TABLE_SIZE / 90)) % TABLE_SIZE;
  if (degree >= 3 * TABLE_SIZE) return -table[TABLE_SIZE - degree2];
  if (degree >= 2 * TABLE_SIZE) return -table[degree2];
  if (degree >= TABLE_SIZE) return table[TABLE_SIZE - degree2];
  return table[degree2];
}

static GLfloat myCos (GLfloat deg) {
  return mySin(deg + 90.0f);
}

/* malt ein Rohr der Lnge l entlang der x-Achse,
 * Schwerpunkt im Ursprung,
 * mit Radius r und Tesselationfaktor d
 */
static void myTube (GLfloat l, GLfloat r, GLint d) {
  GLfloat p[4][3], n[4][3], angle;
  GLint i;

  glPushMatrix();

  /* Idee: Vektoren p[4] definieren eine "Deckplatte" des Rohres */
  /* bzw. im Querschnitt ein "Tortendreieck"                     */

  /* x-Koordinaten der Eckpunkte */
  p[0][0] = l / 2;
  p[1][0] = l / 2;
  p[2][0] = -l / 2;
  p[3][0] = -l / 2;

  /* ...der Normalenvektoren */
  n[0][0] = 0.0f;
  n[1][0] = 0.0f;
  n[2][0] = 0.0f;
  n[3][0] = 0.0f;

  /* y-Koordinaten der Eckpunkte */
  p[0][1] = myCos(360.0f / d) * r;
  p[1][1] = p[0][1];
  p[2][1] = p[0][1];
  p[3][1] = p[0][1];

  /* ...der Normalenvektoren */
  n[0][1] = p[0][1];
  n[1][1] = p[1][1];
  n[2][1] = p[2][1];
  n[3][1] = p[3][1];

  /* z-Koordinaten der Eckpunkte */
  p[0][2] = mySin(360.0f / d) * r;
  p[1][2] = -p[0][2];
  p[2][2] = -p[0][2];
  p[3][2] = p[0][2];

  /* ...der Normalenvektoren */
  n[0][2] = p[0][2];
  n[1][2] = p[1][2];
  n[2][2] = p[2][2];
  n[3][2] = p[3][2];

  angle = 360.0f / d;

  for (i = 0; i < d; ++i) {
    glBegin(GL_QUADS);
      glNormal3fv(n[0]);
      glVertex3fv(p[0]);
      glNormal3fv(n[1]);
      glVertex3fv(p[1]);
      glNormal3fv(n[2]);
      glVertex3fv(p[2]);
      glNormal3fv(n[3]);
      glVertex3fv(p[3]);
    glEnd();
    /* Tortenstück um die x-Achse weiterdrehen */
    glRotatef(angle, 1.0f, 0.0f, 0.0f);
  }

  glPopMatrix();
}

/* konstruiert ein um 90 Grad gebogenes Eckstck eines Rohres
 * Radius r1 der Hauptkrmmung, Radius r2 des Rohres (=r1 von myTube())
 * Tesselationfaktor d
 * Mittelpunktslinie beginnt im Ursprung und verluft in der positiven
 * xy-Ebene endet bei x=r1, y=r1, z=0
 */
static void myMuffenStuck (GLfloat r1, GLfloat r2, GLint d) {
  GLfloat p[4][3], n[4][3], angle, angle_deg, costmp, sintmp,
          sintmp_r1, costmp_r1;
  GLint i, j;

  glPushMatrix();

  angle = 2.0f * M_PI / d;
  angle_deg = 360.0f / d;
  costmp = 1.0f - myCos(angle_deg);
  sintmp = mySin(angle_deg);
  sintmp_r1 = sintmp * r1; /* = r1 * sin(2 Pi / d)      */
  costmp_r1 = costmp * r1; /* = r1 - r1 * cos(2 Pi / d)
                              = Hhenanstieg durch Hauptkrmmung
                                pro Deckplatte */

  /* Idee: Vektoren p[4] definieren "Deckplatten" des gekrmmten Rohres
   * zur Vorstellung gehen wir von einem Punkt mit grossem positiven y-Wert
   * aus und blicken in negative y-Richtung auf die xz-Ebene
   * jede Deckplatte wird dann ccw definiert mit p[0] links unten
   * der Normalenvektor fr jeden Vertex ergibt sich als Differenz zwischen
   * dem Vertex und dem korrelierenden Punkt auf der Mittelpunktslinie
   */
  for (i = 0; i < d/4; ++i) { /* Schleife fr Hauptkrmmung                  */
    p[1][2] = 0.0f;           /* z-Koordinate rechte untere Ecke             */
    p[1][1] = r2;             /* y-Koordinate rechte untere Ecke             */
    for (j = 0; j < d; ++j) { /* Schleife fr Rohr                           */
      /* z */
      /* linke untere Ecke ist rechte untere Ecke der Vorgngerplatte */
      p[0][2] = p[1][2];
      /* rechte untere Ecke (im Querschnitt wie dreieckiges Tortenstck */
      p[1][2] = mySin(angle_deg * (j + 1)) * r2;
      /* rechte obere = rechte untere Ecke (da parallel zur x-Achse) */
      p[2][2] = p[1][2];
      /* linke obere = linke untere Ecke */
      p[3][2] = p[0][2];

      /* y */
      /* linke untere Ecke ist rechte untere Ecke der Vorgngerplatte */
      p[0][1] = p[1][1];
      /* rechte untere Ecke (im Querschnitt wie dreieckiges Tortenstck */
      p[1][1] = myCos(angle_deg * (j + 1)) * r2;
      /* rechte obere = rechte untere Ecke (da parallel zur x-Achse) */
      p[2][1] = p[1][1];
      /* PLUS Hhenanstieg durch Hauptkrmmung */
      p[2][1] += costmp_r1 - p[2][1] * costmp;

      /* linke obere = linke untere Ecke */
      p[3][1] = p[0][1];
      /* PLUS Hhenanstieg durch Hauptkrmmung */
      p[3][1] += costmp_r1 - p[3][1] * costmp;

      /* x */
      /* Hauptlinie beginnt im Ursprung */
      p[0][0] = 0.0f;
      p[1][0] = 0.0f;
      /* x-Koordinaten der rechten und linken oberen Ecke jetzt in xy-Ebene
         betrachtet ergibt Tortendreick der Hauptkrmmung UNTER Beachtung
         der Hhe des jeweiligen Punktes                                               */
      p[2][0] = sintmp_r1 - p[2][1] * sintmp;
      p[3][0] = sintmp_r1 - p[3][1] * sintmp;

      /* Normalenvektoren */
      /* z = z des Vertex */
      n[0][2] = p[0][2];
      n[1][2] = p[1][2];
      n[2][2] = p[2][2];
      n[3][2] = p[3][2];

      /* y = y des Vertex - r1*(1-cos) */
      n[0][1] = p[0][1] - costmp_r1;
      n[1][1] = p[1][1] - costmp_r1;
      n[2][1] = p[2][1] - costmp_r1;
      n[3][1] = p[3][1] - costmp_r1;

      /* x = x des Vertex - r1*sin */
      n[0][0] = p[0][0] - sintmp_r1;
      n[1][0] = p[1][0] - sintmp_r1;
      n[2][0] = p[2][0] - sintmp_r1;
      n[3][0] = p[3][0] - sintmp_r1;

      glBegin(GL_QUADS);
        glNormal3fv(n[0]);
        glVertex3fv(p[0]);
        glNormal3fv(n[1]);
        glVertex3fv(p[1]);
        glNormal3fv(n[2]);
        glVertex3fv(p[2]);
        glNormal3fv(n[3]);
        glVertex3fv(p[3]);
      glEnd();
    }
    /* in xy-Ebene nach rechts oben verschieben... */
    glTranslatef(sintmp_r1, costmp_r1, 0.0f);
    /* ...und passend reindrehen                   */
    glRotatef(angle_deg, 0.0f, 0.0f, 1.0f);
  }

  glPopMatrix();
}

void manager_ausgelenkter_faden (GLfloat auslenkung, int pos)
{
  glPushMatrix();
  glColor4f(1.0f, 1.0f, 1.0f, 0.55f);
  glTranslatef(0.275f,
    1.06f - myCos(auslenkung) * 0.275f * mySin(55.0f) / myCos(55.0f),
    mySin(auslenkung) * 0.275f * mySin(55.0f) / myCos(55.0f) * pos);
  glRotatef(55.0f, 0.0f, 0.0f, 1.0f);
  glRotatef(auslenkung * pos, -myCos(55.0f), mySin(55.0f), 0.0f);
  myTube(0.55f / myCos(55.0f), 0.002f, 18);
  glPopMatrix();
}

void manager_display (GLfloat ypos, GLfloat zpos, GLfloat r, GLfloat lr,
                      GLfloat dlr, GLfloat dlg, GLfloat dlb,
                      GLfloat auslenkung, GLfloat auslenkung2)
{
  GLfloat scale = 10.0f - fabs(zpos);
  GLfloat posLight[4] = { .0f, 0.20f, .0f, 1.0f },
    dirLight[3] = { .0f, -.20f, .0f };

  diffuseLight[0] = dlr;
  diffuseLight[1] = dlg;
  diffuseLight[2] = dlb;

  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glTranslatef(0.0f, ypos, zpos);
  glRotatef(r, 0.0f, 1.0f, 0.0f);
  glScalef(scale, scale, scale);

  glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuseLight);
  /* Licht */
  posLight[0] = mySin(lr) * 1.0f;
  posLight[2] = myCos(lr) * 1.0f;
  dirLight[0] = -posLight[0] / (sqrt(posLight[0] * posLight[0] +
                                     posLight[2] * posLight[2]));
  dirLight[2] = -posLight[2] / (sqrt(posLight[0] * posLight[0] +
                                     posLight[2] * posLight[2]));
  glLightfv(GL_LIGHT0, GL_POSITION, posLight);
  glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, dirLight);

  /* Taschenlampe */
  glPushMatrix();
  glTranslatef(posLight[0], posLight[1], posLight[2]);
  glRotatef(lr-90.0f, 0.0f, 1.0f, 0.0f);
  glColor3f(0.0f, 0.0f, 0.0f);
  myTube(0.2f, 0.01f, 36);
  glPopMatrix();

  /* linke Wand */
  glColor3f(1.0f, 0.0f, 0.0f);
  glBegin(GL_POLYGON);
    glNormal3f(1.0f, 0.0f, 0.0f);
    glVertex3f(-myWidth, -myHeight, +2.0f);
    glVertex3f(-myWidth, -myHeight, -2.0f);
    glVertex3f(-myWidth, +myHeight, -2.0f);
    glVertex3f(-myWidth, +myHeight, +2.0f);
  glEnd();
  /* hintere Wand */
  glBegin(GL_POLYGON);
    glNormal3f(0.0f, 0.0f, 1.0f);
    glVertex3f(-myWidth, -myHeight, -1.90f);
    glVertex3f(+myWidth, -myHeight, -1.90f);
    glVertex3f(+myWidth, +myHeight, -1.90f);
    glVertex3f(-myWidth, +myHeight, -1.90f);
  glEnd();
  /* rechte Wand */
  glBegin(GL_POLYGON);
    glNormal3f(-1.0f, 0.0f, 0.0f);
    glVertex3f(+myWidth, -myHeight, -2.0f);
    glVertex3f(+myWidth, -myHeight, +2.0f);
    glVertex3f(+myWidth, +myHeight, +2.0f);
    glVertex3f(+myWidth, +myHeight, -2.0f);
  glEnd();
  glPushMatrix();

  /* Gestell */
  /* hinten unten */
  glTranslatef(0.0f, 0.0f, -0.54f);
  glCallList(lists[LIST_BAR]);
  glTranslatef(0.5f, 0.0f, 0.0f);
  glCallList(lists[LIST_CURVE]);
  glTranslatef(-1.0f, 0.0f, 0.0f);
  glRotatef(180.0f, 0.0f, 1.0f, 0.0f);
  glCallList(lists[LIST_CURVE]);

  /* vorne unten */
  glTranslatef(-0.5f, 0.0f, -1.08f);
  glCallList(lists[LIST_BAR]);
  glTranslatef(0.5f, 0.0f, 0.0f);
  glCallList(lists[LIST_CURVE]);
  glTranslatef(-1.0f, 0.0f, 0.0f);
  glRotatef(180.0f, 0.0f, 1.0f, 0.0f);
  glCallList(lists[LIST_CURVE]);

  glPopMatrix();
  glPushMatrix();

  /* hinten rechts */
  glTranslatef(0.54f, 0.54f, -0.54f);
  glRotatef(90.0f, 0.0f, 0.0f, 1.0f);
  glCallList(lists[LIST_BAR]);

  /* hinten links */
  glTranslatef(0.0f, 1.08f, 0.0f);
  glCallList(lists[LIST_BAR]);

  /* vorne links */
  glTranslatef(0.0f, 0.0f, 1.08f);
  glCallList(lists[LIST_BAR]);

  /* vorne rechts */
  glTranslatef(0.0f, -1.08f, 0.0f);
  glCallList(lists[LIST_BAR]);

  glPopMatrix();
  glPushMatrix();

  /* links oben */
  glTranslatef(-0.54f, 1.08f, 0.0f);
  glRotatef(90.0f, 0.0f, 1.0f, 0.0f);
  glCallList(lists[LIST_BAR]);
  glTranslatef(-0.54f, -0.04f, 0.0f);
  glRotatef(180.0f, 1.0f, 1.0f, 0.0f);
  glCallList(lists[LIST_CURVE]);
  glTranslatef(0.0f, 1.08f, 0.0f);
  glRotatef(180.0f, 1.0f, 0.0f, 0.0f);
  glCallList(lists[LIST_CURVE]);

  glPopMatrix();
  glPushMatrix();

  /* rechts oben */
  glTranslatef(0.54f, 1.08f, 0.0f);
  glRotatef(90.0f, 0.0f, 1.0f, 0.0f);
  glCallList(lists[LIST_BAR]);
  glTranslatef(-0.54f, -0.04f, 0.0f);
  glRotatef(180.0f, 1.0f, 1.0f, 0.0f);
  glCallList(lists[LIST_CURVE]);
  glTranslatef(0.0f, 1.08f, 0.0f);
  glRotatef(180.0f, 1.0f, 0.0f, 0.0f);
  glCallList(lists[LIST_CURVE]);

  glPopMatrix();
  glPushMatrix();

  /* Bmmels */
  glTranslatef(0.0f,
               1.06f - myCos(auslenkung) * (0.1f + mySin(55.0f) * 0.55f /
               myCos(55.0f)),
               0.4f + mySin(auslenkung) * (0.1f + mySin(55.0f) * 0.55f /
               myCos(55.0f)));
  glutSolidSphere(0.1f, 30, 30);

  glPopMatrix();
  glPushMatrix();

  glTranslatef(0.0f, .96f - mySin(55.0f) / myCos(55.0f) * 0.55f, 0.2f);
  glutSolidSphere(0.1f, 30, 30);
  glTranslatef(0.0f, 0.0f, -0.2f);
  glutSolidSphere(0.1f, 30, 30);
  glTranslatef(0.0f, 0.0f, -0.2f);
  glutSolidSphere(0.1f, 30, 30);
//  glTranslatef(0.0f, 0.0f, -0.2f);

  glPopMatrix();
  glPushMatrix();
  glTranslatef(0.0f,
               1.06f - myCos(360.0f - auslenkung2) *
               (0.1f + mySin(55.0f) / myCos(55.0f) * 0.55f),
               -0.4f + mySin(360.0f - auslenkung2) *
               (0.1f + mySin(55.0f) / myCos(55.0f) * 0.55f));
  glutSolidSphere(0.1f, 30, 30);

  glPopMatrix();
  glPushMatrix();

  /* Seile */
  glCallList(lists[LIST_ROPE]);
  glTranslatef(0.0f, 0.0f, -0.2f);
  glCallList(lists[LIST_ROPE]);
  glTranslatef(0.0f, 0.0f, -0.2f);
  //glCallList(lists[LIST_ROPE]);
  manager_ausgelenkter_faden(360.0f - auslenkung2, 1);
  glTranslatef(0.0f, 0.0f, 0.6f);
  glCallList(lists[LIST_ROPE]);
  glTranslatef(0.0f, 0.0f, 0.2f);
  //glCallList(lists[LIST_ROPE]);
  manager_ausgelenkter_faden(auslenkung, 1);

  glPopMatrix();

  glRotatef(180.0f, 0.0f, 1.0f, 0.0f);
  glCallList(lists[LIST_ROPE]);
  glTranslatef(0.0f, 0.0f, -0.2f);
  glCallList(lists[LIST_ROPE]);
  glTranslatef(0.0f, 0.0f, -0.2f);
  //glCallList(lists[LIST_ROPE]);
  manager_ausgelenkter_faden(auslenkung, -1);
  glTranslatef(0.0f, 0.0f, 0.6f);
  glCallList(lists[LIST_ROPE]);
  glTranslatef(0.0f, 0.0f, 0.2f);
  //glCallList(lists[LIST_ROPE]);
  manager_ausgelenkter_faden(360.0f - auslenkung2, -1);
}

void manager_reshape (int w, int h) {
  GLfloat aspectRatio;

  glViewport(0, 0, w, h);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  aspectRatio = ((w == 0.0f)?(1.0f):(w)) / ((h == 0.0f)?(1.0f):(h));
  if (w < h)
    glFrustum(-myWidth, myWidth, -myHeight / aspectRatio,
              myHeight / aspectRatio, 2.0f, 20.0f);
  else
    glFrustum(-myWidth * aspectRatio, myWidth * aspectRatio,
              -myHeight, myHeight, 2.0f, 20.0f);
}

void manager_init () {
  const GLfloat ambientLight[4] = { .2f, .2f, .2f, 1.0f },
    specularLight[4] = { 1.0f, 1.0f, 1.0f, 1.0f },
    specRef[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
  int i;

  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
  glClearDepth(1.0f);

  glEnable(GL_DEPTH_TEST);
  glEnable(GL_NORMALIZE);

  glEnable(GL_CULL_FACE);
  glCullFace(GL_BACK);
  glFrontFace(GL_CCW);

  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);

  glEnable(GL_MULTISAMPLE);
  glEnable(GL_POINT_SMOOTH);
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_LINE_SMOOTH);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
#if 0
  glEnable(GL_POLYGON_SMOOTH);
  glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
#endif
  glShadeModel(GL_SMOOTH);

  glEnable(GL_LIGHTING);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambientLight);

  glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuseLight);
  glLightfv(GL_LIGHT0, GL_SPECULAR, specularLight);
  glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, 60.0f);
  glEnable(GL_LIGHT0);

  glEnable(GL_COLOR_MATERIAL);
  glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);

  glMaterialfv(GL_FRONT, GL_SPECULAR, specRef);
  glMateriali(GL_FRONT, GL_SHININESS, 128);

  /* Displaylisten anlegen */
  lists[0] = glGenLists(LIST_MAX);
  for (i = 1; i < LIST_MAX; ++i) lists[i] = lists[i - 1] + 1;

  /* dnne Seile */
  glNewList(lists[LIST_ROPE], GL_COMPILE);
    glPushMatrix();
    glColor4f(1.0f, 1.0f, 1.0f, 0.55f);
    glTranslatef(0.275f, 1.06f - mySin(55.0f) / myCos(55.0f) * 0.275f,
                 0.0f);
    glRotatef(55.0f, 0.0f, 0.0f, 1.0f);
    myTube(0.55f / myCos(55.0f), 0.002f, 18);
    glPopMatrix();
  glEndList();

  /* Silberstangen */
  glNewList(lists[LIST_BAR], GL_COMPILE);
    glColor3f(.8f, .7f, .7f);
    myTube(1.0f, 0.02f, 36);
  glEndList();

  /* Kurvenstck */
  glNewList(lists[LIST_CURVE], GL_COMPILE);
    myMuffenStuck(0.04f, 0.02f, 36);
  glEndList();

  printf("%s\n%s\n%s\n",
   glGetString(GL_VENDOR), glGetString(GL_RENDERER),
   glGetString(GL_VERSION));
}
