# --
# Kernel/Modules/DTSVirtualHostAdmin.pm - frontend module to administer virtual hosts
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --

package Kernel::Modules::DTSVirtualHostAdmin;

use strict;
use Kernel::System::Valid;
use Kernel::System::DTSVirtualHost;
use Kernel::System::DTSTheme;

our $VERSION = "1.1";

sub new ()
{
    my $Type = shift;
    my %Param = @_;
    my $Self = {};

    bless ($Self, $Type);

    # adopt all provided objects
    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed stuff
    foreach (qw(LayoutObject ParamObject)) {
        if (!$Self->{$_}) {
            die "Got no $_!";
        }
    }

    # create additional objects
    $Self->{ValidObject} = Kernel::System::Valid->new(%Param);
    $Self->{ThemeObject} = Kernel::System::DTSTheme->new(%Param);
    $Self->{VirtualHostObject} = Kernel::System::DTSVirtualHost->new(%Param);

    return $Self;
}

sub Run ()
{
    my $Self = shift;
    my %Param = @_;

    my $LayoutObject = $Self->{LayoutObject};
    my $ParamObject = $Self->{ParamObject};
    my $VirtualHostObject = $Self->{VirtualHostObject};
    my %VirtualHostList = $VirtualHostObject->VirtualHostList()
        or $LayoutObject->FatalError();
    my %ThemeList = $Self->{ThemeObject}->ThemeList(Valid => 1)
        or $LayoutObject->FatalError();

    # ValidList contains all valid AND invalid IDs
    my %ValidList = $Self->{ValidObject}->ValidList()
        or $LayoutObject->FatalError();
    # ValidIDs contains only valid IDs
    my %ValidIDs = map { $_ => $_ } $Self->{ValidObject}->ValidIDsGet();

    # start HTML output
    my $Output = $LayoutObject->Header(Title => "DTS Virtual Hosts");
    $Output .= $LayoutObject->NavigationBar();

    # fetch submitted form data
    my %Data = ();
    foreach (qw(Subaction VirtualHostID VirtualHostName VirtualHostHost
        VirtualHostHttps VirtualHostIp AgentUrl CustomerUrl PublicUrl SoapUrl
        ThemeID ValidID)) {
        $Data{$_} = $ParamObject->GetParam(Param => $_) || "";
    }
    $Data{VirtualHostHttps} = ($Data{VirtualHostHttps} ? 1 : 0);

    if ($Data{Subaction} eq "Change") {
        # user selected an existing virtual host, wants to modify it, next
        # action will be "Update", unless selected virtual host doesn't exist
        if (exists $VirtualHostList{$Data{VirtualHostID}}) {
            $Data{Subaction} = "Update";
            foreach (qw(VirtualHostName VirtualHostHost VirtualHostHttps
                VirtualHostIp AgentUrl CustomerUrl PublicUrl SoapUrl ThemeID
                ValidID)) {
                $Data{$_} = $VirtualHostList{$Data{VirtualHostID}}->{$_};
            }
        }
        else {
            $Data{Subaction} = "Add";
        }
    }
    elsif ($Data{Subaction} eq "Update") {
        # user sends updated virtual host, display filled mask and hooray message
        if ($VirtualHostObject->VirtualHostModify(%Data)) {
            $Output .= $LayoutObject->Notify(
                Info => "Virtual host updated"
            );
        }
        else {
            $Output .= $LayoutObject->Notify(
                Priority => "Error"
            );
        }
    }
    elsif ($Data{Subaction} eq "New") {
        # user sends new virtual host data, display empty mask and hooray message
        if ($VirtualHostObject->VirtualHostAdd(%Data)) {
            $Output .= $LayoutObject->Notify(
                Info => "Virtual host added"
            );
            %Data = (
                Subaction => "Add"
            );
        }
        else {
            $Output .= $LayoutObject->Notify(
                Priority => "Error"
            );
        }
    }
    else {
        # show empty mask per default, with next action "New"
        $Data{SubactionDisplay} = "Add";
        $Data{Subaction} = "New";
    }

    # title of Add/Update button
    if (!defined $Data{SubactionDisplay}) {
        $Data{SubactionDisplay} = $Data{Subaction};
    }

    # fill select list with virtual hosts
    %VirtualHostList = $VirtualHostObject->VirtualHostList()
        or $LayoutObject->FatalError();
    foreach (@{&SortHashBySubHashKey(\%VirtualHostList, "VirtualHostName")}) {
        my $BlockName = "VirtualHostEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= (exists $ValidIDs{$VirtualHostList{$_}->{ValidID}} ? "Valid" : "Invalid");
        $BlockName .= ($_ == $Data{VirtualHostID} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                VirtualHostID => $_,
                VirtualHostName => $VirtualHostList{$_}->{VirtualHostName}
            }
        );
    }

    # HTTPS checkbox
    my $BlockName = "Https";
    $LayoutObject->Block(Name => $BlockName);
    $BlockName .= ($Data{"VirtualHostHttps"} ? "Selected" : "Deselected");
    $LayoutObject->Block(Name => $BlockName, Data => {});

    # fill select list with themes
    foreach (@{&SortHashBySubHashKey(\%ThemeList, "ThemeName")}) {
        $BlockName = "ThemeEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= ($_ == $Data{ThemeID} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                ThemeID => $_,
                ThemeName => $ThemeList{$_}->{ThemeName}
            }
        );
    }

    # fill select list with valid and invalid ids
    foreach (sort keys %ValidList) {
        $BlockName = "ValidEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= ($_ == $Data{ValidID} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                ValidID => $_,
                ValidName => $ValidList{$_}
            }
        );
    }

    # generate output
    $Output .= $LayoutObject->Output(
        TemplateFile => "DTSVirtualHost",
        Data => \%Data
    );
    $Output .= $LayoutObject->Footer();

    return $Output;
}

sub SortHashBySubHashKey ()
{
    my ($HashRef, $SubHashKey) = @_;

    my @Result = sort {
        $HashRef->{$a}->{$SubHashKey} cmp $HashRef->{$b}->{$SubHashKey}
    } keys %$HashRef;

    return \@Result;
}

1;
