# --
# Kernel/System/DTSVirtualHost.pm - collection of DTSVirtualHost specific functions
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --

package Kernel::System::DTSVirtualHost;

use strict;
use Kernel::System::Valid;
use Kernel::System::User;
use Kernel::System::DTSTheme;
use Kernel::System::DTSLib;

our $VERSION = "1.1";

sub new ()
{
    my $Type = shift;
    my %Param = @_;
    my $Self = {};

    bless ($Self, $Type);

    # adopt all provided objects
    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed stuff
    foreach (qw(DBObject LogObject ConfigObject)) {
        if (!$Self->{$_}) {
            die "Got no $_!";
        }
    }

    # create needed objects
    $Self->{ValidObject} = Kernel::System::Valid->new(%Param);
    $Self->{ThemeObject} = Kernel::System::DTSTheme->new(%Param);
    $Self->{DTSLibObject} = Kernel::System::DTSLib->new(%Param);

    return $Self;
}

# return all virtual hosts as a hash with references to hashes
sub VirtualHostList ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    my $SQL = "SELECT id, name, https, ip, agent_url, customer_url, ".
        "public_url, soap_url, theme, ".
        "valid_id, create_time, create_by, change_time, change_by ".
        "FROM dts_virtual_host";
    my @Clauses = ();
    if ($Param{Valid}) {
        push @Clauses, "valid_id IN (".
            join(",", $Self->{ValidObject}->ValidIDsGet()).")";
    }
    if ($Param{VirtualHostID}) {
        push @Clauses, "id=".$DBObject->Quote($Param{VirtualHostID}, "Integer");
    }
    if (@Clauses) {
        $SQL .= " WHERE ".join(" AND ", @Clauses);
    }
    if (!$DBObject->Prepare(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }
    my %Data = ();
    while (my @Row = $DBObject->FetchrowArray()) {
        my $Id = $Row[0];
        $Data{$Id}->{VirtualHostName} = $Row[1];
        $Data{$Id}->{VirtualHostHttps} = $Row[2];
        $Data{$Id}->{VirtualHostIp} = $Row[3];
        $Data{$Id}->{AgentUrl} = $Row[4];
        $Data{$Id}->{CustomerUrl} = $Row[5];
        $Data{$Id}->{PublicUrl} = $Row[6];
        $Data{$Id}->{SoapUrl} = $Row[7];
        $Data{$Id}->{ThemeID} = $Row[8];
        $Data{$Id}->{ValidID} = $Row[9];
        $Data{$Id}->{CreateTime} = $Row[10];
        $Data{$Id}->{CreateBy} = $Row[11];
        $Data{$Id}->{ChangeTime} = $Row[12];
        $Data{$Id}->{ChangeBy} = $Row[13];
    }
    return %Data;
}

# create a new virtual host
sub VirtualHostAdd ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    if (!$Self->VirtualHostNameCheck(%Param)) {
        return;
    }

    my $SQL = "INSERT INTO dts_virtual_host (name, https, ip, agent_url, ".
       "customer_url, public_url, soap_url, theme, valid_id, ".
       "create_time, create_by, change_time, change_by) VALUES ('".
       $DBObject->Quote($Param{VirtualHostName})."', '".
       $DBObject->Quote($Param{VirtualHostHttps})."', '".
       $DBObject->Quote($Param{VirtualHostIp})."', '".
       $DBObject->Quote($Param{AgentUrl})."', '".
       $DBObject->Quote($Param{CustomerUrl})."', '".
       $DBObject->Quote($Param{PublicUrl})."', '".
       $DBObject->Quote($Param{SoapUrl})."', ".
       $DBObject->Quote($Param{ThemeID}, "Integer").", ".
       $DBObject->Quote($Param{ValidID}, "Integer").", current_timestamp, ".
       $DBObject->Quote($Self->{UserID}, "Integer").", current_timestamp, ".
       $DBObject->Quote($Self->{UserID}, "Integer").")";
    if (!$DBObject->Do(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    if (!$Self->UpdateConfig()) {
        return;
    }

    return 1;
}

# update a virtual host
sub VirtualHostModify ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    if (!$Self->VirtualHostNameCheck(%Param)) {
        return;
    }

    my $SQL = "UPDATE dts_virtual_host SET name='".
       $DBObject->Quote($Param{VirtualHostName})."', https='".
       $DBObject->Quote($Param{VirtualHostHttps})."', ip='".
       $DBObject->Quote($Param{VirtualHostIp})."', agent_url='".
       $DBObject->Quote($Param{AgentUrl})."', customer_url='".
       $DBObject->Quote($Param{CustomerUrl})."', public_url='".
       $DBObject->Quote($Param{PublicUrl})."', soap_url='".
       $DBObject->Quote($Param{SoapUrl})."', theme=".
       $DBObject->Quote($Param{ThemeID}, "Integer").", valid_id=".
       $DBObject->Quote($Param{ValidID}, "Integer").", change_by=".
       $DBObject->Quote($Self->{UserID}, "Integer").", change_time=".
       "current_timestamp WHERE id=".
       $DBObject->Quote($Param{VirtualHostID}, "Integer");
    if (!$DBObject->Do(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    if (!$Self->UpdateConfig()) {
        return;
    }

    return 1;
}

sub UpdateConfig ()
{
    my $Self = shift;
    my %Param = @_;

    # start host-to-theme mapping config
    my $Host2Theme = "";

    # get all themes
    my %ThemeList = $Self->{ThemeObject}->ThemeList();

    # go through all active virtual hosts
    my %VirtualHostList = $Self->VirtualHostList(
        Valid => 1
    );
    foreach my $VirtualHostID(keys %VirtualHostList) {
        my $VirtualHost = $VirtualHostList{$VirtualHostID};
        my $VirtualHostName = $VirtualHost->{VirtualHostName};
        my $ThemeName = $ThemeList{$VirtualHost->{ThemeID}}->{ThemeName};

        # append this virtual host to host-to-theme mapping
        $Host2Theme .= "'".$VirtualHostName."' => '".$ThemeName."',\n";
    }

    # write host-to-theme mapping
    my $File = $Self->{ConfigObject}->Get("DTSVirtualHost::Host2ThemeFile") ||
        $Self->{ConfigObject}->Get("Home")."/var/httpd/host2theme.txt";
    if (!$Self->{DTSLibObject}->WriteVersionedFile(
        FileName => $File,
        Data => \$Host2Theme
    )) {
        return;
    }

    return 1;
}

# check name of host
sub VirtualHostNameCheck ()
{
    my $Self = shift;
    my %Param = @_;

    if ($Param{VirtualHostName} !~ /^([A-Za-z0-9\-\.]+)+\.[A-Za-z0-9\-]{2,}$/) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => "Name of virtual host contains illegal characters"
        );
        return;
    }

    return 1;
}

1;
