# --
# Ticket/Number/DTSTicketNumber.pm - a compact ticket number generator
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --

package Kernel::System::Ticket::Number::DTSTicketNumber;

use strict;

our $VERSION = "1.7";

sub TicketCreateNumber ()
{
    my $Self = shift;

    # check needed stuff
    my $ConfigObject = $Self->{ConfigObject} || die "Got no ConfigObject!";
    my $TimeObject = $Self->{TimeObject} || die "Got no TimeObject!";
    my $LogObject = $Self->{LogObject} || die "Got no LogObject!";

    my $SystemID = $ConfigObject->Get("SystemID") || die "Got no SystemID!";

    my $CounterLog = $ConfigObject->Get("Ticket::CounterLog");
    if (!defined $CounterLog) {
        my $Message = "Got no Ticket::CounterLog!";
        $LogObject->Log(
            Priority => "error",
            Message => $Message
        );
        die $Message;
    }

    # minimum length of daily counter
    my $MinCounterSize =
        $ConfigObject->Get("Ticket::NumberGenerator::MinCounterSize");
    if ($MinCounterSize < 1) {
        $MinCounterSize = 2;
    }

    # get todays date (eg. 07-09-23)
    my $SystemTime = $TimeObject->SystemTime();
    my ($Sec, $Min, $Hour, $Day, $Month, $Year) =
        $TimeObject->SystemTime2Date(SystemTime => $SystemTime);
    $Year = substr($Year, length($Year) - 2);
    my $Today = join("-", ($Year, $Month, $Day));

    # open counterlog, create it safely if not there
    if (!open(COUNTER, "+<$CounterLog")) {
        if (!open(COUNTER, ">>$CounterLog")) {
            my $Message = "Cannot create Ticket::CounterLog: ".
                $CounterLog.": ".$!;
            $LogObject->Log(
                Priority => "error",
                Message => $Message
            );
            die $Message;
        }
    }
    close(COUNTER);

    # open counterlog for reading and writing
    if (!open(COUNTER, "+<$CounterLog")) {
        my $Message = "Cannot open Ticket::CounterLog: ".
            $CounterLog.": ".$!;
        $LogObject->Log(
            Priority => "error",
            Message => $Message
        );
        die $Message;
    }

    # lock counterlog exclusivly
    if (!flock(COUNTER, 2)) {
        my $Message = "Cannot lock Ticket::CounterLog: ".
            $CounterLog.": ".$!;
        $LogObject->Log(
            Priority => "error",
            Message => $Message
        );
        die $Message;
    }

    # read one line, strip off newline characters
    my $Line = <COUNTER>;
    $Line =~ s/\r|\n//go;

    # Line looks like 3;07-08-31
    my ($Count, $LastModify) = split(/;/, $Line);

    # check for midnight wraparound
    if ($LastModify eq $Today) {
        ++$Count;
    }
    else {
        $Count = 0;
    }

    # we read the counterlog, so set back filedescriptor and flush file
    if (!truncate(COUNTER, 0)) {
        my $Message = "Cannot truncate Ticket::CounterLog: ".
            $CounterLog.": ".$!;
        $LogObject->Log(
            Priority => "error",
            Message => $Message
        );
        die $Message;
    }

    # set filedescriptor to beginning of counterlog
    if (!seek(COUNTER, 0, 0)) {
        my $Message = "Cannot seek Ticket::CounterLog: ".
            $CounterLog.": ".$!;
        $LogObject->Log(
            Priority => "error",
            Message => $Message
        );
        die $Message;
    }

    # write new counter and close counterlog (lock gets cleared, too)
    print COUNTER "$Count;$Today\n";
    close(COUNTER);

    # prepend "0" as needed
    my $CounterSize = length($Count);
    if ($CounterSize < $MinCounterSize) {
        $Count = "0" x ($MinCounterSize - $CounterSize) . $Count;
    }

    # construct the whole counter value, eg. 3070923042
    return $SystemID . $Year . $Month . $Day . $Count;
}

sub GetTNByString ()
{
    my $Self = shift;
    my $String = shift || die "Got no TicketNumber String!";

    # check needed stuff
    my $ConfigObject = $Self->{ConfigObject} || die "Got no ConfigObject!";
    my $SystemID = $ConfigObject->Get("SystemID") || die "Got no SystemID!";

    my $MinCounterSize =
        $ConfigObject->Get("Ticket::NumberGenerator::MinCounterSize") ||
        2;

    my $CheckSystemID =
        $ConfigObject->Get("Ticket::NumberGenerator::CheckSystemID");

    my $TicketHook = $ConfigObject->Get("Ticket::Hook");
    my $TicketHookDivider = $ConfigObject->Get("Ticket::HookDivider");

    my $LookupTN = "";
    if ($CheckSystemID) {
        # TN has to match $SystemID
        $LookupTN = $SystemID;
    }
    # TN has to match date part, eg. "070923042"
    $LookupTN .= '\d{6}(\d{' . $MinCounterSize . ',})';

    # Ticket#070923042
    if ($String =~ /\Q$TicketHook$TicketHookDivider\E$LookupTN/i) {
        return $1;
    }

    # Ticket: 070923042
    if ($String =~ /\Q$TicketHook\E:+.{0,2}$LookupTN/i) {
        return $1;
    }

    # no valid TN detected
    return;
}

1;
