# --
# Kernel/Modules/DTSThemeAdmin.pm - frontend module to administer themes
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --

package Kernel::Modules::DTSThemeAdmin;

use strict;
use Kernel::System::Valid;
use Kernel::System::DTSTheme;

our $VERSION = "1.0";

sub new ()
{
    my $Type = shift;
    my %Param = @_;
    my $Self = {};

    bless ($Self, $Type);

    # adopt all provided objects
    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed stuff
    foreach (qw(DBObject LayoutObject ParamObject ConfigObject TimeObject)) {
        if (!$Self->{$_}) {
            die "Got no $_!";
        }
    }

    # create additional objects
    $Self->{ValidObject} = Kernel::System::Valid->new(%Param);
    $Self->{ThemeObject} = Kernel::System::DTSTheme->new(%Param);

    return $Self;
}

sub Run ()
{
    my $Self = shift;
    my %Param = @_;

    my $DBObject = $Self->{DBObject};
    my $LayoutObject = $Self->{LayoutObject};
    my $ParamObject = $Self->{ParamObject};
    my $ThemeObject = $Self->{ThemeObject};
    my %ThemeList = $ThemeObject->ThemeList() or $LayoutObject->FatalError();
    # ValidList contains all valid AND invalid IDs
    my %ValidList = $Self->{ValidObject}->ValidList()
        or $LayoutObject->FatalError();
    # ValidIDs contains only valid IDs
    my %ValidIDs = map { $_ => $_ } $Self->{ValidObject}->ValidIDsGet();

    # fetch submitted form data
    my %Data = ();
    my %File = $ParamObject->GetUploadAll(
        Param => "Favicon",
        Encoding => "Raw",
        Source => "string"
    );
    if (%File) {
        $Data{Favicon} = $File{Content};
    }
    foreach (qw(Action Subaction ThemeID ThemeName ValidID TemplateName TemplateContent)) {
        $Data{$_} = $ParamObject->GetParam(Param => $_) || "";
    }

    # start HTML output
    my $Output = $LayoutObject->Header(Title => "DTS Theme");
    my $TemplateFile = "";
    if (($Data{Subaction} eq "TemplateChange") ||
        ($Data{Subaction} eq "TemplateUpdate")) {
        $TemplateFile = "DTSTemplateEditor";
    }
    else {
        $Output .= $LayoutObject->NavigationBar();
        $TemplateFile = "DTSTheme";
    }

    # load templates for theme
    my %Templates = ();
    if (exists $ThemeList{$Data{ThemeID}}) {
        $Data{ThemeName} = $ThemeList{$Data{ThemeID}}->{ThemeName};
        %Templates = $ThemeObject->ListTemplates(
            ThemeName => $Data{ThemeName}
        );
        if (!%Templates) {
            $Output .= $LayoutObject->Notify(
                Priority => "Error"
            );
        }
    }

    if ($Data{Subaction} eq "TemplateChange") {
        # user wants to modify a template, show it in the editor
        if (exists $Templates{$Data{TemplateName}}) {
            $Data{TemplateContent} =
                $Templates{$Data{TemplateName}}->{TemplateContent};
        }
    }
    elsif ($Data{Subaction} eq "TemplateUpdate") {
        # user sends updated template, save it
        if (exists $Templates{$Data{TemplateName}}) {
            if ($ThemeObject->WriteTemplate(%Data)) {
                $Output .= $LayoutObject->Notify(
                    Info => "Template updated"
                );
                $Templates{$Data{TemplateName}}->{Content} = $Data{TemplateContent};
            }
            else {
                $Output .= $LayoutObject->Notify(
                    Priority => "Error"
                );
            }
        }
    }
    elsif ($Data{Subaction} eq "Change") {
        # user selected an existing theme, wants to modify it
        # next action will be "Update", unless selected theme doesn't exist
        if (exists $ThemeList{$Data{ThemeID}}) {
            $Data{Subaction} = "Update";
            foreach (qw(ThemeName ValidID)) {
                $Data{$_} = $ThemeList{$Data{ThemeID}}->{$_};
            }
            $LayoutObject->Block(
                Name => "Template",
                Data => {
                    ThemeID => $Data{ThemeID}
                }
            );
        }
        else {
            $Data{Subaction} = "Add";
        }
    }
    elsif ($Data{Subaction} eq "Update") {
        # user sends updated theme, display empty mask and hooray message
        if ($ThemeObject->ThemeModify(%Data)) {
            $Output .= $LayoutObject->Notify(Info => "Theme updated");
        }
        else {
            $Output .= $LayoutObject->Notify(Priority => "Error");
        }
    }
    elsif ($Data{Subaction} eq "New") {
        # user sends new theme data, display empty mask and hooray message
        if ($ThemeObject->ThemeAdd(%Data)) {
            $Output .= $LayoutObject->Notify(Info => "Theme added");
        }
        else {
            $Output .= $LayoutObject->Notify(Priority => "Error");
        }
    }
    else {
        # show empty mask per default, with next action "New"
        $Data{SubactionDisplay} = "Add";
        $Data{Subaction} = "New";
    }

    # title of Add/Update button
    if (!defined $Data{SubactionDisplay}) {
        $Data{SubactionDisplay} = $Data{Subaction};
    }

    # fill select list with themes
    %ThemeList = $ThemeObject->ThemeList() or $LayoutObject->FatalError();
    foreach (@{&SortHashBySubHashKey(\%ThemeList, "ThemeName")}) {
        my $BlockName = "ThemeEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= (exists $ValidIDs{$ThemeList{$_}->{ValidID}} ? "Valid" : "Invalid");
        $BlockName .= ($_ == $Data{ThemeID} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                ThemeID => $_ ,
                ThemeName => $ThemeList{$_}->{ThemeName}
            }
        );
    }

    if (-e $ThemeObject->GetFaviconDirectory(ThemeName => $Data{ThemeName})."/favicon.ico") {
        $LayoutObject->Block(
            Name => "Favicon",
            Data => {
                ThemeName => $Data{ThemeName}
            }
        );
    }

    foreach (@{&SortHashBySubHashKey(\%Templates, "IsOriginal")}) {
        my $BlockName = "TemplateEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= ($Templates{$_}->{IsOriginal} ? "NotModified" : "Modified");
        $BlockName .= ($_ eq $Data{TemplateName} ? "Selected" : "Deselected");
        $LayoutObject->Block(
            Name => $BlockName,
            Data => {
                TemplateName => $_
            }
        );
    }
    # fill select list with valid and invalid ids
    foreach (sort keys %ValidList) {
        my $BlockName = "ValidEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= ($_ == $Data{ValidID} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                ValidID => $_,
                ValidName => $ValidList{$_}
            }
        );
    }

    # generate output
    $Output .= $LayoutObject->Output(
        TemplateFile => $TemplateFile,
        Data => \%Data
    );
    $Output .= $LayoutObject->Footer();

    return $Output;
}

sub SortHashBySubHashKey ()
{
    my ($HashRef, $SubHashKey) = @_;

    my @Result = sort {
        $HashRef->{$a}->{$SubHashKey} cmp $HashRef->{$b}->{$SubHashKey} or
        $a cmp $b
    } keys %$HashRef;

    return \@Result;
}

1;
