# --
# Kernel/System/DTSTheme.pm - functions for managing themes
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --

package Kernel::System::DTSTheme;

use strict;
use Kernel::System::Valid;
use Kernel::System::DTSLib;

our $VERSION = "1.1";

sub new ()
{
    my $Type = shift;
    my %Param = @_;
    my $Self = {};

    bless($Self, $Type);

    # adopt all provided objects
    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed objects
    foreach (qw(MainObject ConfigObject DBObject LogObject TimeObject)) {
        if (!$Self->{$_}) {
            die "Got no $_!";
        }
    }

    # create needed objects
    $Self->{ValidObject} = Kernel::System::Valid->new(%Param);
    $Self->{DTSLibObject} = Kernel::System::DTSLib->new(%Param);
    $Self->{ImagesDirectory} = $Self->{ConfigObject}->Get("Home").
        "/var/httpd/htdocs/images/";
    $Self->{TemplatesDirectory} = $Self->{ConfigObject}->Get("TemplateDir").
        "/HTML/";

    return $Self;
}

# save a template
sub WriteTemplate ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(ThemeName TemplateName TemplateContent)) {
        if (!defined $Param{$_}) {
            $Self->{LogObject}->Log(
                Priority => "Error",
                Message => "Got no $_!"
            );
            return;
        }
    }

    if (!$Self->ThemeNameCheck(%Param)) {
        return;
    }

    my $TemplateDir = $Self->{TemplatesDirectory}.$Param{ThemeName};
    if (!$Self->{DTSLibObject}->MakeDirectories(Directories => [
        $TemplateDir
    ])) {
        return;
    }

    if (!$Self->{DTSLibObject}->WriteVersionedFile(
        FileName => $TemplateDir."/".$Param{TemplateName},
        Data => \$Param{TemplateContent}
    )) {
        return;
    }

    return 1;
}

# list all templates for a given theme
sub ListTemplates ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(ThemeName)) {
        if (!$Param{$_}) {
            $Self->{LogObject}->Log(
                Priority => "Error",
                Message => "Got no $_!"
            );
            return;
        }
    }

    my $StandardPath = $Self->{TemplatesDirectory}."Standard";
    my $TemplatePath = $Self->{TemplatesDirectory}.$Param{ThemeName};

    if (!opendir(DH, $StandardPath)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => "Can not open directory $StandardPath: $!"
        );
        return;
    }

    my %Templates = ();
    foreach (readdir(DH)) {
        if (/\.dtl$/o && !/^AAA/o) {
            my $StandardTemplateRef = $Self->{MainObject}->FileRead(
                Directory => $StandardPath,
                Filename => $_
            );
            if (!defined $StandardTemplateRef) {
                return;
            }
            # read w/o barfing to log
            my $TemplateRef = $Self->{MainObject}->FileRead(
                Directory => $TemplatePath,
                Filename => $_,
                DisableWarnings => 1
            );
            if (defined $TemplateRef) {
                $Templates{$_}->{TemplateContent} = $$TemplateRef;
                $Templates{$_}->{IsOriginal} = (
                    $$TemplateRef eq $$StandardTemplateRef
                );
            }
            else {
                $Templates{$_}->{TemplateContent} = $$StandardTemplateRef;
                $Templates{$_}->{IsOriginal} = 1;
            }
        }
    };
    closedir(DH);

    return %Templates;
}

sub GetFaviconDirectory ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(ThemeName)) {
        if (!defined $Param{$_}) {
            $Self->{LogObject}->Log(
                Priority => "Error",
                Message => "Got no $_!"
            );
            return;
        }
    }

    return $Self->{ImagesDirectory}.$Param{ThemeName};
}

# write favicon
sub WriteFavicon ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(ThemeName Favicon)) {
        if (!defined $Param{$_}) {
            $Self->{LogObject}->Log(
                Priority => "Error",
                Message => "Got no $_!"
            );
            return;
        }
    }

    my $FaviconDir = $Self->GetFaviconDirectory();
    if (!$Self->{DTSLibObject}->MakeDirectories(Directories => [
        $FaviconDir
    ])) {
        return;
    }

    if (!$Self->{DTSLibObject}->WriteVersionedFile(
        FileName => $FaviconDir."/favicon.ico",
        Data => \$Param{Favicon}
    )) {
        return;
    }

    return 1;
}

# return all themes as a hash with references to hashes
sub ThemeList ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    my $SQL = "SELECT id, theme, ".
        "valid_id, create_time, create_by, change_time, change_by ".
        "FROM theme";
    my @Clauses = ();
    if ($Param{Valid}) {
        push @Clauses, "valid_id IN (".
            join(",", $Self->{ValidObject}->ValidIDsGet()).")";
    }
    if ($Param{ThemeID}) {
        push @Clauses, "id=".$DBObject->Quote($Param{ThemeID}, "Integer");
    }
    if (@Clauses) {
        $SQL .= " WHERE ".join(" AND ", @Clauses);
    }
    if (!$DBObject->Prepare(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }
    my %Data = ();
    while (my @Row = $DBObject->FetchrowArray()) {
        my $Id = $Row[0];
        my $ThemeName = $Row[1];
        $Data{$Id}->{ThemeName} = $ThemeName;
        $Data{$Id}->{ValidID} = $Row[2];
        $Data{$Id}->{CreateTime} = $Row[3];
        $Data{$Id}->{CreateBy} = $Row[4];
        $Data{$Id}->{ChangeTime} = $Row[5];
        $Data{$Id}->{ChangeBy} = $Row[6];
    }

    return %Data;
}

# check name of theme
sub ThemeNameCheck ()
{
    my $Self = shift;
    my %Param = @_;
    my $ThemeName = $Param{ThemeName};

    if (($ThemeName eq "Standard") || ($ThemeName eq "Lite")) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => "Can not modify default themes"
        );
        return;
    }

    if ($ThemeName !~ /^[A-Za-z0-9\-\_]+$/) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => "Name contains illegal characters"
        );
        return;
    }

    return 1;
}

# create a new theme
sub ThemeAdd ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    if (!$Self->ThemeNameCheck(%Param)) {
        return;
    }

    my $SQL = "INSERT INTO theme (theme, valid_id, ".
        "create_time, create_by, change_time, change_by) VALUES ('".
        $DBObject->Quote($Param{ThemeName})."', ".
        $DBObject->Quote($Param{ValidID}, "Integer").", current_timestamp, ".
        $DBObject->Quote($Self->{UserID}, "Integer").", current_timestamp, ".
        $DBObject->Quote($Self->{UserID}, "Integer").")";
    if (!$DBObject->Do(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    if (defined $Param{Favicon}) {
        if (!$Self->WriteFavicon(%Param)) {
            return;
        }
    }

    if (!$Self->CopyDirectory(
        SourceDirectory => $Self->{ImagesDirectory}."Standard",
        DestinationDirectory => $Self->{ImagesDirectory}.$Param{ThemeName}
    )) {
        return;
    }

    return 1;
}

sub CopyDirectory ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(SourceDirectory DestinationDirectory)) {
        if (!$Param{$_}) {
            $Self->{LogObject}->Log(
                Priority => "Error",
                Message => "Got no $_!"
            );
            return;
        }
    }

    if (!$Self->{DTSLibObject}->MakeDirectories(Directories => [
        $Param{DestinationDirectory}
    ])) {
        return;
    }

    if (!opendir(DH, $Param{SourceDirectory})) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => "Can not open directory $Param{SourceDirectory}: $!"
        );
        return;
    }

    my @Files = readdir(DH);
    closedir(DH);

    foreach my $Filename(@Files) {
        if (-f $Param{SourceDirectory}."/".$Filename &&
            !-e $Param{DestinationDirectory}."/".$Filename) {
            my $ContentRef = $Self->{MainObject}->FileRead(
                Directory => $Param{SourceDirectory},
                Filename => $Filename
            );
            if (!defined $ContentRef) {
                return;
            }
            if (!$Self->{MainObject}->FileWrite(
                Directory => $Param{DestinationDirectory},
                Filename => $Filename,
                Content => $ContentRef
            )) {
                return;
            }
        }
    }

    return 1;
}

# update a theme
sub ThemeModify ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    if (!$Self->ThemeNameCheck(%Param)) {
        return;
    }

    my $SQL = "UPDATE theme SET theme='".
        $DBObject->Quote($Param{ThemeName})."', valid_id=".
        $DBObject->Quote($Param{ValidID}, "Integer").", change_by=".
        $DBObject->Quote($Self->{UserID}, "Integer").", change_time=".
        "current_timestamp WHERE id=".
        $DBObject->Quote($Param{ThemeID}, "Integer");
    if (!$DBObject->Do(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    if (defined $Param{Favicon}) {
        if (!$Self->WriteFavicon(%Param)) {
            return;
        }
    }

    return 1;
}

1;
