# --
# Kernel/Modules/DTSSoapUserAdmin.pm - frontend module to administer soap users
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --

package Kernel::Modules::DTSSoapUserAdmin;

use strict;
use Kernel::System::Valid;
use Kernel::System::DTSSoapUser;
use Kernel::System::DTSWsdl;

our $VERSION = "1.0";

sub new ()
{
    my $Type = shift;
    my %Param = @_;
    my $Self = {};

    bless ($Self, $Type);

    # adopt all provided objects
    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed stuff
    foreach (qw(LayoutObject ParamObject)) {
        if (!$Self->{$_}) {
            die "Got no $_!";
        }
    }

    # create additional objects
    $Self->{ValidObject} = Kernel::System::Valid->new(%Param);
    $Self->{SoapUserObject} = Kernel::System::DTSSoapUser->new(%Param);
    $Self->{WsdlObject} = Kernel::System::DTSWsdl->new(%Param);

    return $Self;
}

sub Run ()
{
    my $Self = shift;
    my %Param = @_;

    my $LayoutObject = $Self->{LayoutObject};
    my $ParamObject = $Self->{ParamObject};
    my $SoapUserObject = $Self->{SoapUserObject};
    my @SoapMethods = $Self->{WsdlObject}->MethodList();
    my %SoapUsers = $SoapUserObject->SoapUserList();

    # ValidList contains all valid AND invalid IDs
    my %ValidList = $Self->{ValidObject}->ValidList()
        or $LayoutObject->FatalError();
    # ValidIDs contains only valid IDs
    my %ValidIDs = map { $_ => $_ } $Self->{ValidObject}->ValidIDsGet();

    # start HTML output
    my $Output = $LayoutObject->Header(Title => "DTS SOAP Users");
    $Output .= $LayoutObject->NavigationBar();

    # fetch submitted form data
    my %Data = ();
    foreach (qw(Subaction Username OldUsername Password Address ValidID)) {
        $Data{$_} = $ParamObject->GetParam(Param => $_) || "";
    }
    foreach (sort @SoapMethods) {
        if ($ParamObject->GetArray(Param => "Method".$_)) {
            push @{$Data{Method}}, $_;
        }
    }
    my %Address = map {
        my ($IP, $Netmask) = split(/\//, $_);
        $IP => $Netmask;
    } split(/\s+/, $Data{Address});
    $Data{Address} = \%Address;

    if ($Data{Subaction} eq "Change") {
        # user selected an existing soap user, wants to modify it, next
        # action will be "Update", unless selected soap user doesn't exist
        if (exists $SoapUsers{$Data{Username}}) {
            $Data{Subaction} = "Update";
            $Data{OldUsername} = $Data{Username};
            foreach (qw(Password Address Method ValidID)) {
                $Data{$_} = $SoapUsers{$Data{Username}}->{$_};
            }
        }
        else {
            $Data{Subaction} = "Add";
        }
    }
    elsif ($Data{Subaction} eq "Update") {
        # user sends updated soap user, display filled mask and hooray message
        if ($SoapUserObject->SoapUserModify(%Data)) {
            $Output .= $LayoutObject->Notify(
                Info => "Soap user updated"
            );
        }
        else {
            $Output .= $LayoutObject->Notify(
                Priority => "Error"
            );
        }
    }
    elsif ($Data{Subaction} eq "New") {
        # user sends new soap user data, display empty mask and hooray message
        if ($SoapUserObject->SoapUserAdd(%Data)) {
            $Output .= $LayoutObject->Notify(
                Info => "Soap user added"
            );
            %Data = (
                Subaction => "Add"
            );
        }
        else {
            $Output .= $LayoutObject->Notify(
                Priority => "Error"
            );
        }
    }
    else {
        # show empty mask per default, with next action "New"
        $Data{SubactionDisplay} = "Add";
        $Data{Subaction} = "New";
    }

    # title of Add/Update button
    if (!defined $Data{SubactionDisplay}) {
        $Data{SubactionDisplay} = $Data{Subaction};
    }

    # fill select list with soap users
    %SoapUsers = $SoapUserObject->SoapUserList();

    foreach (sort keys %SoapUsers) {
        my $BlockName = "UsernameEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= (exists $ValidIDs{$SoapUsers{$_}->{ValidID}} ? "Valid" : "Invalid");
        $BlockName .= ($_ eq $Data{Username} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                Username => $_,
            }
        );
    }

    # create method list with checkboxes
    my %Methods = map { $_ => 1 } @{$Data{Method}};
    foreach (sort @SoapMethods) {
        my $BlockName = "MethodEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= (exists $Methods{$_} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                Method => $_,
            }
        );
    }

    # fill select list with valid and invalid ids
    foreach (sort keys %ValidList) {
        my $BlockName = "ValidEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= ($_ == $Data{ValidID} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                ValidID => $_,
                ValidName => $ValidList{$_}
            }
        );
    }

    # flatten addresses
    my $Addresses = "";
    $Data{AddressRows} = 0;
    foreach (keys %{$Data{Address}}) {
        $Addresses .= $_;
        my $Netmask = $Data{Address}->{$_};
        if (defined $Netmask) {
            $Addresses .= "/".$Netmask;
        }
        $Addresses .= "\n";
        $Data{AddressRows}++;
    }
    $Data{Address} = $Addresses;

    # generate output
    $Output .= $LayoutObject->Output(
        TemplateFile => "DTSSoapUser",
        Data => \%Data
    );
    $Output .= $LayoutObject->Footer();

    return $Output;
}

1;
