package Kernel::System::DTSSoapUser;

use strict;
use warnings;
use Kernel::System::Valid;

our $VERSION = "1.0";

sub new {
    my $Type = shift;
    my %Param = @_;

    my $Self = {};
    bless($Self, $Type);

    # adopt all provided objects
    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed stuff
    foreach (qw(DBObject LogObject)) {
        if (!$Self->{$_}) {
            die "Got no $_!";
        }
    }

    # create needed objects
    $Self->{ValidObject} = Kernel::System::Valid->new(%Param);

    return $Self;
}

# return all soap users as a hash with references to hashes
sub SoapUserList ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    my $SQL = "SELECT username, password, ".
        "valid_id, create_time, create_by, change_time, change_by ".
        "FROM dts_soap_user";
    my @Clauses = ();
    if ($Param{Valid}) {
        push @Clauses, "valid_id IN (".
            join(",", $Self->{ValidObject}->ValidIDsGet()).")";
    }
    if ($Param{Username}) {
        push @Clauses, "username='".$DBObject->Quote($Param{Username})."'";
    }
    if (@Clauses) {
        $SQL .= " WHERE ".join(" AND ", @Clauses);
    }
    if (!$DBObject->Prepare(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }
    my %Data = ();
    while (my @Row = $DBObject->FetchrowArray()) {
        my $Username = $Row[0];
        $Data{$Username}->{Password} = $Row[1];
        $Data{$Username}->{ValidID} = $Row[2];
        $Data{$Username}->{CreateTime} = $Row[3];
        $Data{$Username}->{CreateBy} = $Row[4];
        $Data{$Username}->{ChangeTime} = $Row[5];
        $Data{$Username}->{ChangeBy} = $Row[6];
    }

    $SQL = "SELECT username, address, netmask FROM dts_soap_user_address";
    if (!$DBObject->Prepare(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    while (my @Row = $DBObject->FetchrowArray()) {
        my $Username = $Row[0];
        my $Address = $Row[1];
        my $Netmask = $Row[2];
        $Data{$Username}->{Address}->{$Address} = $Netmask;
    }

    $SQL = "SELECT username, method FROM dts_soap_user_method";
    if (!$DBObject->Prepare(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    while (my @Row = $DBObject->FetchrowArray()) {
        my $Username = $Row[0];
        push @{$Data{$Username}->{Method}}, $Row[1];
    }

    return %Data;
}

# create a new soap user
sub SoapUserAdd ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    my $SQL = "INSERT INTO dts_soap_user (username, password, valid_id, ".
       "create_time, create_by, change_time, change_by) VALUES ('".
       $DBObject->Quote($Param{Username})."', '".
       $DBObject->Quote($Param{Password})."', ".
       $DBObject->Quote($Param{ValidID}, "Integer").", current_timestamp, ".
       $DBObject->Quote($Self->{UserID}, "Integer").", current_timestamp, ".
       $DBObject->Quote($Self->{UserID}, "Integer").")";
    if (!$DBObject->Do(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    if (!$Self->SoapAddAddressMethod(%Param)) {
        return;
    }

    return 1;
}

# update a soap user
sub SoapUserModify ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    my $SQL = "UPDATE dts_soap_user SET username='".
       $DBObject->Quote($Param{Username})."', password='".
       $DBObject->Quote($Param{Password})."', valid_id=".
       $DBObject->Quote($Param{ValidID}, "Integer").", change_by=".
       $DBObject->Quote($Self->{UserID}, "Integer").", change_time=".
       "current_timestamp WHERE username='".
       $DBObject->Quote($Param{OldUsername})."'";
    if (!$DBObject->Do(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    if (!$Self->SoapAddAddressMethod(%Param)) {
        return;
    }

    return 1;
}

sub SoapAddAddressMethod ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};

    my $SQL = "DELETE FROM dts_soap_user_address WHERE username='".
        $DBObject->Quote($Param{Username})."'";
    if (!$DBObject->Do(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    foreach my $Address(keys %{$Param{Address}}) {
        my $Netmask = $Param{Address}->{$Address} || 32;
        if (defined $Self->CalculateNet(Address => $Address, Netmask => $Netmask)) {
            $SQL = "INSERT INTO dts_soap_user_address (username, address, netmask) VALUES ('".
                $DBObject->Quote($Param{Username})."', '".
                $DBObject->Quote($Address)."', '".
                $DBObject->Quote($Netmask)."')";
            if (!$DBObject->Do(SQL => $SQL)) {
                $Self->{LogObject}->Log(
                    Priority => "Error",
                    Message => $DBObject->Error()
                );
            }
        }
    }

    $SQL = "DELETE FROM dts_soap_user_method WHERE username='".
        $DBObject->Quote($Param{Username})."'";
    if (!$DBObject->Do(SQL => $SQL)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => $DBObject->Error()
        );
        return;
    }

    foreach my $Method(@{$Param{Method}}) {
        $SQL = "INSERT INTO dts_soap_user_method (username, method) VALUES ('".
                $DBObject->Quote($Param{Username})."', '".
                $DBObject->Quote($Method)."')";
        if (!$DBObject->Do(SQL => $SQL)) {
            $Self->{LogObject}->Log(
                Priority => "Error",
                Message => $DBObject->Error()
            );
            return;
        }
    }

    return 1;
}

# returns 1 if soap user is allowed to called given function
sub IsSoapUserAllowed ()
{
    my $Self = shift;
    my %Param = @_;
    my $DBObject = $Self->{DBObject};
    my $Username = $Param{Username};
    my $Password = $Param{Password};
    my $Method = $Param{Method};
    my $Address = $Param{Address};

    my %SoapUser = $Self->SoapUserList(
        Username => $Username,
        Valid => 1
    );
    if (!%SoapUser) {
        return;
    }
    if ($SoapUser{$Username}->{Password} ne $Password) {
        return;
    }

    my $Allowed = 0;
    foreach (@{$SoapUser{$Username}->{Method}}) {
        if ($_ eq $Method) {
            $Allowed = 1;
            last;
        }
    }
    if (!$Allowed) {
        return;
    }

    $Allowed = 0;
    foreach (keys %{$SoapUser{$Username}->{Address}}) {
        my $Netmask = $SoapUser{$Username}->{Address}->{$_};
        my $Net1 = $Self->CalculateNet(Address => $_, Netmask => $Netmask);
        if (!defined $Net1) {
            return;
        }
        my $Net2 = $Self->CalculateNet(Address => $Address, Netmask => $Netmask);
        if (!defined $Net2) {
            return;
        }
        if ($Net1 eq $Net2) {
            $Allowed = 1;
            last;
        }
    }
    if (!$Allowed) {
        return;
    }

    return 1;
}

sub CalculateNet ()
{
    my $Self = shift;
    my %Param = @_;
    my $Address = $Param{Address};
    my $Netmask = $Param{Netmask};

    foreach (qw(Address Netmask)) {
        if (!defined $Param{$_}) {
            $Self->{LogObject}->Log(
                Priority => "Error",
                Message => "Got no ".$_
            );
            return;
        }
    }

    if (($Netmask < 0) || ($Netmask > 32)) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => "Invalid netmask in address: ".$Address."/".$Netmask
        );
        return;
    }

    my @Octets = split(/\./, $Address);
    if ($#Octets != 3) {
        $Self->{LogObject}->Log(
            Priority => "Error",
            Message => "Invalid address: ".$Address."/".$Netmask
        );
        return;
    }

    my $Net = 0;
    foreach my $Octet(@Octets) {
        if (($Octet !~ /^\d+$/o) || ($Octet < 0) || ($Octet > 255)) {
            $Self->{LogObject}->Log(
                Priority => "Error",
                Message => "Invalid octet in address: ".$Address."/".$Netmask
            );
            return;
        }
        $Net <<= 8;
        $Net += $Octet;
    }

    $Net &= 0xffffffff << (32 - $Netmask);
    return $Net;
}

1;
