#!/usr/bin/perl

package Kernel::DTSSoap;

use strict;
use warnings;
use Kernel::Config;
use Kernel::System::Time;
use Kernel::System::Log;
use Kernel::System::Main;
use Kernel::System::DB;
use Kernel::System::Group;
use Kernel::System::Valid;
use Kernel::System::Queue;
use Kernel::System::CustomerUser;
use Kernel::System::CustomerCompany;
use Kernel::System::CustomerGroup;
use Kernel::System::DTSSoapUser;
use Kernel::System::DTSWsdl;

our $VERSION = "1.0";

sub new
{
    my $Type = shift;
    my %Param = @_;

    my $Self = {};
    bless($Self, $Type);

    $Self->{ConfigObject} = Kernel::Config->new();
    $Self->{LogObject} = Kernel::System::Log->new(
        LogPrefix => "DTS-SOAP",
        %$Self
    );
    $Self->{TimeObject} = Kernel::System::Time->new(%$Self);
    $Self->{MainObject} = Kernel::System::Main->new(%$Self);
    $Self->{DBObject} = Kernel::System::DB->new(%$Self);
    $Self->{GroupObject} = Kernel::System::Group->new(%$Self);
    $Self->{QueueObject} = Kernel::System::Queue->new(%$Self);
    $Self->{CustomerUserObject} = Kernel::System::CustomerUser->new(%$Self);
    $Self->{CustomerCompanyObject} = Kernel::System::CustomerCompany->new(%$Self);
    $Self->{CustomerGroupObject} = Kernel::System::CustomerGroup->new(%$Self);
    $Self->{SoapUserObject} = Kernel::System::DTSSoapUser->new(%$Self);
    $Self->{WsdlObject} = Kernel::System::DTSWsdl->new(%$Self);

    my $ValidObject = Kernel::System::Valid->new(%$Self);
    my @ValidIDs = $ValidObject->ValidIDsGet();
    $Self->{ValidID} = shift @ValidIDs || 1;
    $Self->{ValidObject} = $ValidObject;

    return $Self;
}

=begin WSDL

_DOC Creates a new debtor, including a new group, a new queue,
        and a new customer company. The new group is placed into
        the role $DTSSoap::AutoProjectRole with r/w permissions.
_IN User $string User to authenticate
_IN Pass $string Password for user
_IN DebtorID $string ID of new debtor
_IN DebtorName $string Name of new debtor
_IN DebtorComment $string Comment for new debtor
_IN DebtorStreet $string Address of new debtor
_IN DebtorZIP $string Postal code of new debtor
_IN DebtorCity $string City of new debtor
_IN DebtorCountry $string Country of new debtor
_IN DebtorURL $string Webaddress for new debtor
_RETURN $string DebtorID on success, dies on error

=end WSDL

=cut

sub CreateDebtor ()
{
    my $Self = shift;
    if (!ref $Self) {
        $Self = $Self->new();
    }
    my ($User, $Pass, $DebtorID, $DebtorName, $DebtorComment, $DebtorStreet,
        $DebtorZIP, $DebtorCity, $DebtorCountry, $DebtorURL) = @_;
    my $AutoProjectRole = $Self->{ConfigObject}->Get("DTSSoap::AutoProjectRole") || "AutoProject";

    $Self->CheckPermission($User, $Pass);

    my $GroupID = $Self->{GroupObject}->GetGroupIdByName(Group => $DebtorName);
    if (!$GroupID) {
        $GroupID = $Self->{GroupObject}->GroupAdd(
            Name => $DebtorName,
            Comment => $DebtorComment,
            ValidID => $Self->{ValidID},
            UserID => 1,
        );
    }
    if (!$GroupID) {
        $Self->Die("Unknown group: ".$DebtorName);
    }

    my $RoleID = $Self->{GroupObject}->GetRoleIdByName(Role => $AutoProjectRole);
    if (!$RoleID) {
        $Self->Die("Unknown role: ".$AutoProjectRole);
    }
    if (!$Self->{GroupObject}->GroupRoleMemberAdd(
        GID => $GroupID,
        RID => $RoleID,
        Permission => {
            rw => 1
        },
        UserID => 1
    )) {
        $Self->Die();
    }

    my $QueueID = $Self->{QueueObject}->QueueLookup(
        Queue => $DebtorName
    );
    if (!$QueueID) {
        $QueueID = $Self->{QueueObject}->QueueAdd(
            Name => $DebtorName,
            GroupID => $GroupID,
            SystemAddressID => 1,
            SalutationID => 1,
            SignatureID => 1,
            MoveNotify => 0,
            StateNotify => 0,
            LockNotify => 0,
            OwnerNotify => 0,
            Comment => $DebtorComment,
            ValidID => $Self->{ValidID},
            UserID => 1
        );
    }
    if (!$QueueID) {
        $Self->Die();
    }

    my %Company = $Self->{CustomerCompanyObject}->CustomerCompanyGet(
        CustomerID => $DebtorID
    );
    if (!%Company) {
        $DebtorID = $Self->{CustomerCompanyObject}->CustomerCompanyAdd(
            CustomerID => $DebtorID,
            CustomerCompanyName => $DebtorName,
            CustomerCompanyStreet => $DebtorStreet,
            CustomerCompanyZIP => $DebtorZIP,
            CustomerCompanyLocation => $DebtorCity,
            CustomerCompanyCountry => $DebtorCountry,
            CustomerCompanyComment => $DebtorComment,
            CustomerCompanyURL => $DebtorURL,
            ValidID => $Self->{ValidID},
            UserID => 1,
        );
        if (!$DebtorID) {
            $Self->Die();
        }
    }

    return SOAP::Data->type("string")
                     ->name("CreateDebtorReturn")
                     ->value($DebtorID);
}

=begin WSDL

_DOC Creates a new project beneath the given debtor, including a new queue.
        It assumes that the name of the customer company equals the group name
        into which the new project/queue will be placed.
_IN User $string User to authenticate
_IN Pass $string Password for user
_IN ProjectName $string Name of new project
_IN ProjectDebtorID $string ID of debtor to which this project belongs
_RETURN $string ID of new queue on success, dies on error

=end WSDL

=cut

sub CreateProject ()
{
    my $Self = shift;
    if (!ref $Self) {
        $Self = $Self->new();
    }
    my ($User, $Pass, $ProjectName, $ProjectDebtorID) = @_;

    $Self->CheckPermission($User, $Pass);

    my %Company = $Self->{CustomerCompanyObject}->CustomerCompanyGet(
        CustomerID => $ProjectDebtorID
    );
    if (!%Company) {
        $Self->Die("Unknown CustomerID/DebtorID: ".$ProjectDebtorID);
    }
    my $CompanyName = $Company{CustomerCompanyName};

    my $GroupID = $Self->{GroupObject}->GetGroupIdByName(Group => $CompanyName);
    if (!$GroupID) {
        $Self->{LogObject}->Log(
            Priority => "error",
            Message => "Unknown group: ".$CompanyName
        );
        return;
    }

    my $QueueName = $CompanyName."::".$ProjectName;

    my $QueueID = $Self->{QueueObject}->QueueLookup(
        Queue => $QueueName
    );
    if (!$QueueID) {
        $QueueID = $Self->{QueueObject}->QueueAdd(
            Name => $QueueName,
            GroupID => $GroupID,
            SystemAddressID => 1,
            SalutationID => 1,
            SignatureID => 1,
            MoveNotify => 0,
            StateNotify => 0,
            LockNotify => 0,
            OwnerNotify => 0,
            Comment => $ProjectName,
            ValidID => $Self->{ValidID},
            UserID => 1
        );
    }

    if (!$QueueID) {
        $Self->Die();
    }

    return SOAP::Data->type("string")
                     ->name("CreateProjectReturn")
                     ->value($QueueID);
}

=begin WSDL

_DOC Creates a new customer person, and gives him/her full
        permissions in the group denoted by ContactDebtorID
_IN User $string User to authenticate
_IN Pass $string Password for user
_IN ContactLogin $string Login for new contact
_IN ContactPass $string Password for new contact
_IN ContactSalutation $string Salutation of new contact
_IN ContactFirstname $string First name of new contact
_IN ContactSurname $string Surname of new contact
_IN ContactEmail $string email address of new contact
_IN ContactDebtorID $string ID of debtor to which the new contact belongs
_IN ContactPhone $string phone number of new contact
_IN ContactFax $string fax number of new contact
_IN ContactComment $string comment for new contact
_RETURN $string ContactLogin on success, dies on error

=end WSDL

=cut

sub CreateContact ()
{
    my $Self = shift;
    if (!ref $Self) {
        $Self = $Self->new();
    }
    my ($User, $Pass, $ContactLogin, $ContactPass, $ContactSalutation, $ContactFirstname,
        $ContactSurname, $ContactEmail, $ContactDebtorID, $ContactPhone, $ContactFax,
        $ContactComment) = @_;

    $Self->CheckPermission($User, $Pass);

    my %Company = $Self->{CustomerCompanyObject}->CustomerCompanyGet(
        CustomerID => $ContactDebtorID
    );
    if (!%Company) {
        $Self->Die("Unknown CustomerID/DebtorID: ".$ContactDebtorID);
    }

    my $GroupID = $Self->{GroupObject}->GetGroupIdByName(Group => $Company{CustomerCompanyName});
    if (!$GroupID) {
        $Self->Die("Unknown group: ".$Company{CustomerCompanyName});
    }

    $ContactLogin = $Self->{CustomerUserObject}->CustomerUserAdd(
        UserSalutation => $ContactSalutation,
        UserFirstname => $ContactFirstname,
        UserLastname => $ContactSurname,
        UserLogin => $ContactLogin,
        UserPassword => $ContactPass,
        UserEmail => $ContactEmail,
        UserCustomerID => $ContactDebtorID,
        UserComment => $ContactComment,
        UserPhone => $ContactPhone,
        UserFax => $ContactFax,
        ValidID => $Self->{ValidID},
        UserID => 1
    );
    if (!$ContactLogin) {
        $Self->Die();
    }

    my $ID = $Self->{CustomerGroupObject}->GroupMemberAdd(
        GID => $GroupID,
        UID => $ContactLogin,
        Permission => {
            rw => 1
        },
        UserID => 1
    );
    if (!$ID) {
        $Self->Die();
    }

    return SOAP::Data->type("string")
                     ->name("CreateContactReturn")
                     ->value($ContactLogin);
}

sub WSDL ()
{
    my $Self = shift;
    if (!ref $Self) {
        $Self = $Self->new();
    }
    my ($User, $Pass) = @_;

    $Self->CheckPermission($User, $Pass);

    return $Self->{WsdlObject}->WSDL();
}

sub CheckPermission ()
{
    my $Self = shift;
    if (!ref $Self) {
        $Self = $Self->new();
    }
    my ($User, $Pass) = @_;

    my @CallStack = caller(1);
    my $Method = $CallStack[3];
    $Method =~ s/.*:://go;

    my $Address = $ENV{HTTP_X_FORWARDED_FOR} || $ENV{REMOTE_ADDR};
    $Address =~ s/,.*//go;

    if (!$Self->{SoapUserObject}->IsSoapUserAllowed(
        Username => $User,
        Password => $Pass,
        Method => $Method,
        Address => $Address
    )) {
        $Self->Die("User ".$User." is not authorized to call method ".
                   $Method." from remote host ".$Address);
    }
}

sub Die ()
{
    my $Self = shift;
    if (!ref $Self) {
        $Self = $Self->new();
    }
    my ($Message) = @_;

    if ($Message) {
        $Self->{LogObject}->Log(
            Priority => "error",
            Message => $Message
        );
    }

    $Message = $Self->{LogObject}->GetLogEntry(
        Type => "error",
        What => "Message"
    );

    die $Message;
}

1;
