# --
# Kernel/System/GenericAgent/DTSNotifyAgentAsterisk.pm -
#     agent notifications via Asterisk Management Interface
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --
# ...based on NotifyAgentGroupWithWritePermission.pm
# --

package Kernel::System::GenericAgent::DTSNotifyAgentAsterisk;

use strict;
use Kernel::System::User;
use Kernel::System::Group;
use Kernel::System::Queue;
use Socket;
use Asterisk::Manager;

our $VERSION = "1.0";

sub new {
    my $Type = shift;
    my %Param = @_;

    # allocate new hash for object
    my $Self = {};
    bless ($Self, $Type);

    # check needed objects
    foreach (qw(DBObject ConfigObject LogObject TicketObject TimeObject)) {
        $Self->{$_} = $Param{$_} || die "Got no $_!";
    }

    $Self->{UserObject} = Kernel::System::User->new(%Param);
    $Self->{GroupObject} = Kernel::System::Group->new(%Param);
    $Self->{QueueObject} = Kernel::System::Queue->new(%Param);
    $Self->{AMIObject} = Asterisk::Manager->new();

    return $Self;
}

sub Run {
    my $Self = shift;
    my %Param = @_;

    my $ConfigObject = $Self->{ConfigObject};
    my $TicketObject = $Self->{TicketObject};
    my $TimeObject = $Self->{TimeObject};
    my $AMIObject = $Self->{AMIObject};

    # check if business hours is, then send escalation info
    my $CountedTime = $TimeObject->WorkingTime(
        StartTime => $TimeObject->SystemTime()-(30*60),
        StopTime => $TimeObject->SystemTime(),
    );
    if (!$CountedTime) {
        return 1;
    }

    # get connection info for asterisk manager interface
    my $AMIHost = $ConfigObject->Get("DTSAsterisk::AMIHostname") || "asterisk";
    my $AMIPort = $ConfigObject->Get("DTSAsterisk::AMIPort") || "5038";
    my $AMIUser = $ConfigObject->Get("DTSAsterisk::AMIUsername") || "asterisk";
    my $AMIPass = $ConfigObject->Get("DTSAsterisk::AMIPassword") || "asterisk";
    my $AsteriskChannel = $ConfigObject->Get("DTSAsterisk::Channel") || "CAPI/g0-9/<PHONE_NUMBER>";
    my $AsteriskExtension = $ConfigObject->Get("DTSAsterisk::Extension") || "666666";
    my $AsteriskTimeout = $ConfigObject->Get("DTSAsterisk::Timeout") || "15000";
    my $AsteriskContext = $ConfigObject->Get("DTSAsterisk::Context") || "default";
    my $FestivalTextKey = $ConfigObject->Get("DTSAsterisk::FestivalTextKey") || "Text";
    my $FestivalText = $ConfigObject->Get("DTSAsterisk::FestivalText") ||
        "Attention! Attention! Ticket number <TICKET_NUMBER> is escalated! ".
        "Visit the trouble ticket system immediately.";
    my $FestivalTicketNumberKey = $ConfigObject->Get("DTSAsterisk::FestivalTicketNumberKey") || "TicketNumber";

    # set connection parameters for ami server
    $AMIObject->host($AMIHost);
    $AMIObject->port($AMIPort);
    $AMIObject->user($AMIUser);
    $AMIObject->secret($AMIPass);

    # get ticket
    my %Ticket = $TicketObject->TicketGet(%Param);

    # get rw member of group
    my %Queue = $Self->{QueueObject}->QueueGet(
        ID => $Ticket{QueueID},
        Cache => 1,
    );
    my @UserIDs = $Self->{GroupObject}->GroupMemberList(
        GroupID => $Queue{GroupID},
        Type => 'rw',
        Result => 'ID',
    );

    my $TicketNumber = $Ticket{TicketNumber};
    my $TicketNumberSpaced = $TicketNumber;
    $TicketNumberSpaced =~ s/(\d)/$1 /g;
    $FestivalText =~ s/\<TICKET_NUMBER\>/$TicketNumberSpaced/g;
    my $FestivalVariable = $FestivalTextKey."=".$FestivalText."|".
        $FestivalTicketNumberKey."=".$TicketNumber;

    # call each agent
    foreach my $UserID(@UserIDs) {
        my %Preferences = $Self->{UserObject}->GetPreferences(UserID => $UserID);
        my $PhoneNumber = $Preferences{AgentPhoneNumber};
        if (!$PhoneNumber) {
            next;
        }

        # check if today a reminder is already sent
        my ($Sec, $Min, $Hour, $Day, $Month, $Year) = $TimeObject->SystemTime2Date(
            SystemTime => $TimeObject->SystemTime(),
        );
        my @Lines = $TicketObject->HistoryGet(
            TicketID => $Ticket{TicketID},
            UserID => 1,
        );
        my $Sent = 0;
        foreach my $Line(@Lines) {
            if (($Line->{Name} =~ /PhoneEscalation/) &&
                ($Line->{Name} =~ /$PhoneNumber/i) &&
                ($Line->{CreateTime} =~ /$Year-$Month-$Day/)) {
                $Sent = 1;
            }
        }
        if ($Sent) {
            next;
        }

        if (!$AMIObject->connect()) {
            die "can not connect to AMI server $AMIUser\@$AMIHost:$AMIPort: ".
                $AMIObject->error();
        }

        my $Channel = $AsteriskChannel;
        $Channel =~ s/\<PHONE_NUMBER\>/$PhoneNumber/g;

        my %AMIResult = $AMIObject->sendcommand(
            Action => "Originate",
            Channel => $Channel,
            Exten => $AsteriskExtension,
            Priority => "1",
            Async => "0",
            Timeout => $AsteriskTimeout,
            Context => $AsteriskContext,
            Variable => $FestivalVariable
        );

        $AMIObject->connfd()->shutdown(SHUT_RDWR);

        if ($AMIResult{Response} eq "Success") {
            # add history entry
            $TicketObject->HistoryAdd(
                Name => "\%\%PhoneEscalation\%\%$PhoneNumber",
                HistoryType => "SendAgentNotification",
                TicketID => $Ticket{TicketID},
                CreateUserID => 1
            );
            # log success
            $Self->{LogObject}->Log(
                Priority => 'notice',
                Message => "Sent agent phone escalation to '$PhoneNumber'."
            );
        }
        else {
            # log failure
            $Self->{LogObject}->Log(
                Priority => 'notice',
                Message => "Agent phone escalation to '$PhoneNumber' failed: ".
                    $AMIResult{Message}
            );
        }
    }

    return 1;
}

1;
