#!/usr/bin/perl

use strict;
use warnings;
use Getopt::Std;
use DTSMaster;

$Getopt::Std::STANDARD_HELP_VERSION = 1;

my %Opts = ();
getopts("f:", \%Opts);

my $MasterObject = DTSMaster->new();

if (!$MasterObject->LoadConfig(Configfile => $Opts{f})) {
    die $MasterObject->Error();
}

# connect to master database
if (!$MasterObject->ConnectDatabase()) {
    die $MasterObject->Error();
}

# get all active otrs instances
my $AllInstances = $MasterObject->GetAllInstances();
if (!$AllInstances) {
    die $MasterObject->Error();
}

# disconnect from master database
if (!$MasterObject->DisconnectDatabase()) {
    die $MasterObject->Error();
}

# track all virtual hosts, https ip addresses, email addresses, and email domains
my %AllHostnames = ();
my %AllIPAddresses = ();
my %AllEmailAddresses = ();
my %AllDomains = ();

my @Commands = (
    {
        Stylesheet => "PostfixTransportMaps",
        Configfile => "/usr/local/etc/postfix/transport_maps",
        PostCommand => "postmap /usr/local/etc/postfix/transport_maps"
    },
    {
        Stylesheet => "PostfixRelayDomains",
        Configfile => "/usr/local/etc/postfix/relay_domains",
        PostCommand => "postmap /usr/local/etc/postfix/relay_domains"
    },
    {
        Stylesheet => "PostfixMaster",
        Configfile => "/usr/local/etc/postfix/master.cf",
        PostCommand => "/usr/local/etc/rc.d/postfix restart"
    },
    {
        Stylesheet => "Proxy",
        Configfile => "/usr/local/etc/dtsmaster/proxy.conf",
        PreCommand => "/usr/local/etc/rc.d/apache22 stop proxy",
        PostCommand => "/usr/local/etc/rc.d/apache22 start proxy"
    }
);

# go through all instances
my $Instances = $AllInstances->{Instances}->{Instance};
my $InstanceIdx = 0;
while ($InstanceIdx <= $#$Instances) {
    # load instance
    my $Instance = $Instances->[$InstanceIdx];
    my $SystemID = $Instance->{system_id};
    if (!$MasterObject->LoadInstance(Instance => $Instance)) {
        warn "Instance ".$SystemID.": ".$MasterObject->Error();
        splice(@$Instances, $InstanceIdx);
        next;
    }

    # go through all virtual hosts of this instance
    my $VirtualHosts = $Instance->{VirtualHosts}->{VirtualHost};
    my $Idx = 0;
    while ($Idx <= $#$VirtualHosts) {
        my $Error = "";
        my $VirtualHost = $VirtualHosts->[$Idx];
        my $Hostname = $VirtualHost->{name};
        if (!$MasterObject->IsHostnameValid(Hostname => $Hostname)) {
            $Error = $MasterObject->Error();
        }
        elsif (defined $AllHostnames{$Hostname}) {
            $Error = "Duplicate hostname ".$Hostname.
                " already defined for instance ".$AllHostnames{$Hostname};
        }
        elsif ($VirtualHost->{https}) {
            my $IP = $VirtualHost->{ip};
            if (!$MasterObject->IsAddressValid(Address => $IP, Port => 443)) {
                $Error = $MasterObject->Error();
            }
            elsif (defined $AllIPAddresses{$IP}) {
                $Error = "Duplicate https address ".$IP.":443 ".
                    " already defined for instance ".$AllIPAddresses{$IP};
            }
            else {
                my $SSLConf = "/usr/local/etc/dtsmaster/otrs".$SystemID.".".$Hostname.".ssl.conf";
                my $KeyFile = "/usr/local/etc/dtsmaster/otrs".$SystemID.".".$Hostname.".key";
                my $CertFile = "/usr/local/etc/dtsmaster/otrs".$SystemID.".".$Hostname.".cert";

                push @Commands, {
                    Stylesheet => "OpenSSL",
                    Configfile => $SSLConf,
                    PreCommand => "openssl genrsa -out ".$KeyFile,
                    Command => "openssl req -new -x509 -config ".$SSLConf." -key ".$KeyFile." -out ".$CertFile,
                    MustNotExist => [
                        $KeyFile,
                        $CertFile
                    ],
                    Parameters => {
                        ProcessVirtualHostName => "'".$Hostname."'"
                    }
                };

                $AllIPAddresses{$IP} = $SystemID;
            }
        }
        if ($Error) {
            warn "Instance ".$SystemID.": ".$Error;
            splice(@$VirtualHosts, $Idx);
        }
        else {
            $AllHostnames{$Hostname} = $SystemID;
            $Idx++;
        }
    }

    # go through all email addresses of this instance
    my $EmailAddresses = $Instance->{EmailAddresses}->{EmailAddress};
    $Idx = 0;
    while ($Idx <= $#$EmailAddresses) {
        my $Error = "";
        my $EmailAddress = $EmailAddresses->[$Idx]->{address};
        if (!$MasterObject->IsEmailAddressValid(EmailAddress => $EmailAddress)) {
            $Error = $MasterObject->Error();
        }
        elsif (defined $AllEmailAddresses{$EmailAddress} &&
                ($AllEmailAddresses{$EmailAddress} != $SystemID)) {
            $Error = "Duplicate email address ".$EmailAddress.
                " already defined for instance ".$AllEmailAddresses{$EmailAddress};
        }
        if ($Error) {
            warn "Instance ".$SystemID.": ".$Error;
            splice(@$EmailAddresses, $Idx);
        }
        else {
            $AllEmailAddresses{$EmailAddress} = $SystemID;
            my ($Username, $Hostname) = $MasterObject->SplitEmailAddress(
                EmailAddress => $EmailAddress
            );
            if (!exists $AllDomains{$Hostname}) {
                push @{$Instance->{Domains}->{Domain}}, { domain => $Hostname };
                $AllDomains{$Hostname} = $SystemID;
            }
            $Idx++;
        }
    }

    $InstanceIdx++;
    push @Commands, {
        Stylesheet => "Server",
        Configfile => "/usr/local/etc/dtsmaster/otrs".$SystemID.".conf",
        PreCommand => "/usr/local/etc/rc.d/apache22 stop otrs".$SystemID,
        PostCommand => "/usr/local/etc/rc.d/apache22 start otrs".$SystemID,
        Parameters => { ProcessSystemID => $SystemID }
    };
}

my $XmlAllInstances = $MasterObject->Hash2XML(Hash => $AllInstances);
if (!$XmlAllInstances) {
    die $MasterObject->Error();
}

my @PostCommands = ();
foreach my $Command(@Commands) {
    my $NewConfig = $MasterObject->ApplyStylesheet(
        StylesheetName => $Command->{Stylesheet},
        ParsedXML => \$XmlAllInstances,
        Parameters => $Command->{Parameters}
    );

    if (!$NewConfig) {
        warn $MasterObject->Error();
        next;
    }

    my $CurrentConfig = $MasterObject->Readfile(Filename => $Command->{Configfile});
    my $Altered = !$CurrentConfig || (join("", @$CurrentConfig) ne $NewConfig);
    if (!$Altered) {
        next;
    }

    if ($Command->{MustNotExist}) {
        foreach (@{$Command->{MustNotExist}}) {
            if (-e $_) {
                next;
            }
        }
    }

    if ($Command->{PreCommand}) {
        if (!$MasterObject->ExecuteCommand(Command => $Command->{PreCommand})) {
            warn $MasterObject->Error()."\n...continuing anyway...";
        }
    }

    if (!$MasterObject->Writefile(
        Filename => $Command->{Configfile},
        Data => [ $NewConfig ]
    )) {
        warn $MasterObject->Error();
        next;
    }

    if ($Command->{Command}) {
        if (!$MasterObject->ExecuteCommand(Command => $Command->{Command})) {
            warn $MasterObject->Error()."\n...continuing anyway...";
        }
    }

    if ($Command->{PostCommand}) {
        push @PostCommands, $Command->{PostCommand};
    }
}

my $MasterConfig = $MasterObject->ApplyStylesheet(
    StylesheetName => "DTSMaster",
    ParsedXML => \$XmlAllInstances
);
if (!$MasterConfig) {
    die $MasterObject->Error();
}

if (!$MasterObject->Writefile(
    Filename => "/usr/local/etc/dtsmaster.conf",
    Data => [ $MasterConfig ]
)) {
    die $MasterObject->Error();
}

foreach (@PostCommands) {
    if (!$MasterObject->ExecuteCommand(Command => $_)) {
        warn $MasterObject->Error();
        next;
    }
}
