#!/usr/bin/perl

package DTSMaster;

use strict;
use warnings;
use DBI;
use Socket;
use IO::Socket;
use XML::LibXML;
use XML::LibXSLT;

sub new ()
{
    my $Type = shift;
    my %Param = @_;
    my $Self = {};

    bless($Self, $Type);

    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    $Self->{XMLObject} = XML::LibXML->new();
    $Self->{XSLTObject} = XML::LibXSLT->new();

    return $Self;
}

sub LoadConfig ()
{
    my $Self = shift;
    my %Param = @_;

    my %Defaults = ();
    $Defaults{DBDriver} = "Pg";
    $Defaults{DBHostname} = "/tmp";
    $Defaults{DBPort} = "5432";
    $Defaults{DBUsername} = "otrs";
    $Defaults{DBPassword} = "otrs";
    $Defaults{DBName} = "otrs";
    $Defaults{StylesheetDTSMaster} = "templates/dtsmaster.conf.xsl";
    $Defaults{StylesheetServer} = "templates/httpd-server.conf.xsl";
    $Defaults{StylesheetProxy} = "templates/httpd-proxy.conf.xsl";
    $Defaults{StylesheetOpenSSL} = "templates/openssl.conf.xsl";
    $Defaults{StylesheetPostfixMaster} = "templates/master.cf.xsl";
    $Defaults{StylesheetPostfixTransportMaps} = "templates/transport_maps.xsl";
    $Defaults{StylesheetPostfixRelayDomains} = "templates/relay_domains.xsl";

    while (my ($Key, $Value) = each %Defaults) {
        $Self->{$Key} = $Value;
    }

    my $Config = [];
    if (defined $Param{ConfigFile}) {
        $Config = $Self->Readfile(Filename => $Param{Configfile});
        if (!$Config) {
            $Self->Error(Error => $Param{ConfigFile}.": ".$!);
            return;
        }
    }
    foreach my $Line(@$Config) {
        if ($Line =~ /^[\s#]/sgio) {
            next;
        }
        $Line =~ s/\r|\n//sgio;
        if ($Line =~ /^([^\s]+)\s*[=:]\s*(.*)/sgio) {
            my $Key = $1;
            my $Value = $2;
            if (exists $Defaults{$Key}) {
                $Self->{$Key} = $Value;
            }
            else {
                $Self->Error(
                    Error => $Param{ConfigFile}.": Unknown option ".$Key
                );
                return;
            }
        }
        else {
            $Self->Error(Error => $Param{ConfigFile}.": Invalid line ".$Line);
            return;
        }
    }

    return 1;
}

sub Error ()
{
    my $Self = shift;
    my %Param = @_;

    if (defined $Param{Error}) {
        my ($Package, $Filename, $Line, $Subroutine, $HasArgs, $WantArray,
            $Evaltext, $IsRequire, $Hints, $Bitmask) = caller(1);
        my $Error = $Subroutine;

        ($Package, $Filename, $Line, $Subroutine, $HasArgs, $WantArray,
            $Evaltext, $IsRequire, $Hints, $Bitmask) = caller(0);
        $Error .= " (".$Filename.":".$Line."): ".$Param{Error};

        $Self->{Error} = $Error;
    }

    return $Self->{Error};
}

### add a new system group
sub CreateOsGroup ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(GroupName GroupID)) {
        if (!defined $Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my $Cmd = "pw groupadd '$Param{GroupName}' -g '$Param{GroupID}'";

    if (!$Self->ExecuteCommand(Command => $Cmd)) {
        return;
    }

    return 1;
}

### add a new system user
sub CreateOsUser ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(UserName UserID UserComment UserHome UserShell UserGroup)) {
        if (!defined $Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my $Cmd = "pw useradd '$Param{UserName}' -u '$Param{UserID}' ".
        "-c '$Param{UserComment}' -d '$Param{UserHome}' ".
        "-g '$Param{UserGroup}' -s '$Param{UserShell}'";

    if (!$Self->ExecuteCommand(Command => $Cmd)) {
        return;
    }

    return 1;
}

### execute a system command
sub ExecuteCommand ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Command)) {
        if (!defined $Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    $! = 0;
    my $Error;
    my @Output = `$Param{Command} 2>&1`;

    if ($!) {
        $Error = $!;
    }
    elsif (($? >> 8)) {
        $Error = join("", @Output);
    }
    if ($Error) {
        $Self->Error(Error => $Param{Command}.": ".$Error);
        return;
    }

    return 1;
}

### connect to database
sub ConnectDatabase ()
{
    my $Self = shift;
    my %Param = @_;

    $Self->{DBConnectString} = "DBI:".$Self->{DBDriver}.
        ":dbname=".$Self->{DBName}.
        ";host=".$Self->{DBHostname}.
        ";port=".$Self->{DBPort};
    $Self->{DBHandle} = DBI->connect(
        $Self->{DBConnectString},
        $Self->{DBUsername},
        $Self->{DBPassword},
        { PrintError => 0, RaiseError => 0 }
    );
    if (!$Self->{DBHandle}) {
        $Self->Error(Error => $Self->{DBConnectString}.
                     " (".$Self->{DBUsername}."): ".DBI::errstr);
        return;
    }

    return 1;
}

### disconnect from database
sub DisconnectDatabase ()
{
    my $Self = shift;
    my %Param = @_;

    if (!$Self->{DBHandle}) {
        return 1;
    }

    if (!$Self->{DBHandle}->disconnect()) {
        $Self->Error(Error => $Self->{DBConnectString}.
                     " (".$Self->{DBUsername}."): ".DBI::errstr);
        return;
    }

    $Self->{DBHandle} = undef;

    return 1;
}

sub LoadInstance ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Instance)) {
        if (!defined $Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my $Instance = $Param{Instance};

    if (!$Self->ConnectDatabase(
        DBDriver => $Instance->{db_driver},
        DBHostname => $Instance->{db_host},
        DBPort => $Instance->{db_port},
        DBUsername => $Instance->{db_user},
        DBPassword => $Instance->{db_password},
        DBName => $Instance->{db_name}
    )) {
        return;
    }

    my $SQL = "SELECT dts_virtual_host.*, theme.theme AS theme_name ".
        "FROM dts_virtual_host ".
        "INNER JOIN valid ON valid.id=dts_virtual_host.valid_id ".
        "INNER JOIN theme ON theme.id=dts_virtual_host.theme ".
        "WHERE valid.name='valid'";

    my $VirtualHosts = $Self->{DBHandle}->selectall_arrayref($SQL, { Slice => {} });
    if (!$VirtualHosts) {
        $Self->Error(Error => $Self->{DBHandle}->errstr);
        return;
    }

    $SQL = "SELECT DISTINCT value0 AS address FROM system_address ".
        "INNER JOIN valid ON valid.id=valid_id ".
        "WHERE valid.name='valid'";

    my $EmailAddresses = $Self->{DBHandle}->selectall_arrayref($SQL, { Slice => {} });
    if (!$EmailAddresses) {
        $Self->Error(Error => $Self->{DBHandle}->errstr);
        return;
    }

    if (!$Self->DisconnectDatabase()) {
        return;
    }

    $Instance->{VirtualHosts} = { VirtualHost => $VirtualHosts };
    $Instance->{EmailAddresses} = { EmailAddress => $EmailAddresses };

    return 1;
}

sub GetNextSystemID ()
{
    my $Self = shift;
    my %Param = @_;

    my $SQL = "SELECT max(system_id) AS current_system_id FROM dts_master";
    my $Result = $Self->{DBHandle}->selectrow_hashref($SQL);
    if (!$Result) {
        $Self->Error(Error => $Self->{DBHandle}->errstr);
        return;
    }

    my $CurrentSystemID = $Result->{current_system_id} || 0;

    return $CurrentSystemID + 1;
}

sub GetAllInstances ()
{
    my $Self = shift;
    my %Param = @_;

    my $SQL = "SELECT * FROM dts_master WHERE enabled";
    my $AllInstances = $Self->{DBHandle}->selectall_arrayref($SQL, { Slice => {} });
    if (!$AllInstances) {
        $Self->Error(Error => $Self->{DBHandle}->errstr);
        return;
    }

    return { Instances => { Instance => $AllInstances }};
}

sub CreateInstance ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Settings)) {
        if (!defined $Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my %AllOptions = ();
    my @SQLFields = ();
    my @SQLParams = ();
    my @SQLValues = ();

    foreach my $Category(@{$Param{Settings}}) {
        foreach my $Option(@{$Category->{Options}}) {
            push @SQLFields, $Option->{ID};
            push @SQLParams, "?";
            push @SQLValues, $Option->{Value};
            $AllOptions{$Option->{ID}} = $Option->{Value};
        }
    }

    my $SQL = "INSERT INTO dts_master (".
        join(",", map { '"'.$_.'"' } @SQLFields).
        ") VALUES (".
        join(",", @SQLParams).
        ")";

    if (!$Self->{DBHandle}->do($SQL, undef, @SQLValues)) {
        $Self->Error(Error => $Self->{DBHandle}->errstr);
        return;
    }

    $SQL = "CREATE USER ".$AllOptions{db_user}." UNENCRYPTED PASSWORD ?";
    if (!$Self->{DBHandle}->do($SQL, undef, $AllOptions{db_password})) {
        $Self->Error(Error => $Self->{DBHandle}->errstr);
        return;
    }

    $SQL = "CREATE DATABASE ".$AllOptions{db_name}." OWNER=".
        $AllOptions{db_user}." ENCODING='UTF8'";
    if (!$Self->{DBHandle}->do($SQL)) {
        $Self->Error(Error => $Self->{DBHandle}->errstr);
        return;
    }

    if (!$Self->CreateOsGroup(
        GroupName => $AllOptions{web_group},
        GroupID => $AllOptions{web_gid}
    )) {
        return;
    }

    if (!$Self->CreateOsGroup(
        GroupName => $AllOptions{os_group},
        GroupID => $AllOptions{os_gid}
    )) {
        return;
    }

    if (!$Self->CreateOsUser(
        UserName => $AllOptions{web_user},
        UserID => $AllOptions{web_uid},
        UserGroup => $AllOptions{web_group},
        UserComment => $AllOptions{os_comment}." (Apache user)",
        UserHome => "/nonexistent",
        UserShell => "/sbin/nologin"
    )) {
        return;
    }

    if (!$Self->CreateOsUser(
        UserName => $AllOptions{os_user},
        UserID => $AllOptions{os_uid},
        UserGroup => $AllOptions{os_group},
        UserComment => $AllOptions{os_comment},
        UserHome => $AllOptions{os_home},
        UserShell => "/sbin/nologin"
    )) {
        return;
    }

# TODO: create otrs in database & filesystem!

    return 1;
}

sub RandomPassword ()
{
    my $Self = shift;
    my %Param = @_;

    # don't use percent sign here (format identifier for printf etc.)
    # don't use paragraph sign here (developer's platform (MacOS X) enters
    # them as two-byte values)
    my @Dictionary = (
        "a".."z", "A".."Z", "0".."9", "#", ",",
        qw(! " $ & / ( ) = ? * + ' - _ . : ; < > ^ @)
    );

    my $Length = $Param{Length} || 8;
    my $Password = "";

    while ($Length) {
        my $Index = int(rand($#Dictionary + 1));
        $Password .= $Dictionary[$Index];
        $Length--;
    }

    return $Password;
}

sub LoadDefaults ()
{
    my $Self = shift;
    my %Param = @_;

    my $NextSystemID = $Self->GetNextSystemID();
    if (!defined $NextSystemID) {
        return;
    }
    if ($NextSystemID !~ /^\d+$/sgio) {
        $Self->Error(Error => "Got non-numeric System ID: ".$NextSystemID);
        return;
    }

    # each option ID has to match a field in the dts_master table!
    my @Defaults = ({
        Category => "base",
        Name => "Base settings",
        Options => [{
            ID => "system_id",
            Name => "System ID",
            Value => $NextSystemID
        }, {
            ID => "hostname",
            Name => "Primary hostname",
            Value => "otrs".$NextSystemID.".dts-online.net"
        }, {
            ID => "web_port",
            Name => "Backend TCP port",
            Value => 3000 + $NextSystemID
        }, {
            ID => "os_comment",
            Name => "Customer name",
            Value => "Random OTRS customer #".$NextSystemID
        }]}, {
        Category => "otrs",
        Name => "OTRS administrator",
        Options => [{
            ID => "admin_login",
            Name => "Login",
            Value => "otrs".$NextSystemID
        }, {
            ID => "admin_fname",
            Name => "Firstname",
            Value => "John OTRS".$NextSystemID
        }, {
            ID => "admin_lname",
            Name => "Surname",
            Value => "Smith OTRS".$NextSystemID
        }, {
            ID => "admin_email",
            Name => "Email",
            Value => "otrs".$NextSystemID."\@otrs".$NextSystemID.
                ".dts-online.net"
        }, {
            ID => "admin_password",
            Name => "Password",
            Value => $Self->RandomPassword()
        }]}, {
        Category => "os",
        Name => "Operating system settings",
        Options => [{
            ID => "os_user",
            Name => "OS user",
            Value => "otrs".$NextSystemID
        }, {
            ID => "os_uid",
            Name => "OS user id",
            Value => 2000+$NextSystemID
        }, {
            ID => "os_group",
            Name => "OS group",
            Value => "otrs".$NextSystemID
        }, {
            ID => "os_gid",
            Name => "OS group id",
            Value => 2000+$NextSystemID
        }, {
            ID => "os_home",
            Name => "Home directory",
            Value => "/var/otrs/otrs".$NextSystemID
        }]}, {
        Category => "web",
        Name => "Apache settings",
        Options => [{
            ID => "web_user",
            Name => "Apache user",
            Value => "otrswww".$NextSystemID
        }, {
            ID => "web_uid",
            Name => "Apache user id",
            Value => 3000+$NextSystemID
        }, {
            ID => "web_group",
            Name => "Apache group",
            Value => "otrswww".$NextSystemID
        }, {
            ID => "web_gid",
            Name => "Apache group id",
            Value => 3000+$NextSystemID
        }]}, {
        Category => "db",
        Name => "Database settings",
        Nextstep => "db2",
        Options => [{
            ID => "db_driver",
            Name => "Database driver",
            Value => "Pg"
        }, {
            ID => "db_host",
            Name => "Database host",
            Value => "/tmp"
        }, {
            ID => "db_port",
            Name => "Port# of database",
            Value => "5432"
        }]}, {
        Category => "db2",
# no Name here, because db2 is Nextstep of db
        Options => [{
            ID => "db_name",
            Name => "Database name",
            Value => "otrs".$NextSystemID
        }, {
            ID => "db_user",
            Name => "Database user",
            Value => "otrs".$NextSystemID
        }, {
            ID => "db_password",
            Name => "Database password",
            Value => $Self->RandomPassword(Length => 16)
        }]}
    );

    return \@Defaults;
}

sub IsAddressValid ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Address Port)) {
        if (!$Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my $Address = $Param{Address};
    my $Port = $Param{Port};

    if ($Address =~ /^127\./) {
        return;
    }

    my $Socket = IO::Socket::INET->new(
        LocalAddr => $Address,
        LocalPort => $Port,
        Listen => 1,
        ReuseAddr => 1,
        ReusePort => 1,
        Proto => "tcp"
    );

    if (!$Socket) {
        $Self->Error(
            Error => "Invalid local address: ".$Address.":".$Port
        );
        return;
    }

    shutdown($Socket, SHUT_RDWR);

    return 1;
}

sub IsHostnameValid ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Hostname)) {
        if (!$Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    if ($Param{Hostname} !~ /^([A-Za-z0-9\-\.]+)+\.[A-Za-z0-9\-]{2,}$/) {
        $Self->Error(
            Error => "Invalid hostname: ".$Param{Hostname}
        );
        return;
    }

    return 1;
}

sub IsEmailAddressValid ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(EmailAddress)) {
        if (!$Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my ($Username, $Hostname) = $Self->SplitEmailAddress(EmailAddress => $Param{EmailAddress});
    if (!$Self->IsHostnameValid(Hostname => $Hostname) ||
        ($Username !~ /^[A-Za-z0-9\-\_\.]+$/)) {
        $Self->Error(
            Error => "Invalid email address: ".$Param{EmailAddress}
        );
        return;
    }

    return 1;
}

sub SplitEmailAddress ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(EmailAddress)) {
        if (!$Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my ($Username, $Hostname) = split(/\@/, $Param{EmailAddress}, 2);

    return ($Username, $Hostname);
}

sub _hash2xml ()
{
    my $Self = shift;
    my %Param = @_;

    my $XML = "";
    if ($Param{Node}) {
        $XML = "<".$Param{Node}.">\n";
    }
    else {
        $XML = "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n";
    }

    while (my ($Key, $Value) = each %{$Param{Hash}}) {
        if (ref($Value) eq "HASH") {
            $XML .= $Self->_hash2xml(Hash => $Value, Node => $Key);
        }
        elsif (ref($Value) eq "ARRAY") {
            foreach (@$Value) {
                $XML .= $Self->_hash2xml(Hash => $_, Node => $Key);
            }
        }
        elsif (ref($Value) eq "SCALAR") {
            $XML .= "<".$Key.">".$$Value."</".$Key.">\n";
        }
        elsif (!ref($Value)) {
            $XML .= "<".$Key.">".$Value."</".$Key.">\n";
        }
        else {
            warn "Unhandled ".ref($Value).": ".$Key;
        }
    }

    if ($Param{Node}) {
        $XML .= "</".$Param{Node}.">\n";
    }

    return $XML;
}

sub Hash2XML ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Hash)) {
        if (!$Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my $XML = $Self->_hash2xml(Hash => $Param{Hash});
    my $ParsedXML = $Self->{XMLObject}->parse_string($XML);
    if (!$ParsedXML) {
        $Self->Error(Error => "can not parse hash into XML!\n".
            "Document follows:\n".$XML);
        return;
    }

    return $ParsedXML;
}

sub Readfile ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Filename)) {
        if (!$Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    if (!open(FH, $Param{Filename})) {
        $Self->Error(Error => "can not read ".$Param{Filename}.": ".$!);
        return;
    }

    my @Data = <FH>;
    close(FH);

    return \@Data;
}

sub Writefile ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Filename Data)) {
        if (!$Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my $Tempfile = $Param{Filename}.".tmp";

    if (!open(FH, ">".$Tempfile)) {
        $Self->Error(Error => "can not write ".$Tempfile.": ".$!);
        return;
    }

    if (!flock(FH, 2)) {
        $Self->Error(Error => "can not lock ".$Tempfile.": ".$!);
        return;
    }

    if (!seek(FH, 0, 0)) {
        $Self->Error(Error => "can not seek ".$Tempfile.": ".$!);
        return;
    }

    print FH @{$Param{Data}};
    close(FH);

    if (!rename($Tempfile, $Param{Filename})) {
        $Self->Error(Error => "can not rename ".$Tempfile." to ".$Param{Filename}.": ".$!);
        return;
    }

    return 1;
}

sub ApplyStylesheet ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(StylesheetName ParsedXML)) {
        if (!$Param{$_}) {
            $Self->Error(Error => "Got no $_!");
            return;
        }
    }

    my $StylesheetName = $Param{StylesheetName};
    my $CacheName = "Cache".$StylesheetName;
    my $Stylesheet = $Self->{$CacheName};

    if (!$Stylesheet) {
        my $Filename = $Self->{"Stylesheet".$StylesheetName};
        $Stylesheet = $Self->{XSLTObject}->parse_stylesheet(
            $Self->{XMLObject}->parse_file($Filename)
        );
        if (!$Stylesheet) {
            $Self->Error(Error => "can not parse stylesheet ".$StylesheetName.
                         " (filename ".$Filename.")");
            return;
        }

        $Self->{$CacheName} = $Stylesheet;
    }

    my $Result = $Stylesheet->transform(
        ${$Param{ParsedXML}},
        %{$Param{Parameters}}
    );
    if (!defined $Result) {
        $Self->Error(Error => "can not apply stylesheet ".$StylesheetName);
        return;
    }

    return $Stylesheet->output_string($Result);
}

1;
