#!/usr/bin/perl

### modules
use strict;
use warnings;
use Getopt::Std;
use DTSMaster;

$Getopt::Std::STANDARD_HELP_VERSION = 1;

my $DisplayObject = DTSDisplay->new(
    Title => "DTS Master",
    ScreenWidth => 80,
    ScreenHeight => 24
);

my $Result = &LaunchInScreen($0, @ARGV);
if ($Result) {
    $DisplayObject->Error(Messages => [
        "This program makes several important changes to the ".
        "operating system. To become uninterruptable, it executes ".
        "itself in a screen session. This failed due to the ".
        "following reason: ",
        "",
        $Result,
        "",
        "Giving up. No changes have been made so far."
    ]);
}

my %Opts = ();
getopts("f:", \%Opts);

$DisplayObject->Message(
    Title => "OTRS Setup",
    Messages => [
        "You are now going to setup a new OTRS instance on this machine. ".
        "This instance will be available as soon as the local Apache proxy ".
        "server reloads its configuration (once per 5 minutes per default).",
        "For full functionality you also have to",
        "",
        "- add/change the new hostname to point to this machine in the DNS",
        "- inform the administrators and users about this new instance.",
    ]
);

my $MasterObject = DTSMaster->new();
if (!$MasterObject->LoadConfig(ConfigFile => $Opts{f})) {
    die $MasterObject->Error();
}

if (!$MasterObject->ConnectDatabase()) {
    $DisplayObject->Error(Messages => [ $MasterObject->Error() ]);
}

my $Settings = $MasterObject->LoadDefaults();
if (!$Settings) {
    $DisplayObject->Error(Messages => [ $MasterObject->Error() ]);
}

$Result = "";
do {
    $Result = $DisplayObject->Chooser(
        Title => "Options",
        Settings => $Settings
    );

    $DisplayObject->Input(
        Category => $Result,
        Settings => $Settings
    );
} while (($Result ne $DisplayObject->GetOK()) &&
    ($Result ne $DisplayObject->GetCancel()));

if ($Result eq $DisplayObject->GetOK()) {
    my @Table = ();
    foreach my $Category(@$Settings) {
        foreach my $Option(@{$Category->{Options}}) {
            push @Table, [ $Option->{Name}, $Option->{Value} ];
        }
    }
    $Result = $DisplayObject->Table(
        Title => "Overview",
        Table => \@Table,
        Cancel => 1
    );
}

if ($Result eq $DisplayObject->GetOK()) {
    if (!$MasterObject->CreateInstance(Settings => $Settings)) {
        $DisplayObject->Error(Messages => [ $MasterObject->Error() ]);
    }
    $DisplayObject->Message(
        Title => "OTRS Setup",
        Messages => [
            "The parameters of the new instance have successfully been ".
            "saved to the database. Be patient and wait a few minutes for ".
            "it to become active."
        ]
    );
}

if (!$MasterObject->DisconnectDatabase()) {
    $DisplayObject->Error(Messages => [ $MasterObject->Error() ]);
}

exit(0);


sub LaunchInScreen ()
{
    my ($Command, @Arguments) = @_;

    my $ScreenCommand = "screen";
    my $ScreenTerminalType = "screen";
    my $LoopProtect = "LoopProtector-for-".$Command;

    my $Result = "";

    if (($ENV{TERM} ne $ScreenTerminalType) && (!exists $ENV{$LoopProtect})) {
        $ENV{$LoopProtect} = 1;
        my $State = exec($ScreenCommand, $Command, @Arguments);
        $Result = $ScreenCommand.": ".$!;
    }

    return $Result;
}

sub HELP_MESSAGE ()
{
    print "help not implemented yet\n";
}

sub VERSION_MESSAGE ()
{
    my $Version = "1.0";
    print "DTSMaster.pl - Version ".$Version."\n";
}


package DTSDisplay;
use strict;
use Dialog;

sub new ()
{
    my $Type = shift;
    my %Param = @_;
    my $Self = {};

    bless($Self, $Type);

    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    foreach (qw(Title ScreenWidth ScreenHeight)) {
        if (!defined $Param{$_}) {
            die "Got no $_!";
        }
    }

    $Self->{Width} = $Self->{ScreenWidth} - 12;
    $Self->{Height} = $Self->{ScreenHeight} - 11;
    $Self->{InnerBorder} = 2;

    return $Self;
}

sub Error ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Messages)) {
        if (!defined $Param{$_}) {
            die "Got no $_!";
        }
    }

    $Self->Message(Title => "Error", Messages => $Param{Messages});

    exit($Param{ExitCode} or -1);
}

sub GetObjectWidth ()
{
    my $Self = shift;
    my %Param = @_;

    my $Object = $Param{Object};
    my $Length = length($Param{String});
    my $ExtraLength = 0;

    if ($Object eq "Button") { $ExtraLength = 5; }
    if ($Object eq "InputLine") {
        $ExtraLength = ($Length < 10 ? 14 - $Length : 4);
    }

    return $Length + $ExtraLength;
}

sub GetObjectHeight ()
{
    my $Self = shift;
    my %Param = @_;

    my $Object = $Param{Object};
    my $Height = 1;

    if ($Object eq "Button") { $Height = 3; }
    if ($Object eq "InputLine") { $Height = 3; }

    return $Height;
}

sub GetOK ()
{
    my $Self = shift;
    my %Param = @_;
    return "OK";
}

sub GetCancel ()
{
    my $Self = shift;
    my %Param = @_;
    return "CANCEL";
}

sub MakeDialog ()
{
    my $Self = shift;
    my %Param = @_;

    my $OKText = "  OK  ";
    my $CancelText = "Cancel";

    foreach (qw(Title Width Height)) {
        if (!defined $Param{$_}) {
            die "Got no $_!";
        }
    }

    my $Title = ".: ".$Self->{Title}." :: ".$Param{Title};
    if ($Param{PageNumber}) {
        if (!${$Param{PageNumber}}) {
            ${$Param{PageNumber}} = 1;
        }
        else {
            $Title .= " ".${$Param{PageNumber}};
        }
        ${$Param{PageNumber}}++;
    }
    $Title .= " :.";

    my $Width = $Self->GetObjectWidth(Object => "Title", String => $Title);
    if ($Width > $Param{Width}) {
        $Param{Width} = $Width;
    }
    else {
        $Width = $Param{Width};
    }
    $Width += 2 * $Self->{InnerBorder};
    my $Height = $Param{Height} + 2 * $Self->{InnerBorder} +
        $Self->GetObjectHeight(Object => "Button");

    my $Dialog = Dialog->new(
        $Title,
        ($Self->{ScreenHeight} - $Height) / 2,
        ($Self->{ScreenWidth} - $Width) / 2,
        $Height,
        $Width
    );

    my $ButtonPosY = $Height - $Self->GetObjectHeight(Object => "Button") - 1;
    my $ButtonPosX = $Width / 2;
    my $Shift = 2;
    if ($Param{Cancel}) {
        $Shift = 1;
        $Dialog->button(
            "ButtonCancel",
            $ButtonPosY,
            $ButtonPosX,
            $CancelText,
            $DisplayObject->GetCancel()
        );
    }
    $Dialog->button(
        "ButtonOk",
        $ButtonPosY,
        $ButtonPosX - $Self->GetObjectWidth(
            Object => "Button",
            String => $OKText,
        ) / $Shift,
        $OKText,
        $DisplayObject->GetOK()
    );

    return $Dialog;
}

sub Message ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Title Messages)) {
        if (!defined $Param{$_}) {
            die "Got no $_!";
        }
    }

    my @Lines = ();
    my $Index = 0;
    my $Width = 0;

    my @Messages = map { split(/[\r\n]+/, $_) } @{$Param{Messages}};
    foreach my $Message(@Messages) {
        $Message =~ s/[\t\r\n]//sgio;
        $Lines[$Index] = "";
        my $Limiter = "";
        while ($Message =~ /([^\s]+)(\s*)/sgio) {
            my $Word = $1;
            my $NewIndex = $Index;
            do {
                $Index = $NewIndex;
                if (!$Lines[$Index]) {
                    $Lines[$Index] = $Word;
                }
                elsif ($Self->{Width} >= $Self->GetObjectWidth(
                    Object => "Label",
                    String => $Lines[$Index].$Limiter.$Word
                )) {
                    $Lines[$Index] .= $Limiter.$Word;
                }
                else {
                    $NewIndex++;
                }
                my $LineWidth = $Self->GetObjectWidth(
                    Object => "Label",
                    String => $Lines[$Index]
                );
                if ($Width < $LineWidth) {
                    $Width = $LineWidth;
                }
            } while ($NewIndex != $Index);
            $Limiter = $2 || "";
        }
        $Index++;
    }

    my $PageNumber;
    my $Result;
    do {
        my $Height = $Self->GetObjectHeight(Object => "Label") * ($#Lines + 1);
        if ($Height > $Self->{Height}) {
            $Height = $Self->{Height};
        }
        my $MaxIndex = $Height / $Self->GetObjectHeight(Object => "Label");

        my $Dialog = $Self->MakeDialog(
            Title => $Param{Title},
            Height => $Height,
            Width => $Width,
            PageNumber => \$PageNumber,
            Cancel => $Param{Cancel}
        );

        $Index = 0;
        my $YPos = $Self->{InnerBorder};
        while (defined(my $Line = shift @Lines)) {
            my $Label = $Dialog->label(
                "Label".$Index,
                $YPos,
                $Self->{InnerBorder},
                $Line
            );
            $Label->draw();
            $Index++;
            $YPos += $Self->GetObjectHeight(Object => "Label");
            if ($Index >= $MaxIndex) {
                last;
            }
        }

        $Result = $Dialog->run();
    } while (@Lines && ($Result eq $DisplayObject->GetOK()));

    return $Result;
}

sub Table ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Title Table)) {
        if (!defined $Param{$_}) {
            die "Got no $_!";
        }
    }

    my $Width = 0;
    foreach my $Fields(@{$Param{Table}}) {
        my $FieldLen = length($Fields->[0]);
        if ($Width < $FieldLen) {
            $Width = $FieldLen;
        }
        $Fields->[2] = $FieldLen;
    }

    my @Messages = ();
    foreach my $Fields(@{$Param{Table}}) {
        my $FieldLenDiff = $Width - $Fields->[2];
        my $Line = $Fields->[0].":   ".(" " x $FieldLenDiff).$Fields->[1];
        push @Messages, $Line;
    }

    return $DisplayObject->Message(
        Title => $Param{Title},
        Messages => \@Messages,
        Cancel => $Param{Cancel}
    );
}

sub Chooser ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Title Settings)) {
        if (!defined $Param{$_}) {
            die "Got no $_!";
        }
    }

    my @Categories = @{$Param{Settings}};
    my $Width = 0;
    my $CountCategories = 0;

    foreach my $Category(@Categories) {
        if (!exists $Category->{Name}) {
            next;
        }
        $CountCategories++;
        my $CategoryID = $Category->{Category};
        my $CategoryName = $Category->{Name};
        my $ButtonWidth = $Self->GetObjectWidth(
            Object => "Button",
            String => $CategoryName
        );
        if ($Width < $ButtonWidth) {
            $Width = $ButtonWidth;
        }
    }

    my $Height = $Self->GetObjectHeight(Object => "Button") * $CountCategories;

    my $Dialog = $Self->MakeDialog(
        Title => $Param{Title},
        Height => $Height,
        Width => $Width,
        Cancel => 1
    );

    my $Index = 0;
    my $YPos = $Self->{InnerBorder};
    foreach my $Category(@Categories) {
        if (!exists $Category->{Name}) {
            next;
        }
        my $CategoryID = $Category->{Category};
        my $CategoryName = $Category->{Name};
        $Dialog->button(
            "Button".$Index,
            $YPos,
            $Self->{InnerBorder},
            $CategoryName,
            $CategoryID
        );
        $Index++;
        $YPos += $Self->GetObjectHeight(Object => "Button");
    }

    return $Dialog->run();
}

sub Input ()
{
    my $Self = shift;
    my %Param = @_;

    foreach (qw(Category Settings)) {
        if (!defined $Param{$_}) {
            die "Got no $_!";
        }
    }

    my @Categories = @{$Param{Settings}};
    my $Category;
    foreach (@Categories) {
        if ($Param{Category} eq $_->{Category}) {
            $Category = $_;
            last;
        }
    }
    if (!$Category) {
        return;
    }
    my @Options = @{$Category->{Options}};

    my $Width = 0;
    my $LabelWidth = 0;
    my $InputWidth = 0;

    foreach my $Option(@Options) {
        my $OptionID = $Option->{ID};
        my $OptionName = $Option->{Name};
        my $OptionValue = $Option->{Value};
        my $LineWidth = $Self->GetObjectWidth(
            Object => "Label",
            String => $OptionName.": "
        );
        if ($LabelWidth < $LineWidth) {
            $LabelWidth = $LineWidth;
        }
        $LineWidth = $Self->GetObjectWidth(
            Object => "InputLine",
            String => $OptionValue
        );
        if ($InputWidth < $LineWidth) {
            $InputWidth = $LineWidth;
        }
    }

    $Width = $LabelWidth + $InputWidth;
    if ($Width > $Self->{Width}) {
        $InputWidth -= $Width - $Self->{Width};
        $Width = $Self->{Width};
    }

    my $LineHeight = 0;
    my $LabelHeight = $Self->GetObjectHeight(Object => "Label");
    my $InputHeight = $Self->GetObjectHeight(Object => "InputLine");
    my $LabelShift = 0;
    my $InputShift = 0;
    if ($InputHeight > $LabelHeight) {
        $LineHeight = $InputHeight;
        $LabelShift = ($InputHeight - $LabelHeight) / 2;
    }
    else {
        $LineHeight = $LabelHeight;
        $InputShift = ($LabelHeight - $InputHeight) / 2;
    }
    my $Height = $LineHeight * ($#Options + 1);

    if (!$Param{PageNumber}) {
        my $PageNumber = 0;
        $Param{PageNumber} = \$PageNumber;
    }
    my $Title = $Param{Title} || $Category->{Name};
    my $Dialog = $Self->MakeDialog(
        Title => $Title,
        PageNumber => $Param{PageNumber},
        Height => $Height,
        Width => $Width,
        Cancel => 1
    );

    my $Index = 0;
    my $YPos = $Self->{InnerBorder};
    my %InputLines = ();
    foreach my $Option(@Options) {
        my $OptionID = $Option->{ID};
        my $OptionName = $Option->{Name};
        my $OptionValue = $Option->{Value};
        my $Label = $Dialog->label(
            "Label".$Index,
            $YPos + $LabelShift,
            $Self->{InnerBorder},
            $OptionName.": "
        );
        $Label->draw();
        $InputLines{$OptionID} = $Dialog->inputline(
            "InputLine".$Index,
            $YPos + $InputShift,
            $Self->{InnerBorder} + $LabelWidth,
            $InputWidth,
            $OptionValue
        );
        $Index++;
        $YPos += $LineHeight;
    }

    my $Result = $Dialog->run();

    if ($Result eq $DisplayObject->GetOK()) {
        foreach my $Option(@Options) {
            my $OptionID = $Option->{ID};
            $Option->{Value} = $InputLines{$OptionID}->data();
            $Option->{Value} =~ s/^[\s\r\n]+//sgio;
            $Option->{Value} =~ s/[\s\r\n]+$//sgio;
        }
        if (exists $Category->{Nextstep}) {
            $Self->Input(
                Title => $Title,
                PageNumber => $Param{PageNumber},
                Category => $Category->{Nextstep},
                Settings => $Param{Settings}
            );
        }
    }
}
