# --
# Kernel/Modules/DTSFreetextAdmin.pm - frontend module to administer
#                                      acl based freetexts
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --

package Kernel::Modules::DTSFreetextAdmin;

use strict;
use Kernel::System::Valid;
use Kernel::System::DTSFreetext;

our $VERSION = "1.0";

sub new ()
{
    my $Type = shift;
    my %Param = @_;
    my $Self = {};

    bless ($Self, $Type);

    # adopt all provided objects
    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed stuff
    foreach (qw(LayoutObject ParamObject QueueObject)) {
        if (!$Self->{$_}) {
            die "Got no $_!";
        }
    }

    # create additional objects
    $Self->{ValidObject} = Kernel::System::Valid->new(%Param);
    $Self->{FreetextObject} = Kernel::System::DTSFreetext->new(%Param);

    return $Self;
}

sub Run ()
{
    my $Self = shift;
    my %Param = @_;

    my $LayoutObject = $Self->{LayoutObject};
    my $ParamObject = $Self->{ParamObject};
    my $FreetextObject = $Self->{FreetextObject};

    # ValidList contains all valid AND invalid IDs
    my %ValidList = $Self->{ValidObject}->ValidList()
        or $LayoutObject->FatalError();
    # ValidIDs contains only valid IDs
    my %ValidIDs = map { $_ => $_ } $Self->{ValidObject}->ValidIDsGet();
    # QueueList contains all valid queue IDs and names
    my %QueueList = $Self->{QueueObject}->GetAllQueues();

    my %FreetextObjects = $FreetextObject->FreetextObjects();

    # start HTML output
    my $Output = $LayoutObject->Header(Title => "DTS Freetext");
    $Output .= $LayoutObject->NavigationBar();

    # fetch submitted form data
    my %Data = ();
    foreach (qw(Subaction QueueID Object Name Newname ValidID)) {
        $Data{$_} = $ParamObject->GetParam(Param => $_) || "";
    }
    if ($Data{Subaction} =~ /^Step(\d)$/o) {
        $Data{Step} = $1;
    }

    # sanity checks
    if (($Data{Step} >= 2) && !exists $FreetextObjects{$Data{Object}}) {
        $Data{Step} = 1;
    }
    if (($Data{Step} >= 1) && !exists $QueueList{$Data{QueueID}}) {
        $Data{Step} = 0;
    }

    if ($Data{Step} == 6) {
        # user sends new freetext, add it to database
        if ($FreetextObject->FreetextAdd(
            QueueID => $Data{QueueID},
            Object => $Data{Object},
            Newname => $Data{Newname},
            ValidID => $Data{ValidID}
        )) {
            $Output .= $LayoutObject->Notify(
                Info => "Freetext added"
            ); 
            $Data{Name} = $Data{Newname};
            $Data{Step} = 3;
        }
        else {
            $Output .= $LayoutObject->Notify(
                Priority => "Error"
            );
        }
    }

    if (($Data{Step} == 5) || ($Data{Step} == 6)) {
        # show mask for new freetext
        $LayoutObject->Block(
            Name => "FreetextMask",
            Data => {
                QueueID => $Data{QueueID},
                Object => $Data{Object},
                Newname => $Data{Newname},
                Subaction => "Step6",
                SubactionDisplay => "Add"
            }
        );
    }

    if ($Data{Step} == 4) {
        # user sends updated freetext, change in the database
        if ($FreetextObject->FreetextModify(
            Newname => $Data{Newname},
            ValidID => $Data{ValidID},
            QueueID => $Data{QueueID},
            Object => $Data{Object},
            Name => $Data{Name}
        )) {
            $Output .= $LayoutObject->Notify(
                Info => "Freetext updated"
            );
            $Data{Name} = $Data{Newname};
        }
        else {
            $Output .= $LayoutObject->Notify(
                Priority => "Error"
            );
        }
    }

    if (($Data{Step} == 3) || ($Data{Step} == 4)) {
        # show mask for changeable freetext
        my @FreetextList = $FreetextObject->FreetextList(
            QueueID => $Data{QueueID},
            Object => $Data{Object},
            Name => $Data{Name}
        );
        foreach (qw(QueueID Object Name ValidID)) {
            $Data{$_} = $FreetextList[0]->{$_};
        }
        $LayoutObject->Block(
            Name => "FreetextMask",
            Data => {
                QueueID => $Data{QueueID},
                Object => $Data{Object},
                Name => $Data{Name},
                Newname => $Data{Name},
                ValidID => $Data{ValidID},
                Subaction => "Step4",
                SubactionDisplay => "Update"
            }
        );
    }

    if ($Data{Step} >= 3) {
        # fill valid list
        foreach (sort keys %ValidList) {
            my $BlockName = "ValidEntry";
            $LayoutObject->Block(Name => $BlockName);
            $BlockName .= ($_ == $Data{ValidID} ? "Selected" : "Deselected");
            $LayoutObject->Block(Name => $BlockName,
                Data => {
                    ValidID => $_,
                    ValidName => $ValidList{$_}
                }
            );
        }
    }

    if ($Data{Step} >= 2) {
        # user has selected a freetext object, display select list with keys
        # or values
        my @FreetextList = $FreetextObject->FreetextList(
            QueueID => $Data{QueueID},
            Object => $Data{Object}
        );
        $LayoutObject->Block(
            Name => "FreetextBlock",
            Data => {
                QueueID => $Data{QueueID},
                Object => $Data{Object}
            }
        );
        foreach (@FreetextList) {
            my $BlockName = "FreetextEntry";
            $LayoutObject->Block(Name => $BlockName);
            $BlockName .= (exists $ValidIDs{$_->{ValidID}} ? "Valid" : "Invalid");
            $BlockName .= ($_->{Name} eq $Data{Name} ? "Selected" : "Deselected");
            $LayoutObject->Block(Name => $BlockName,
                Data => {
                    Name => $_->{Name}
                }
            );
        }
    }

    if ($Data{Step} >= 1) {
        # user has selected a queue, display select list with freetext keys &
        # values
        $LayoutObject->Block(
            Name => "ObjectBlock",
            Data => {
                QueueID => $Data{QueueID}
            }
        );
        my $BlockName = "ObjectEntry";
        foreach (sort {
            ($FreetextObjects{$a} <=> $FreetextObjects{$b})
            or ($a cmp $b)
        } keys %FreetextObjects) {
            my $Selected = ($_ eq $Data{Object} ? "Selected" : "Deselected");
            $LayoutObject->Block(Name => $BlockName);
            $LayoutObject->Block(
                Name => $BlockName.$Selected,
                Data => {
                    Object => $_
                }
            );
        }
    }

    # fill select list with queues
    foreach (sort { $QueueList{$a} cmp $QueueList{$b} } keys %QueueList) {
        my $BlockName = "QueueEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= ($_ == $Data{QueueID} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                QueueID => $_,
                QueueName => $QueueList{$_}
            }
        );
    }

    # generate output
    $Output .= $LayoutObject->Output(
        TemplateFile => "DTSFreetext",
        Data => \%Data
    );
    $Output .= $LayoutObject->Footer();

    return $Output;
}

1;
