# --
# Kernel/Modules/DTSAddressAdmin.pm - frontend module to administer email addresses
#                                     to other OTRS systems
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --

package Kernel::Modules::DTSAddressAdmin;

use strict;
use Kernel::System::Valid;
use Kernel::System::DTSAddress;

our $VERSION = "1.0";

sub new ()
{
    my $Type = shift;
    my %Param = @_;
    my $Self = {};

    bless ($Self, $Type);

    # adopt all provided objects
    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed stuff
    foreach (qw(LayoutObject ParamObject)) {
        if (!$Self->{$_}) {
            die "Got no $_!";
        }
    }

    # create additional objects
    $Self->{ValidObject} = Kernel::System::Valid->new(%Param);
    $Self->{AddressObject} = Kernel::System::DTSAddress->new(%Param);

    return $Self;
}

sub Run ()
{
    my $Self = shift;
    my %Param = @_;

    my $LayoutObject = $Self->{LayoutObject};
    my $ParamObject = $Self->{ParamObject};
    my $AddressObject = $Self->{AddressObject};
    my %Addresses = $AddressObject->AddressList();

    # ValidList contains all valid AND invalid IDs
    my %ValidList = $Self->{ValidObject}->ValidList()
        or $LayoutObject->FatalError();
    # ValidIDs contains only valid IDs
    my %ValidIDs = map { $_ => $_ } $Self->{ValidObject}->ValidIDsGet();

    # start HTML output
    my $Output = $LayoutObject->Header(Title => "DTS Addresses");
    $Output .= $LayoutObject->NavigationBar();

    # fetch submitted form data
    my %Data = ();
    foreach (qw(Subaction Name OldName Address ValidID)) {
        $Data{$_} = $ParamObject->GetParam(Param => $_) || "";
    }

    if ($Data{Subaction} eq "Change") {
        # user selected an existing address, wants to modify it, next
        # action will be "Update", unless selected address doesn't exist
        if (exists $Addresses{$Data{Name}}) {
            $Data{Subaction} = "Update";
            $Data{OldName} = $Data{Name};
            foreach (qw(Address ValidID)) {
                $Data{$_} = $Addresses{$Data{Name}}->{$_};
            }
        }
        else {
            $Data{Subaction} = "Add";
        }
    }
    elsif ($Data{Subaction} eq "Update") {
        # user sends updated address, display filled mask and hooray message
        if ($AddressObject->AddressModify(%Data)) {
            $Output .= $LayoutObject->Notify(
                Info => "Address updated"
            );
        }
        else {
            $Output .= $LayoutObject->Notify(
                Priority => "Error"
            );
        }
    }
    elsif ($Data{Subaction} eq "New") {
        # user sends new address, display empty mask and hooray message
        if ($AddressObject->AddressAdd(%Data)) {
            $Output .= $LayoutObject->Notify(
                Info => "Address added"
            );
            %Data = (
                Subaction => "Add"
            );
        }
        else {
            $Output .= $LayoutObject->Notify(
                Priority => "Error"
            );
        }
    }
    else {
        # show empty mask per default, with next action "New"
        $Data{SubactionDisplay} = "Add";
        $Data{Subaction} = "New";
    }

    # title of Add/Update button
    if (!defined $Data{SubactionDisplay}) {
        $Data{SubactionDisplay} = $Data{Subaction};
    }

    # fill select list with addresses
    %Addresses = $AddressObject->AddressList();

    foreach (sort keys %Addresses) {
        my $BlockName = "NameEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= (exists $ValidIDs{$Addresses{$_}->{ValidID}} ? "Valid" : "Invalid");
        $BlockName .= ($_ eq $Data{Name} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                Name => $_,
            }
        );
    }

    # fill select list with valid and invalid ids
    foreach (sort keys %ValidList) {
        my $BlockName = "ValidEntry";
        $LayoutObject->Block(Name => $BlockName);
        $BlockName .= ($_ == $Data{ValidID} ? "Selected" : "Deselected");
        $LayoutObject->Block(Name => $BlockName,
            Data => {
                ValidID => $_,
                ValidName => $ValidList{$_}
            }
        );
    }

    $Output .= $LayoutObject->Notify(
        Priority => "Error"
    );

    # generate output
    $Output .= $LayoutObject->Output(
        TemplateFile => "DTSAddress",
        Data => \%Data
    );
    $Output .= $LayoutObject->Footer();

    return $Output;
}

1;
