# --
# Kernel/Modules/AgentTicketDelegate.pm -
#     delegates tickets to another OTRS instance
# Copyright (C) 2007, DTS Service GmbH, http://www.dts.de/
# --

package Kernel::Modules::AgentTicketDelegate;

use strict;
use warnings;
use Kernel::System::Email;
use Kernel::System::DTSAddress;

sub new {
    my $Type = shift;
    my $Self = {};
    my %Param = @_;

    bless($Self, $Type);

    # adopt all provided objects
    foreach (keys %Param) {
        $Self->{$_} = $Param{$_};
    }

    # check needed stuff
    foreach (qw(LayoutObject ParamObject TicketObject TimeObject QueueObject)) {
        if (!$Self->{$_}) {
            die "Got no $_!";
        }
    }

    $Self->{EmailObject} = Kernel::System::Email->new(%Param);
    $Self->{AddressObject} = Kernel::System::DTSAddress->new(%Param);
    $Self->{Config} = $Self->{ConfigObject}->Get("Ticket::Frontend::$Self->{Action}");

    return $Self;
}

sub Run ()
{
    my $Self = shift;
    my %Param = @_;

    my $TicketObject = $Self->{TicketObject};
    my $LayoutObject = $Self->{LayoutObject};

    foreach (qw(TicketID)) {
        if (!$Self->{$_}) {
            return $LayoutObject->ErrorScreen(
                Message => "Got no TicketID!"
            );
        }
    }

    my $TicketID = $Self->{TicketID};

    my %Ticket = $TicketObject->TicketGet(
        TicketID => $TicketID,
        UserID => $Self->{UserID}
    );

    my @ArticleIDs = $TicketObject->ArticleIndex(
        TicketID => $TicketID
    );

    my @Articles = ();
    my %Attachments = ();
    foreach my $ArticleID(@ArticleIDs) {
        my %Article = $TicketObject->ArticleGet(
            ArticleID => $ArticleID
        );
        push @Articles, \%Article;
        my %Attachment = $TicketObject->ArticleAttachmentIndex(
            ArticleID => $ArticleID,
            UserID => $Self->{UserID}
        );
        foreach my $AttachmentID(keys %Attachment) {
            my $Key = $ArticleID."-".$AttachmentID;
            $Attachment{$AttachmentID}->{AttachmentID} = $AttachmentID;
            $Attachment{$AttachmentID}->{ArticleID} = $ArticleID;
            $Attachments{$Key} = $Attachment{$AttachmentID};
            $Key = "Attachment".$Key;
            $Param{$Key} = $Self->{ParamObject}->GetParam(Param => $Key);
        }
    }

    foreach (qw(TicketID To Subject Body TimeUnits ComposeStateID)) {
        $Param{$_} = $Self->{ParamObject}->GetParam(Param => $_) || "";
    }

    $Param{Ticket} = \%Ticket;
    $Param{Articles} = \@Articles;
    $Param{Attachments} = \%Attachments;

    # get lock state
    if ($Self->{Config}->{RequiredLock}) {
        if (!$TicketObject->LockIsTicketLocked(TicketID => $Self->{TicketID})) {
            # set owner
            $TicketObject->OwnerSet(
                TicketID => $Self->{TicketID},
                UserID => $Self->{UserID},
                NewUserID => $Self->{UserID},
            );
            # set lock
            if ($TicketObject->LockSet(
                TicketID => $Self->{TicketID},
                Lock => 'lock',
                UserID => $Self->{UserID}
            )) {
                # show lock state
                $LayoutObject->Block(
                    Name => 'TicketLocked',
                    Data => {
                        %Param,
                        TicketID => $Self->{TicketID},
                    },
                );
            }
        }
        else {
            my $AccessOk = $TicketObject->OwnerCheck(
                TicketID => $Self->{TicketID},
                OwnerID => $Self->{UserID},
            );
            if (!$AccessOk) {
                my $Output = $LayoutObject->Header();
                $Output .= $LayoutObject->Warning(
                    Message => "Sorry, you need to be the owner to do this action!",
                    Comment => 'Please change the owner first.'
                );
                $Output .= $LayoutObject->Footer();
                return $Output;
            }
        }
    }

    my $Output;
    if ($Self->{Subaction} eq "DelegateTicket") {
        $Output = $Self->Delegate(%Param);
    }
    else {
        $Output = $Self->Mask(%Param);
    }

    return $Output;
}

sub Delegate ()
{
    my $Self = shift;
    my %Param = @_;

    my $TicketObject = $Self->{TicketObject};
    my $LayoutObject = $Self->{LayoutObject};

    foreach (qw(To Subject Body TicketID)) {
        if (!$Param{$_}) {
            return $Self->Mask(
                %Param,
                Error => "You need to provide a valid and non empty ".$_."!"
            );
        }
    }

    if ($Self->{ConfigObject}->Get('Ticket::Frontend::AccountTime')) {
        if ($Param{TimeUnits} !~ /^\d+$/o) {
            return $Self->Mask(
                %Param,
                Error => "You need to provide a valid and non empty account time!"
            );
        }
    }

    my %StateData = $TicketObject->{StateObject}->StateGet(
        ID => $Param{ComposeStateID}
    );
    my $NextState = $StateData{Name};

    # check pending date
    if ($StateData{TypeName} && $StateData{TypeName} =~ /^pending/i) {
        if (!$Self->{TimeObject}->Date2SystemTime(%Param, Second => 0)) {
            return $Self->Mask(
                %Param,
                Error => "Invalid date!"
            );
        }
    }

    my %Address = $Self->{AddressObject}->AddressList(
        Name => $Param{To},
        Valid => 1
    );

    if (!exists $Address{$Param{To}}) {
        return $Self->Mask(
            %Param,
            Error => "Unknown OTRS instance: ".$Param{To}
        );
    }

    %Address = %{$Address{$Param{To}}};

    my @Attachments = ();
    my @AttachmentFilenames = ();
    foreach my $ID(keys %{$Param{Attachments}}) {
        my $Key = "Attachment".$ID;
        if ($Param{$Key}) {
            my %Attachment = $TicketObject->ArticleAttachment(
                ArticleID => $Param{Attachments}->{$ID}->{ArticleID},
                FileID => $Param{Attachments}->{$ID}->{AttachmentID},
                UserID => $Self->{UserID}
            );
            push @Attachments, \%Attachment;
            push @AttachmentFilenames, $Attachment{Filename};
        }
    }

    my %QueueAddress = $Self->{QueueObject}->GetSystemAddress(
        QueueID => $Param{Ticket}->{QueueID}
    );

    my $Result = $Self->{EmailObject}->Send(
        From => $QueueAddress{Email},
        To => $Address{Address},
        Subject => $Param{Subject},
        Body => $Param{Body},
        Charset => "iso-8859-15",
        Attachment => \@Attachments
    );

    if (!$Result) {
        return $Self->Mask(
            %Param,
            Error => "Can not delegate ticket to ".$Param{To}
        );
    }

    my $ArticleID = $TicketObject->ArticleCreate(
        TicketID => $Param{TicketID},
        ArticleType => "email-external",
        SenderType => "agent",
        From => $QueueAddress{Email},
        To => $Address{Address},
        Subject => $Param{Subject},
        Body => $Param{Body},
        HistoryType => "Forward",
        HistoryComment => "Delegation to ".$Param{To}."\%\%".join(",", @AttachmentFilenames),
        UserID => $Self->{UserID}
    );

    # set state
    $TicketObject->StateSet(
        TicketID => $Self->{TicketID},
        ArticleID => $ArticleID,
        State => $NextState,
        UserID => $Self->{UserID},
    );
    # should I set an unlock?
    if ($StateData{TypeName} =~ /^close/i) {
        $TicketObject->LockSet(
            TicketID => $Self->{TicketID},
            Lock => "unlock",
            UserID => $Self->{UserID},
        );
    }
    # set pending time
    elsif ($StateData{TypeName} =~ /^pending/i) {
        $TicketObject->TicketPendingTimeSet(
            UserID => $Self->{UserID},
            TicketID => $Self->{TicketID},
            Year => $Param{Year},
            Month => $Param{Month},
            Day => $Param{Day},
            Hour => $Param{Hour},
            Minute => $Param{Minute},
        );
    }

    if ($StateData{TypeName} =~ /^close/i) {
        return $Self->{LayoutObject}->Redirect(
            OP => $Self->{LastScreenOverview}
        );
    }
    else {
        return $Self->{LayoutObject}->Redirect(
            OP => "Action=AgentTicketZoom&TicketID=$Self->{TicketID}&ArticleID=$ArticleID"
        );
    }
}

sub Mask ()
{
    my $Self = shift;
    my %Param = @_;

    my $TicketObject = $Self->{TicketObject};
    my $LayoutObject = $Self->{LayoutObject};

    my $Ticket = $Param{Ticket};
    my $Articles = $Param{Articles};
    my $Attachments = $Param{Attachments};

    if (!$Self->{Subaction}) {
        foreach my $Article(sort { $a->{IncomingTime} <=> $b->{IncomingTime} } @$Articles) {
            $Param{Body} .= $Article->{From}.
                " wrote on ".
                $Self->{TimeObject}->SystemTime2TimeStamp(
                    SystemTime => $Article->{IncomingTime}
                ).
                ":\n\n".
                $Article->{Body}.
                "\n\n-------------------------------------------------\n";
        }

        $Param{Subject} = $Ticket->{Title};
    }

    # show time accounting box
    if ($Self->{ConfigObject}->Get('Ticket::Frontend::AccountTime')) {
        $LayoutObject->Block(
            Name => 'TimeUnitsJs',
            Data => \%Param,
        );
        $LayoutObject->Block(
            Name => 'TimeUnits',
            Data => \%Param,
        );
    }
    # show spell check
    if ($Self->{ConfigObject}->Get('SpellChecker') && $LayoutObject->{BrowserJavaScriptSupport}) {
        $LayoutObject->Block(
            Name => 'SpellCheck',
            Data => {},
        );
    }

    # get next states
    my %NextStates = $TicketObject->StateList(
        Action => $Self->{Action},
        TicketID => $Self->{TicketID},
        UserID => $Self->{UserID},
    );

    # build next states string
    my %State = ();
    if (!$Param{ComposeStateID}) {
        $State{Selected} = $Self->{Config}->{StateDefault};
    }
    else {
        $State{SelectedID} = $Param{ComposeStateID};
    }

    $Param{NextStatesStrg} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => \%NextStates,
        Name => 'ComposeStateID',
        \%State
    );

    # pending data string
    $Param{PendingDateString} = $LayoutObject->BuildDateSelection(
        %Param,
        Format => 'DateInputFormatLong',
        DiffTime => $Self->{ConfigObject}->Get('Ticket::Frontend::PendingDiffTime') || 0,
    );

    my %Addresses = $Self->{AddressObject}->AddressList(Valid => 1);
    foreach my $Address(keys %Addresses) {
        my $BlockName = "ToEntry";
        $LayoutObject->Block(
            Name => $BlockName,
            Data => {}
        );
        $BlockName .= ($Param{To} eq $Address ? "Selected" : "Deselected");
        $LayoutObject->Block(
            Name => $BlockName,
            Data => {
                To => $Address
            }
        );
    }

    foreach my $AttachmentID(keys %$Attachments) {
        my $BlockName = "Attachment";
        $LayoutObject->Block(
            Name => $BlockName,
            Data => $Attachments->{$AttachmentID}
        );
        $BlockName .= ($Param{"Attachment".$AttachmentID} ? "Selected" : "Deselected");
        $LayoutObject->Block(
            Name => $BlockName,
            Data => {
                AttachmentID => $AttachmentID
            }
        );
    }

    my $Output = $LayoutObject->Header(Value => $Ticket->{TicketNumber});
    if ($Param{Error}) {
        $Output .= $LayoutObject->Notify(
            Priority => "Error",
            Info => $Param{Error}
        );
    }
    $Output .= $LayoutObject->Output(
        TemplateFile => "AgentTicketDelegate",
        Data => {
            TicketNumber => $Ticket->{TicketNumber},
            %Param
        }
    );
    $Output .= $LayoutObject->Footer();

    return $Output;
}

1;
