#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <malloc.h>
#include <sys/stat.h>

#define STDIN 0
#define STDOUT 1
#define STDERR 2
#define BUFFER_SIZE 4096 * 4096

void die (char *s1, char *s2)
{
  write(STDERR, s1, strlen(s1));
  write(STDERR, ": ", 2);
  write(STDERR, s2, strlen(s2));
  write(STDERR, "\n", 1);
  exit(1);
}

int openfile_or_die (char *fn, int flags, int prot)
{
  int ret;

  if (prot) ret = open(fn, flags, prot);
  else ret = open(fn, flags, prot);

  if (ret < 0) die("open()", strerror(errno));

  return ret;
}

void closefile_or_die (int fd)
{
  if (close(fd)) die("close()", strerror(errno));
}

void mem2fd (int fd, void *p, long len)
{
  if (write(fd, p, len) != len) die("write()", strerror(errno));
  free(p);
}

void *fd2mem (int fd, long *len)
{
  void *ret;
  size_t i;

  for (ret = NULL, *len = 0;;) {
    ret = realloc(ret, *len + BUFFER_SIZE);
    if (!ret) die("realloc()", strerror(errno));
    i = read(fd, ret, BUFFER_SIZE);
    if (i < 0) die("read()", strerror(errno));
    if (!i) return ret;
    *len += i;
  }
}
    
void *file2mem (char *fn, long *len, int *prot)
{
  struct stat st;
  void *ret;
  int fd;

  fd = openfile_or_die(fn, O_RDONLY, *prot);
  if (fstat(fd, &st)) die("fstat()", strerror(errno));
  ret = fd2mem(fd, len);
  closefile_or_die(fd);
  *prot = st.st_mode & (S_IRWXU | S_IRWXG | S_IRWXO);
  return ret;
}

void mem2file (char *fn, void *p, long len, int prot)
{
  int fd;

  fd = openfile_or_die(fn, O_WRONLY | O_CREAT | O_TRUNC, prot);
  mem2fd(fd, p, len);
  closefile_or_die(fd);
}

void d2u (char *p, long *len)
{
  long i;
  char *c;

  for (c = p, i = *len; i; ++c, --i)
    if (*c == '\r') {
      memmove(c, c + 1, *len - (c - p));
      --(*len);
    }
}

int main (int argc, char **argv)
{
  void *p;
  long len;
  int prot = 0;

  switch (argc) {
    /* stdin -> stdout */
    case 1:
      p = fd2mem(STDIN, &len);
      d2u(p, &len);
      mem2fd(STDOUT, p, len);
      break;

    /* file1 -> file1 */
    case 2:
    /* file1 -> file2 */
    case 3:
      p = file2mem(argv[1], &len, &prot);
      d2u(p, &len);
      mem2file(argv[argc-1], p, len, prot);
      break;

    default:
      die("usage",
"d2u - converts sequences of CR/LF (\"\\r\\n\") to LF (\"\\n\")\n\n"
"examples:\n"
"d2u dos2unix.txt                      - converts the file dos2unix.txt\n"
"d2u dos.txt unix.txt                  - reads dos.txt, writes to unix.txt\n"
"cat dos.txt | d2u > unix.txt          - same as above\n"
"d2u < dos.txt                         - reads dos.txt, writes to stdout\n"
"d2u > unit.txt                        - reads stdin, writes to unix.txt\n"
"cat dos2unix.txt | d2u > dos2unix.txt - dont try!\n" );
  }

  return 0;
}
