#!/usr/bin/perl -w

# $Rev: 69 $

# modules
use strict;
use warnings;
use CGI;
use POSIX ":sys_wait_h";
use Time::Local;
use XML::Parser;
use URI::Escape;
use Socket;
use IO::Socket::INET6;
use Sys::Hostname;
use FindBin;

# constants
my $PORT = 8080;
my $HOME_DIR = glob("~");
my $ARCHIVE_DIR = $HOME_DIR . "/Documents/EyeTV Archive";
my $MOVIES_DIR = $HOME_DIR . "/Movies";
my $MUSIC_DIR = $HOME_DIR . "/Music";
my $CHANNELLIST = "/Library/Application Support/EyeTV/Shared/ChannelList.xml";

my @WEEKDAYS = ("Sunday", "Monday", "Tuesday", "Wednesday", "Thursday",
                "Friday", "Saturday", "Sunday");

my %HTTP_CODES = (
	200 => "OK",
	206 => "Partial Content",
	400 => "Bad Request",
	403 => "Forbidden",
	404 => "Not Found",
	405 => "Method Not Allowed",
	416 => "Requested range not satisfiable",
);

my %MIME_TYPES = (
	"mp3"	=> "audio/mpeg",
	"mp2"	=> "audio/mpeg",
	"m3u"	=> "audio/x-mpegurl",
	"mp4"	=> "video/mp4",
	"mpg"	=> "video/mpeg",
	"mpeg"	=> "video/mpeg",
	"avi"	=> "video/x-msvideo",
	"wmv"	=> "video/x-ms-wmv",
);

my $FAVICON = pack("H*", "000001000100101010000100040028010000160000002800" .
                         "000010000000200000000100040000000000000000000000" .
                         "0000000000000000000000000000000000001c1c1c003333" .
                         "3300737373008080800098989800dddddd00e6e6e600ffff" .
                         "ff0000000000000000000000000000000000000000000000" .
                         "000000000000000000000000000000000000000000000000" .
                         "000000000000000000000000000000153000000351000058" .
                         "400000048500007840000004870000584000000485000015" .
                         "300000035100000000000000000000000000000000006666" .
                         "666666666666222222222222222222222222222222220000" .
                         "0000000000000000000000000000ffff0000ffff00000000" .
                         "000000000000000000000000000000000000000000000000" .
                         "00000000000000000000000000000000000000000000ffff" .
                         "0000ffff0000");

my $IDENT = '$Rev: 69 $';
$IDENT =~ s/[^\d\.]//g;
$IDENT = "augtvd.pl/" . $IDENT;

# display helpers
sub display_value_length($) {
	our $cgi;
	my $sec = $_[0] % 60;
	my $tmp = $_[0] / 60;
	my $min = $tmp % 60;
	my $hour = $tmp /= 60;
	return $cgi->escapeHTML(join(":", map { sprintf("%02d", $_) }
	                                  ($hour, $min, $sec)));
}

sub display_value_size($) {
	our $cgi;
	my $size = $_[0] * 100;
	my @suffixes = qw(B kB MB GB TB);
	my $i;
	for ($i = 0; ($i <= $#suffixes) && ($size >= 102400); $i++) {
		$size /= 1024;
	}
	return $cgi->escapeHTML(sprintf("%.2f %s", $size / 100,
	                                $suffixes[$i]));
}

sub compare_string($$$) {
	my ($a, $b, $field) = @_;
	my $a_val = $a->{$field} || "";
	my $b_val = $b->{$field} || "";

	return (lc($a_val) cmp lc($b_val));
}

sub compare_integer($$$) {
	my ($a, $b, $field) = @_;
	my $a_val = $a->{$field} || 0;
	my $b_val = $b->{$field} || 0;

	return ($a_val <=> $b_val);
}

# fields shown in html overview
my @FIELDS = ({
	name => "id",
	display_header => "Delete",
	style => "zeilem",
	plist => {
		0 => "id",
		1 => "id",
	},
	display_value => sub {
		our $cgi;
		if (defined $_[0]->{id}) {
			return "<a href=\"?action=delete" .
				($_[1] == 0 ? "movie" : "schedule") . "&id=" .
				$cgi->escapeHTML($_[0]->{id}) . "\">X</a>";
		} else {
			return "";
		}
	},
}, {
	name => "title",
	display_header => "Title",
	sort_func => sub {
		return compare_string($a, $b, "title");
	},
	style => "zeilen",
	plist => {
		0 => "display title",
		1 => "display title",
	},
	display_value => sub {
		our $cgi;
		my $html = "";
		if (defined $_[0]->{mpgfile}) {
			$html .= "<a href=\"/" . $_[0]->{url} . "/" .
				$cgi->escapeHTML($_[0]->{mpgfile}) .
				"\">";
		}
		if (defined $_[0]->{title}) {
			$html .= $cgi->escapeHTML($_[0]->{title});
		}
		if (defined $_[0]->{mpgfile}) {
			$html .= "</a>";
		}
		return $html;
	},
}, {
	name => "start",
	display_header => "Start",
	sort_func => sub {
		return compare_string($a, $b, "start");
	},
	style => "zeiler",
	plist => {
		0 => "actual start time",
		1 => "start",
	},
	display_value => sub {
		our $cgi;
		# 2010-05-09T01:55:00Z
		if (!defined $_[0]->{start}) {
			return "";
		} elsif ($_[0]->{start} =~ /^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})Z$/) {
			my ($sec, $min, $hour, $day, $mon, $year, @rest) =
				localtime(timegm($6, $5, $4, $3, $2 - 1,
				                 $1 - 1900));
			$mon++;
			$year += 1900;
			foreach ($sec, $min, $hour, $day, $mon) {
				$_ = sprintf("%02d", $_);
			}
			return $cgi->escapeHTML("$year/$mon/$day $hour:$min");
		} else {
			return $cgi->escapeHTML($_[0]->{start});
		}
	},
}, {
	name => "length",
	display_header => "Length",
	sort_func => sub {
		return compare_integer($a, $b, "length");
	},
	style => "zeiler",
	plist => {
		0 => "actual duration",
		1 => "duration",
	},
	display_value => sub {
		if (defined $_[0]->{length}) {
			return display_value_length($_[0]->{length});
		} else {
			return "";
		}
	},
}, {
	name => "size",
	display_header => "Size",
	sort_func => sub {
		return compare_integer($a, $b, "size");
	},
	style => "zeiler",
	plist => {
		0 => undef,
	},
	display_value => sub {
		if (defined $_[0]->{size}) {
			return display_value_size($_[0]->{size});
		} else {
			return "";
		}
	},
}, {
	name => "repeat",
	display_header => "Repeat",
	sort_func => sub {
		return compare_string($a, $b, "repeat");
	},
	style => "zeile",
	plist => {
		1 => "repeats",
	},
	display_value => sub {
		our $cgi;
		if (defined $_[0]->{repeat}) {
			return $cgi->escapeHTML($_[0]->{repeat});
		} else {
			return "";
		}
	},
}, {
	name => "channel",
	display_header => "Channel",
	sort_func => sub {
		return compare_string($a, $b, "channel");
	},
	style => "zeilen",
	plist => {
		0 => "channel name",
		1 => "station name",
	},
	display_value => sub {
		our $cgi;
		if (defined $_[0]->{channel}) {
			return $cgi->escapeHTML($_[0]->{channel});
		} else {
			return "";
		}
	},
}, {
	name => "description",
	display_header => "Description",
	sort_func => sub {
		return compare_string($a, $b, "description");
	},
	style => "zeile",
	plist => {
		0 => "description",
		1 => "description",
	},
	display_value => sub {
		our $cgi;
		if (defined $_[0]->{description}) {
			return $cgi->escapeHTML($_[0]->{description});
		} else {
			return "";
		}
	},
});

# subroutines
sub get_channels() {
	my %channels = ();
	my $string = "";
	my $found_channel = 0;
	my $found_number = 0;
	my $channel;
	my $number;
	my $parser = XML::Parser->new(Handlers => {
		End => sub {
			my $element = $_[1];
			if ($element eq "string") {
				$channel = $string if $found_channel;
				$number = $string if $found_number;
				if (defined $channel && defined $number) {
					$channels{$channel} = $number;
					$channel = undef;
					$number = undef;
				}
			} elsif ($element eq "key") {
				$found_channel = ($string eq "channel name");
				$found_number = ($string eq "display number");
			}
		},
		Char => sub {
			$string = $_[1];
		},
	});
	eval {
		$parser->parsefile($CHANNELLIST);
	};
	fatal("cannot parse channelist: $@") if $@;
	return \%channels;
}

sub get_programs() {
	my $raw = osascript("tell application \"EyeTV\"\n" .
		"set progno to number of programs\n" .
		"set output to \"\"\n" .
		"repeat while progno > 0\n" .
		"set {" .
			"start time:starttime, " .
			"duration:durationseconds, " .
			"title:thetitle, " .
			"description:desc, " .
			"channel number:channelnumber, " .
			"station name:stationname, " .
			"input source:inputsource, " .
			"repeats:repeatperiod, " .
			"quality:qualitylevel, " .
			"enabled:isenabled, " .
			"unique ID:uid " .
		"} to program progno\n" .
		"set {" .
			"year:syear, " .
			"month:smonth, " .
			"day:sday, " .
			"hours:shour, " .
			"minutes:sminute, " .
			"seconds:ssecond" .
		"} to starttime\n" .
		"set output to output & " .
			"progno & (ASCII character 9) & " .
			"syear & (ASCII character 9) & " .
			"(smonth as integer) & (ASCII character 9) & " .
			"sday & (ASCII character 9) & " .
			"shour & (ASCII character 9) & " .
			"sminute & (ASCII character 9) & " .
			"ssecond & (ASCII character 9) & " .
			"durationseconds & (ASCII character 9) & " .
			"thetitle & (ASCII character 9) & " .
			"desc & (ASCII character 9) & " .
			"channelnumber & (ASCII character 9) & " .
			"stationname & (ASCII character 9) & " .
			"inputsource & (ASCII character 9) & " .
			"repeatperiod & (ASCII character 9) & " .
			"qualitylevel & (ASCII character 9) & " .
			"isenabled & (ASCII character 9) & " .
			"uid & (ASCII character 10)\n" .
		"set progno to progno - 1\n" .
		"end repeat\n" .
		"get output\n" .
	"end tell\n");

	my @programs = ();
	foreach my $line(@$raw) {
		chomp($line);
		next if !$line;
		my @fields = split(/\t/, $line);
		format_datetime(\@fields);
		$fields[7] =~ s/\,/\./g;
		$fields[7] = int($fields[7]);
		$fields[16] = int($fields[16]);
		my ($starttime, $endtime) = start_endtime((@fields)[1..7]);
		push @fields, ($starttime, $endtime);
		my $repeat = $fields[13];
		if ($repeat eq "none") {
			push @programs, \@fields;
		} else {
			my $repeats = build_repeats(\@fields);
			for (my $day = -1; $day < 60; $day++) {
				repeat_program(\@programs, \@fields, $repeats,
				               $day);
			}
		}
	}

	return \@programs;
}

sub repeat_program($$$$) {
	my ($programs, $fields, $repeats, $day) = @_;
	my @tmp = @$fields;
	$tmp[17] += $day * 86400;
	$tmp[18] += $day * 86400;
	my $dayofweek;
	($tmp[6], $tmp[5], $tmp[4], $tmp[3], $tmp[2], $tmp[1], $dayofweek) =
							localtime($tmp[17]);

	if (exists $repeats->{$dayofweek}) {
		$tmp[1] += 1900;
		$tmp[2]++;
		format_datetime(\@tmp);
		push @$programs, \@tmp;
	}
}

sub build_repeats($) {
	my ($fields) = @_;

	my $repeat = $fields->[13];
	my %repeat = ();
	if ($repeat eq "daily") {
		for (my $weekday = 0; $weekday < 8; $weekday++) {
			$repeat{$weekday} = undef;
		}
	} elsif ($repeat eq "weekdays") {
		for (my $weekday = 1; $weekday < 6; $weekday++) {
			$repeat{$weekday} = undef;
		}
	} elsif ($repeat eq "weekends") {
		foreach my $weekday(qw(0 6 7)) {
			$repeat{$weekday} = undef;
		}
	} else {
		for (my $weekday = 0; $weekday <= $#WEEKDAYS; $weekday++) {
			if ($repeat =~ /$WEEKDAYS[$weekday]/) {
				$repeat{$weekday} = undef;
			}
		}
	}

	return \%repeat;
}

sub format_datetime($) {
	my ($fields) = @_;
	for (my $i = 2; $i <= 6; $i++) {
		$fields->[$i] = sprintf("%02d", $fields->[$i]);
	}
}

sub format_programs($) {
    my ($programs) = @_;
    return join("\n", map { join("\t", @$_) } sort {
            $a->[1] <=> $b->[1] ||
            $a->[2] <=> $b->[2] ||
            $a->[3] <=> $b->[3] ||
            $a->[4] <=> $b->[4] ||
            $a->[5] <=> $b->[5] ||
            $a->[6] <=> $b->[6] ||
            $a->[8] cmp $b->[8]
    } @$programs);
}

sub datetime_ampm($$$$$$) {
	my $hour = $_[3] % 12;
	$hour = 12 if !$hour;
	my $ampm = $_[3] < 12 ? "AM" : "PM";
#	return "$_[1]/$_[2]/$_[0] $hour:$_[4]:$_[5] $ampm";
	return "$_[2]/$_[1]/$_[0] $_[3]:$_[4]:$_[5]";
}

sub start_endtime($$$$$$$) {
	my $starttime = timelocal($_[5], $_[4], $_[3], $_[2], $_[1] - 1,
	                          $_[0] - 1900);
	my $endtime = $starttime + $_[6];
	return ($starttime, $endtime);
}

sub osascript($) {
	my ($command) = @_;

	my ($RHCGI, $RHAS, $WHCGI, $WHAS);
	pipe($RHCGI, $WHAS) || fatal("pipe(): $!");
	pipe($RHAS, $WHCGI) || fatal("pipe(): $!");

	my $pid = fork();
	fatal("fork(): $!") if ($pid < 0);

	if (!$pid) {
		close($RHCGI);
		close($WHCGI);
		open(STDIN, ">&", $RHAS);
		open(STDOUT, ">&", $WHAS);
		open(STDERR, ">&", $WHAS);
		exec("/usr/bin/osascript");
		die $!;
	}

	close($RHAS);
	close($WHAS);

	print $WHCGI $command;
	close($WHCGI);

	my @output = <$RHCGI>;
	do {
		$pid = waitpid(-1, WNOHANG);
		fatal("osascript ($pid) returned $?: " .
		      join(" ", @output)) if (($pid > 0) && $?);
	} while ($pid > 0);

	return \@output;
}

sub http_date() {
	my @now = gmtime();
	my @MONTHS = qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
	return sprintf("%s, %02d %s %04d %02d:%02d:%02d",
	               substr($WEEKDAYS[$now[6]], 0, 3), $now[3],
	               $MONTHS[$now[4]], $now[5] + 1900,
	               $now[2], $now[1], $now[0]);
}

sub logger($) {
	print "[", http_date(), "] pid ", sprintf("% 6d", $$), ": ",
	      $_[0], "\n";
}

sub output_generic($$$$$) {
	my ($status, $content_type, $filename, $extra_header, $data) = @_;
	our $client;
	our $http_version;
	my $header = "HTTP/1.0 $status " . $HTTP_CODES{$status} . "\r\n" .
	             "Date: " . http_date() . "\r\n" .
	             "Server: " . $IDENT . "\r\n" .
	             "Accept-Ranges: bytes\r\n" .
	             "Content-Type: $content_type\r\n";
	if (defined $data) {
		$header .= "Content-Length: " . length($data) . "\r\n";
	}
	if (defined $filename) {
		$header .= "Content-Disposition: attachment; " .
		           "filename=\"$filename\"\r\n";
	}
	$header .= $extra_header . "\r\n";

	if ($http_version >= 1.0) {
		print $client $header;
		print $client $data if defined $data;
	} elsif ($content_type eq "text/html") {
		print $client $data if defined $data;
	} else {
		print $client "<html><body>\n" .
		              "<p>Your browser is too old.</p>\n" .
		              "</body></html>";
	}
}

sub output_m3u($$) {
	my ($filename, $movies) = @_;
	our $server_address;

	my $m3u = "#EXTM3U\n";
	foreach (@$movies) {
		next if !defined $_->{mpgfile};
		$m3u .= "#EXTINF:";
		if (defined $_->{length}) {
			$m3u .= int($_->{length});
		} else {
			$m3u .= "-1";
		}
		$m3u .= ",";
		if (defined $_->{title}) {
			$m3u .= $_->{title};
		} else {
			$m3u .= $_->{mpgfile};
		}
		$m3u .= "\n" . $server_address .
		        URI::Escape::uri_escape($_->{url} . "/" . $_->{mpgfile},
			                        '^A-Za-z0-9-._~/') . "\n";
	}

	output_generic(200, "application/x-mpegurl", $filename . ".m3u", "", $m3u);
}

sub output_html($) {
	output_generic(200, "text/html", undef, "", $_[0]);
}

sub ok($) {
	output_generic(200, "text/plain", undef, "", "OK\n" . $_[0]);
}

sub fatal($) {
	my $error = shift;

	foreach ($error) {
		s/[^\S ]//sgio;
	}

	output_generic(200, "text/plain", undef, "", "ERR\n" . $error);
	logger($error);
	exit(1);
}

sub html_error($) {
	my ($status) = @_;
	my $message = $HTTP_CODES{$status};
	output_generic($status, "text/html", undef, "",
	               "<html><head><title>Error</title></head>\n" .
	               "<body>$message</body></html>\n");
	logger($message);
	exit(1);
}

sub start_html($@) {
	my ($subtitle, $location, $field_index, $sort_order) = @_;
	if ($location) {
		$location = "<meta http-equiv=\"refresh\" content=\"0; URL=" .
		            $location . "\">\n";
	} else {
		$location = "";
	}
	my $html = <<EOF;
<html>
<head>
<title>AugTV</title>$location
<style type="text/css">
<!--
body {
  background-color:white;
  color:black;
  font-family:Arial, sans-serif;
  font-size:12px;
}
table {
  background-color:orange;
  border:none solid black;
  border-spacing:2px;
}
th, th a {
  background-color:#777;
  color:white;
  font-size:12px;
  font-weight:bold;
  text-decoration:none;
  white-space:nowrap;
}
td {
  padding:2px;
  font-size:12px;
}
td a {
  color:black;
}
td.zeile0, td.zeiler0, td.zeilen0, td.zeilem0 {
  background-color:#CCC;
}
td.zeile1, td.zeiler1, td.zeilen1, td.zeilem1 {
  background-color:#EEE;
}
td.zeiler0, td.zeiler1 {
  text-align:right;
}
td.zeilem0, td.zeilem1 {
  text-align:center;
}
td.zeilen0, td.zeilen1, td.zeiler0, td.zeiler1 {
  white-space:nowrap;
}
//-->
</style>
</head>
<body>
<h1>AugTV</h1>
<p>[<a href="?">Movies</a>] [<a href="?action=schedule">Schedules</a>]
EOF

	if (defined $field_index && defined $sort_order) {
		$html .= <<EOF;
<br>Playlists:
[<a href="?action=m3umovie&field_index=$field_index&sort_order=$sort_order">Movies</a>]
[<a href="?action=m3umusic">Music</a>]
EOF
	}

	$html .= <<EOF;
</p>
<h2>$subtitle</h2>
EOF

	return $html;
}

sub end_html() {
	return "</body>\n</html>";
}

sub matches_one($@) {
	my ($filename, @extensions) = @_;
	my $idx = rindex($filename, ".");
	return 0 if ($idx < 0);
	my $ext = lc(substr($filename, $idx + 1));

	foreach (@extensions) {
		return 1 if (lc($_) eq $ext);
	}

	return 0;
}

sub traverse_dir($$$@);
sub traverse_dir($$$@) {
	my ($dir, $reldir, $url, @extensions) = @_;

	opendir(DH, $dir) || return [];
	my @entries = readdir(DH);
	closedir(DH);

	my @files;
	foreach my $entry(sort @entries) {
		my $absfile = $dir . "/" . $entry;
		my $relfile = $reldir . ($reldir ? "/" : "") . $entry;
		if (-d $absfile && ($entry ne ".") && ($entry ne "..")) {
			push @files, @{traverse_dir($absfile, $relfile, $url,
			                             @extensions)};
		}
		if (-f $absfile && matches_one($entry, @extensions)) {
			my @stat = stat($absfile);
			my ($sec, $min, $hour, $day, $mon, $year) =
							gmtime($stat[9]);
			my $start = sprintf("%04d-%02d-%02dT%02d:%02d:%02dZ",
			                    $year + 1900, $mon + 1, $day,
			                    $hour, $min, $sec);
			push @files, {
				url     => $url,
				mpgfile => $relfile,
				size    => $stat[7],
				start   => $start,
				title   => substr($entry, 0,
				                  rindex($entry, ".")),
			};
		}
	}

	return \@files;
}

sub get_field($$) {
	my ($value, $type) = @_;

	foreach my $field(@FIELDS) {
		if ($field->{plist}->{$type} &&
		    ($value eq $field->{plist}->{$type})) {
			return $field->{name};
		}
	}

	return undef;
}

sub parse_movie($$$) {
	my ($movie, $file, $type) = @_;
	my $field;
        my $value = "";

	my $xml = XML::Parser->new(Handlers => {
                Start => sub {
                        $value = "";
                },
		End => sub {
			my $element = $_[1];
			$movie->{$field} = $value if $field;
			$field = undef;
			if ($element eq "key") {
				$field = get_field($value, $type);
			}
                        $value = "";
		},
		Char => sub {
			$value .= $_[1];
		},
	});
	$xml->parsefile($file);
}

sub scan_moviedir($$$) {
	my ($movie, $dir, $type) = @_;
	my $extension = ($type == 0 ? "eyetvr" : "eyetvp");

	opendir(DH, "$ARCHIVE_DIR/$dir");
	my @entries = readdir(DH);
	closedir(DH);

	foreach (@entries) {
		my $file = "$ARCHIVE_DIR/$dir/$_";
		if (-f $file) {
			parse_movie($movie, $file, $type) if /\.$extension$/;
			if (/\.mpg$/) {
				$movie->{mpgfile} = "$dir/$_";
				$movie->{size} = -s $file;
				$movie->{url} = "eyetv";
			}
		}
	}
}

sub scan_archivedir($) {
	my ($type) = @_;
	my $extension = ($type == 0 ? "eyetv" : "eyetvsched");

	opendir(DH, $ARCHIVE_DIR) || return [];
	my @entries = readdir(DH);
	closedir(DH);

	my @movies = ();
	foreach (@entries) {
		my $dir = "$ARCHIVE_DIR/$_";
		if (-d $dir && !/^Live TV Buffer/) {
			my %movie = ();
			if (/\.$extension$/) {
				scan_moviedir(\%movie, $_, $type);
				push @movies, \%movie;
			}
		}
	}

	return \@movies;
}

sub get_allmovies($) {
	my ($type) = @_;

	my $movies = scan_archivedir($type);
	push @$movies, @{traverse_dir($MOVIES_DIR, "", "movies",
	                              qw(avi mpg mpeg mkv m4v asf wmv mov))};

	return $movies;
}

sub ip_to_string($) {
	my ($error, $hostname) = Socket::getnameinfo($_[0],
	                                             Socket::NI_NUMERICHOST,
	                                             Socket::NIx_NOSERV);
	if ($hostname =~ /:/) {
		return "[$hostname]";
	}

	return $hostname;
}

# action_* routines are called from the client's main event loop
sub action_record() {
	our $cgi;

	my $programs = get_programs();
	my $station = $cgi->param("station") || fatal("need station name");
	my $title = $cgi->param("title") || fatal("need title");
	my $description = $cgi->param("description") || "";
	my $duration = $cgi->param("duration") || fatal("need duration");
	fatal("invalid duration") if ($duration !~ /^\d+$/);
	my $year = $cgi->param("year") || fatal("need year");
	my $month = $cgi->param("month") || fatal("need month");
	my $day = $cgi->param("day") || fatal("need day");
	my $hour = $cgi->param("hour");
	fatal("need hour") if !defined $hour;
	my $minute = $cgi->param("minute");
	fatal("need minute") if !defined $minute;
	my $repeats = $cgi->param("repeats");
	fatal("need repeats") if !defined $repeats;
	my %possible = map {
		$_ => undef
	} @WEEKDAYS, "none", "never", "daily", "weekdays", "weekends";
	fatal("invalid repeats $repeats") if !exists $possible{$repeats};

	my $channels = get_channels();
	my $channel = $channels->{$station};
	fatal("unknown station $station") if !defined $channel;

	my ($starttime, $endtime) = start_endtime($year, $month, $day, $hour,
	                                          $minute, 0, $duration);
	foreach my $program(@$programs) {
		my $starttime0 = $program->[17];
		my $endtime0 = $program->[18];
		my $enabled = $program->[15];
		if (($starttime < $endtime0) && ($endtime > $starttime0) &&
		    ($enabled ne "false")) {
			fatal("overlaps with " . $program->[8]. " at " .
			      $program->[4] . ":" . $program->[5] . " on " .
			      $program->[11]);
		}
	}

	foreach ($title, $description) {
		s//Ae/sgo;
		s//Oe/sgo;
		s//Ue/sgo;
		s//ae/sgo;
		s//oe/sgo;
		s//ue/sgo;
		s//ss/sgo;
		s/[^A-Za-z0-9\,\.\-\;\:\_\<\>\&\(\)=\?\+\*\#\!\ ]/_/sgio;
	}

	my $uid = osascript("tell application \"EyeTV\"\n" .
	                    "make new program with properties {" .
	                    "title:\"$title\", " .
	                    "repeats:$repeats, " .
	                    "description:\"$description\", " .
	                    "channel number:\"$channel\", " .
	                    "station name:\"$station\", " .
	                    "start time:date \"" .
	                    datetime_ampm($year, $month, $day, $hour, $minute,
	                                  0) .
	                    "\", " .
	                    "duration:$duration}\n" .
	                    "end tell\n");
	ok(join("\t", @$uid));
}

sub action_deleteprogram() {
	our $cgi;

	my $programs = get_programs();
	my $id = $cgi->param("id") || fatal("need id");
	foreach my $program(@$programs) {
		if ($id eq $program->[16]) {
			my $raw = osascript("tell application \"EyeTV\"\n" .
			                    "delete program id " .
			                    $program->[16] . "\n" .
			                    "end tell\n");
			ok(join("\t", @$raw));
			return;
		}
	}
	fatal("no such program");
}

sub action_channels() {
	our $cgi;

	my $channels = get_channels();
	ok(join("\n", map { "$_\t" . $channels->{$_} } sort {
		$channels->{$a} <=> $channels->{$b}
	} keys %$channels));
}

sub action_datetime() {
	our $cgi;

	my @datetime = localtime();
	$datetime[4]++;
	$datetime[5] += 1900;
	format_datetime(\@datetime);
	ok(join("\t", reverse((@datetime)[0..5])));
}

sub action_programs() {
	our $cgi;

	ok(format_programs(get_programs()));
}

sub action_delete_movie_or_schedule() {
	our $cgi;

	my $type = ($1 eq "movie" ? 0 : 1);
	my $type_string = ($type == 0 ? "movie" : "schedule");
	my $html = start_html("Delete " . $type_string);
	my @movies = @{scan_archivedir($type)};
	my $id = $cgi->param("id") || "";
	$html .= "<table>\n<tr>\n";
	for (my $i = 1; $i <= $#FIELDS; $i++) {
		if (exists $FIELDS[$i]->{plist}->{$type}) {
			$html .= "<th>" . $FIELDS[$i]->{display_header} .
			         "</th>\n";
		}
	}
	$html .= "</tr>\n";
	my $count = 0;
	my $zeile = 0;
	foreach my $movie(@movies) {
		next if ($id ne $movie->{id});
		$html .= "<tr>\n";
		for (my $i = 1; $i <= $#FIELDS; $i++) {
			next if !exists $FIELDS[$i]->{plist}->{$type};
			$html .= "<td class=\"" .
				$FIELDS[$i]->{style} .
				"$zeile\">" .
				&{$FIELDS[$i]->{display_value}}($movie, $type) .
				"</td>\n";
		}
		$html .= "</tr>\n";
		$zeile = 1 - $zeile;
		$count++;
	}
	$html .= "</tr>\n</table>\n<p>\n";
	my $back = "?action=" . $type_string;
	if ($count == 0) {
		$html .= "Nothing to delete!<br><a href=\"" . $back .
			"\">Back</a>\n";
	} else {
		$html .= "Delete this " . $type_string;
		$html .= "s" if ($count != 1);
		$html .= "?<br>\n<a href=\"?action=dodelete" . $type_string .
			"&id=" . $id . "\">Yes</a> <a href=\"" . $back .
			"\">No</a>\n";
	}
	$html .= "</p>\n";
	$html .= end_html();
	output_html($html);
}

sub action_dodelete_movie_or_schedule() {
	our $cgi;

	my $type = ($1 eq "movie" ? 0 : 1);
	my $type_string = ($type == 0 ? "movie" : "schedule");
	my $id = $cgi->param("id") || fatal("need id");
	osascript("tell application \"EyeTV\"\n" .
	          "delete " .
	          ($type == 0 ? "recording" : "program") .
	          " id " . $id . "\n" .
	          "end tell\n");
	my $redir = "/?action=" . $type_string;
	my $html = start_html("Delete " . $type_string, $redir);
	$html .= "<p>If redirection doesn't work, click <a href=\"" .
		$redir . "\">here</a>.</p>\n";
	$html .= end_html();
	output_html($html);
}

sub action_m3umusic() {
	our $cgi;

	my $music = traverse_dir($MUSIC_DIR, "", "music",
	                         qw(mp3 wav wma ogg mp4 mp2 m4a mod midi mid
	                            aif aiff));
	output_m3u("music", $music);
}

sub action_m3umovie() {
	our $cgi;

	my @movies = @{get_allmovies(0)};
	my $field_index = $cgi->param("field_index") || -1;
	my $sort_order = $cgi->param("sort_order") || "";

	$sort_order = "desc" if !$sort_order;
	if (($field_index < 0) || ($field_index > $#FIELDS) ||
		!defined $FIELDS[$field_index]->{sort_func}) {
		$field_index = 2;
	}

	@movies = sort { &{$FIELDS[$field_index]->{sort_func}} } @movies;
	@movies = reverse @movies if ($sort_order eq "desc");

	output_m3u("movies", \@movies);
}

sub action_default($) {
	my ($action) = @_;
	our $cgi;

	my $type = ($action eq "schedule" ? 1 : 0);
	my @movies = ($type ? @{scan_archivedir($type)} :
	                      @{get_allmovies($type)});
	my $field_index = $cgi->param("field_index") || -1;
	my $sort_order = $cgi->param("sort_order") || "";

	$sort_order = "desc" if !$sort_order;
	if (($field_index < 0) || ($field_index > $#FIELDS) ||
		!defined $FIELDS[$field_index]->{sort_func}) {
		$field_index = 2;
	}

	@movies = sort { &{$FIELDS[$field_index]->{sort_func}} } @movies;
	@movies = reverse @movies if ($sort_order eq "desc");

	my $movie_count = @movies;
	my $total_size = 0;
	my $total_length = 0;

	foreach (@movies) {
		if (defined $_->{size} && ($_->{size} > 0)) {
			$total_size += $_->{size};
		}
		if (defined $_->{length} && ($_->{length} > 0)) {
			$total_length += $_->{length};
		}
	}

	my $html = start_html(($type == 0 ? "Movies" : "Schedules"), undef,
	                      $field_index, $sort_order);
	$html .= "<p>" . $movie_count . " " .
		($type == 0 ? "movie" : "schedule") .
		($movie_count == 1 ? "" : "s") .
		", " . display_value_length($total_length) . " play time";
	if ($type == 0) {
		$html .= ", " .	display_value_size($total_size);
	}
	$html .= "</p>\n<table>\n<tr>\n";

	for (my $i = 0; $i <= $#FIELDS; $i++) {
		if (exists $FIELDS[$i]->{plist}->{$type}) {
			$html .= "<th>";
			my $updown;
			if (defined $FIELDS[$i]->{sort_func}) {
				$html .= "<a href=\"?action=" .
                                        ($type == 0 ? "movie" : "schedule") .
                                        "&field_index=" . $i . "&sort_order=";
				if ($i != $field_index) {
					$html .= "asc";
					$updown = "";
				} elsif ($sort_order eq "asc") {
					$html .= "desc";
					$updown = " &uarr;";
				} else {
					$html .= "asc";
					$updown = " &darr;";
				}
				$html .= "\">";
			}
			$html .= $FIELDS[$i]->{display_header};
			if (defined $FIELDS[$i]->{sort_func}) {
				$html .= $updown . "</a>";
			}
			$html .= "</th>\n";
		}
	}

	$html .= "</tr>\n";

	my $zeile = 0;
	foreach my $movie(@movies) {
		$html .= "<tr>\n";
		foreach my $field(@FIELDS) {
			next if !exists $field->{plist}->{$type};
			$html .= "<td class=\"" . $field->{style} .
				"$zeile\">" .
				&{$field->{display_value}}($movie, $type) .
				"</td>\n";
		}
		$html .= "</tr>\n";
		$zeile = 1 - $zeile;
	}

	$html .= "</tr>\n</table>\n";
	$html .= end_html();
	output_html($html);
}

sub parse_range($$) {
	my ($filesize, $range) = @_;

	return if ($range !~ s/^bytes=//i);
	my ($start, $stop) = split(/\-/, $range);
	return if (!defined $start || !defined $stop);

	if ($start =~ /^\d+$/) {
		if ($stop =~ /^\d+$/) {
			$start = int($start);
			$stop = int($stop);
		} else {
			$start = int($start);
			$stop = $filesize - 1;
		}
	} elsif ($stop =~ /^\d+$/) {
		$start = $filesize - int($stop);
		$stop = $filesize - 1;
	} else {
		return;
	}

	return if ($stop < $start);
	return if ($start < 0);
	return if ($stop >= $filesize);

	return ($start, $stop);
}

sub mime_types($$) {
	my ($extension, $mimefile) = @_;
	my $type;

	return if !open(MIMEFH, "<", $mimefile);
	while (<MIMEFH>) {
		next if /^#/;
		next if /^\s*$/;
		s/\s+$//;
		my @parts = split(/\s+/);
		my $candidate = shift @parts;
		if (grep(/$extension/, @parts)) {
			$type = $candidate;
			last;
		}
	}
	close(MIMEFH);
	return $type;
}

sub serve_file($$@) {
	my ($basedir, $url, %header) = @_;
	our $client;

	my $mime_type;
	if ($url =~ /\.([^\.]+)$/) {
		my $ext = lc($1);
		$mime_type = ($MIME_TYPES{$ext} ||
		              mime_types("/etc/apache2/mime.types", $ext) ||
		              mime_types("/etc/apache/mime.types", $ext) ||
		              mime_types("/etc/httpd2/mime.types", $ext) ||
		              mime_types("/etc/httpd/mime.types", $ext) ||
		              mime_types("/etc/mime.types", $ext));
	}
	$mime_type = "application/octet-stream" if !$mime_type;

	my $filename = $basedir . $url;
	html_error(404) if !open(VIDEOFH, "<", $filename);

	my $filesize = -s VIDEOFH;
	html_error(404) if !defined $filesize;

	my ($pos, $start, $stop);
	if (defined $header{range}) {
		logger("Range: " . $header{range});
		($start, $stop) = parse_range($filesize, $header{range});
		if (!defined $start || !defined $stop ||
		    !seek(VIDEOFH, $start, 0)) {
			html_error(416);
		}
		my $extra_header = "Content-Length: " . ($stop + 1 - $start) .
		                   "\r\n" .
		                   "Content-Range: " . $start . "-" . $stop .
		                   "/" . $filesize . "\r\n";
		output_generic(206, $mime_type, undef, $extra_header, undef);
 	} else {
		$start = 0;
		output_generic(200, $mime_type, undef,
		               "Content-Length: $filesize\r\n", undef);
 	}

	for ($pos = $start;;) {
		local $SIG{PIPE} = "IGNORE";
		my $read = (defined $stop ? $stop + 1 - $pos : 65536);
		last if ($read <= 0);
		$read = 65536 if ($read > 131072);
		my $buffer;
		$read = read(VIDEOFH, $buffer, $read);
		if (!defined $read) {
			logger("$filename: $!");
			last;
		}
		if ($read <= 0) {
			if (!defined $stop) {
				logger("$filename: unexpected end of file");
			}
			last;
		}
		if (!print $client $buffer) {
			logger($!);
			last;
		}
		$pos += $read;
	}

	logger("Served " . ($pos - $start) . " bytes");
	close(VIDEOFH);
}

### MAIN ###
# make me a server - el cheapo
my $myself = $FindBin::Bin . "/augtvd.pl";
die "$myself is not executable" if !-e $myself;
if (!$ARGV[0] || ($ARGV[0] ne "--server")) {
	exec(("screen", "-d", "-m", $myself, "--server"));
	die $!;
}

# ignore all signals
foreach (keys %SIG) {
	$SIG{$_} = "IGNORE";
}

# reap dead processes
$SIG{CHLD} = sub {
	my $pid;
	do {
		$pid = waitpid(-1, WNOHANG);
	} while ($pid > 0);
};

# shutdown routine
my $running = 1;
$SIG{TERM} = sub {
	logger("Shutting down...");
	$running = 0;
};

# shutdown any other instance
while (my $socket = IO::Socket::INET6->new(PeerAddr => '[::1]:8080')) {
	print $socket "GET /shutdown\r\n";
	close($socket);
}

# start webserver
my $http_server = IO::Socket::INET6->new(
	Proto => "tcp",
	LocalPort => $PORT,
	Listen => 8,
	ReuseAddr => 1,
	ReusePort => 1
) || die "Cannot start webserver: $!";

# resolve my own ip address
my $hostname = Sys::Hostname::hostname;
our $server_address = "http://$hostname:$PORT/";
my ($error, @addresses) = Socket::getaddrinfo($hostname);
my %unique_addresses = map { ip_to_string($_->{addr}) => undef } @addresses;
@addresses = map { "http://$_:$PORT/" } sort keys %unique_addresses;

logger($IDENT . " starting...");
foreach ($server_address, @addresses) {
	logger("Listening on $_");
}
logger("Press CTRL+C to stop, or send TERM to $$, e.g. 'kill $$'");

# main loop
my $master_pid = $$;
while ($running) {
	our ($client, $peer_addr);
	($client, $peer_addr) = $http_server->accept();
	if (!$client || !$peer_addr) {
		next if $!{EINTR};
		die "Cannot accept(): $!";
	}

	$peer_addr = ip_to_string($peer_addr);

	my $pid = fork();
	die "Cannot fork(): $!" if !defined $pid;

	if ($pid) {
		# parent returns
		$client->close();
		next;
	}
	# child
	logger("Connection from $peer_addr");
	$http_server->close();

	my $line = <$client>;
	exit(0) if !$line;

	$line =~ s/\s+$//;
	logger($line);

	my ($method, $uri, $proto_version) = split(/ /, $line, 3);
	our $http_version;
	if ($proto_version && ($proto_version =~ /\/([\d\.]+)$/)) {
		$http_version = $1;
	} else {
		$http_version = 0.9;
	}

	html_error(400) if (!$method || !$uri);
	html_error(405) if (lc($method) ne "get");

	my ($url, $query) = split(/\?/, $uri, 2);
	html_error(400) if !$url;

	# read http headers
	my %header;
	if ($http_version >= 1.0) {
		while ($line ne "") {
			$line = <$client>;
			exit(0) if !defined $line;
			$line =~ s/\s+$//;
			my ($key, $value) = split(/\s*:\s+/, $line, 2);
			if (defined $key && defined $value) {
				$header{lc($key)} = $value;
			}
		}
	}
	
	# handle urls

	my $tmp = "url=" . $url;
	$tmp .= "&" . $query if $query;
	our $cgi = CGI->new($tmp);

	$url = $cgi->param("url") || "";
	my $action = $cgi->param("action") || "";

	html_error(403) if ($url =~ /\.\./);

	if ((($peer_addr eq '[::1]') || ($peer_addr eq '[::ffff:127.0.0.1]')) &&
            ($url eq "/shutdown")) {
		kill(15, $master_pid);
	} elsif ($url =~ /^\/eyetv(\/.+)$/) {
		serve_file($ARCHIVE_DIR, $1, %header);
	} elsif ($url =~ /^\/movies(\/.+)$/) {
		serve_file($MOVIES_DIR, $1, %header);
	} elsif ($url =~ /^\/music(\/.+)$/) {
		serve_file($MUSIC_DIR, $1, %header);
	} elsif ($url eq "/favicon.ico") {
		output_generic(200, "image/x-icon", undef, "", $FAVICON);
	} elsif ($url ne "/") {
		html_error(404);
	} elsif ($action eq "record") {
		action_record();
	} elsif ($action eq "deleteprogram") {
		action_deleteprogram();
	} elsif ($action eq "channels") {
		action_channels();
	} elsif ($action eq "datetime") {
		action_datetime();
	} elsif ($action eq "programs") {
		action_programs();
	} elsif ($action =~ /^delete(movie|schedule)$/) {
		action_delete_movie_or_schedule();
	} elsif ($action =~ /^dodelete(movie|schedule)$/) {
		action_dodelete_movie_or_schedule();
	} elsif ($action eq "m3umusic") {
		action_m3umusic();
	} elsif ($action eq "m3umovie") {
		action_m3umovie();
	} else {
		action_default($action);
	}

	$client->close();
	exit(0);
}

logger("Exiting...");
