#!/usr/bin/perl -w

# $Rev: 46 $

use strict;
use warnings;
use CGI;
use POSIX ":sys_wait_h";
use Time::Local;
use XML::Parser;
use URI::Escape;

#  1. Put this script to /Library/WebServer/CGI-Executables
#
#  2. Replace fjo by your username (or the name of the user how runs EyeTV):
#
our $TVUSER = "fjo";

#  3. Make it executable:
#     chmod 0755 /Library/WebServer/CGI-Executables/augtv.pl
#
#  4. Edit /etc/sudoers by calling "sudo visudo". Add the following line:
#     _www	ALL=(TVUSER) NOPASSWD: /usr/bin/osascript
#     (replace TVUSER by the username who runs EyeTV)
#
#  5. Edit /etc/apache2/users/TVUSER.conf. Ensure it contains:
#
#     <Directory /Users/TVUSER/Sites>
#       Order allow,deny
#       Allow from all
#     </Directory>
#
#  6. Start the Apache webserver:
#     sudo launchctl load org.apache.httpd
#     sudo launchctl start org.apache.httpd
#
#  7. Close EyeTV.
#
#  8. Move your existing EyeTV archive (e.g. from inside your home folder) to
#     the "Sites" folder inside your home directory.
#
#  9. Open EyeTV -> Preferences -> Recording. Set "EyeTV Archive Location" to:
#     /Users/TVUSER/Sites
#
#  10. Launch your webbrowser and go to:
#      http://<IP_ADDR_OF_YOUR_MAC/cgi-bin/augtv.pl

# physical location of your EyeTV archive
our $ARCHIVEDIR = "/Users/" . $TVUSER . "/Sites/EyeTV Archive";
# web url of your EyeTV archive
our $ARCHIVEURL = "/~" . $TVUSER . "/EyeTV Archive";

# create a symlink ~/Sites/Movies -> ~/Movies
our $EXTRA_MOVIES_DIR = "/Users/" . $TVUSER . "/Sites/Movies";
our $EXTRA_MOVIES_URL = "/~" . $TVUSER . "/Movies";

# create a symlink ~/Sites/Music -> ~/Music
our $EXTRA_MUSIC_DIR = "/Users/" . $TVUSER . "/Sites/Music/iTunes/iTunes Music";
our $EXTRA_MUSIC_URL = "/~" . $TVUSER . "/Music/iTunes/iTunes Music";

our $CHANNELLIST = "/Library/Application Support/EyeTV/Shared/ChannelList.xml";
our $SERVERNAME = $ENV{SERVER_NAME} || $ENV{HTTP_HOST} || "localhost";

my $script_name = $ENV{SCRIPT_NAME} || "/cgi-bin/augtv.pl";
my $cgi = CGI->new();

my @fields = ({
	name => "id",
	display_header => "Delete",
	style => "zeilem",
	plist => {
		0 => "id",
		1 => "id",
	},
	display_value => sub {
		if (defined $_[0]->{id}) {
			return "<a href=\"?action=delete" .
				($_[1] == 0 ? "movie" : "schedule") . "&id=" .
				$cgi->escapeHTML($_[0]->{id}) . "\">X</a>";
		} else {
			return "";
		}
	},
}, {
	name => "title",
	display_header => "Title",
	sort_func => sub {
		return &compare_string($a, $b, "title");
	},
	style => "zeilen",
	plist => {
		0 => "display title",
		1 => "display title",
	},
	display_value => sub {
		my $html = "";
		if (defined $_[0]->{mpgfile}) {
			$html .= "<a href=\"" . $_[0]->{url} . "/" .
				$cgi->escapeHTML($_[0]->{mpgfile}) .
				"\">";
		}
		if (defined $_[0]->{title}) {
			$html .= $cgi->escapeHTML($_[0]->{title});
		}
		if (defined $_[0]->{mpgfile}) {
			$html .= "</a>";
		}
		return $html;
	},
}, {
	name => "start",
	display_header => "Start",
	sort_func => sub {
		return &compare_string($a, $b, "start");
	},
	style => "zeiler",
	plist => {
		0 => "actual start time",
		1 => "start",
	},
	display_value => sub {
		# 2010-05-09T01:55:00Z
		if (!defined $_[0]->{start}) {
			return "";
		} elsif ($_[0]->{start} =~ /^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})Z$/) {
			my ($sec, $min, $hour, $day, $mon, $year, @rest) =
				localtime(timegm($6, $5, $4, $3, $2 - 1,
				                 $1 - 1900));
			$mon++;
			$year += 1900;
			foreach ($sec, $min, $hour, $day, $mon) {
				$_ = sprintf("%02d", $_);
			}
			return $cgi->escapeHTML("$year/$mon/$day $hour:$min");
		} else {
			return $cgi->escapeHTML($_[0]->{start});
		}
	},
}, {
	name => "length",
	display_header => "Length",
	sort_func => sub {
		return &compare_integer($a, $b, "length");
	},
	style => "zeiler",
	plist => {
		0 => "actual duration",
		1 => "duration",
	},
	display_value => sub {
		if (defined $_[0]->{length}) {
			return &display_value_length($_[0]->{length});
		} else {
			return "";
		}
	},
}, {
	name => "size",
	display_header => "Size",
	sort_func => sub {
		return &compare_integer($a, $b, "size");
	},
	style => "zeiler",
	plist => {
		0 => undef,
	},
	display_value => sub {
		if (defined $_[0]->{size}) {
			return &display_value_size($_[0]->{size});
		} else {
			return "";
		}
	},
}, {
	name => "repeat",
	display_header => "Repeat",
	sort_func => sub {
		return &compare_string($a, $b, "repeat");
	},
	style => "zeile",
	plist => {
		1 => "repeats",
	},
	display_value => sub {
		if (defined $_[0]->{repeat}) {
			return $cgi->escapeHTML($_[0]->{repeat});
		} else {
			return "";
		}
	},
}, {
	name => "channel",
	display_header => "Channel",
	sort_func => sub {
		return &compare_string($a, $b, "channel");
	},
	style => "zeilen",
	plist => {
		0 => "channel name",
		1 => "station name",
	},
	display_value => sub {
		if (defined $_[0]->{channel}) {
			return $cgi->escapeHTML($_[0]->{channel});
		} else {
			return "";
		}
	},
}, {
	name => "description",
	display_header => "Description",
	sort_func => sub {
		return &compare_string($a, $b, "description");
	},
	style => "zeile",
	plist => {
		0 => "description",
		1 => "description",
	},
	display_value => sub {
		if (defined $_[0]->{description}) {
			return $cgi->escapeHTML($_[0]->{description});
		} else {
			return "";
		}
	},
});

my $action = $cgi->param("action") || "";

if ($action eq "record") {
	my $programs = &get_programs();
	my $station = $cgi->param("station") || &fatal("need station name");
	my $title = $cgi->param("title") || &fatal("need title");
	my $description = $cgi->param("description") || "";
	my $duration = $cgi->param("duration") || &fatal("need duration");
	&fatal("invalid duration") if ($duration !~ /^\d+$/);
	my $year = $cgi->param("year") || &fatal("need year");
	my $month = $cgi->param("month") || &fatal("need month");
	my $day = $cgi->param("day") || &fatal("need day");
	my $hour = $cgi->param("hour");
	&fatal("need hour") if !defined $hour;
	my $minute = $cgi->param("minute");
	&fatal("need minute") if !defined $minute;
	my $repeats = $cgi->param("repeats");
	&fatal("need repeats") if !defined $repeats;
	&fatal("invalid repeats $repeats") if (($repeats < 0) ||
	                                       ($repeats > 127));

	my $channels = &get_channels();
	my $channel = $channels->{$station};
	&fatal("unknown station $station") if !defined $channel;

	my ($starttime, $endtime) = &start_endtime($year, $month, $day, $hour,
	                                           $minute, 0, $duration);
	foreach my $program(@$programs) {
		my $starttime0 = $program->[17];
		my $endtime0 = $program->[18];
		if (($starttime < $endtime0) && ($endtime > $starttime0)) {
			&fatal("overlaps with " . $program->[8]. " at " .
				$program->[4] . ":" . $program->[5] . " on " .
				$program->[11]);
		}
	}

	foreach ($title, $description) {
		s//Ae/sgo;
		s//Oe/sgo;
		s//Ue/sgo;
		s//ae/sgo;
		s//oe/sgo;
		s//ue/sgo;
		s//ss/sgo;
		s/[^A-Za-z0-9\,\.\-\;\:\_\<\>\&\(\)=\?\+\*\#\!\ ]/_/sgio;
	}

	my $uid = &osascript("tell application \"EyeTV\"\n" .
		"make new program with properties {" .
			"title:\"$title\", " .
			"repeats:$repeats, " .
			"description:\"$description\", " .
			"channel number:\"$channel\", " .
			"station name:\"$station\", " .
			"start time:date \"" . &datetime_ampm(
				$year, $month, $day, $hour, $minute, 0
			) . "\", " .
			"duration:$duration}\n" .
		"end tell\n");
	&ok(join("\t", @$uid));
} elsif ($action eq "deleteprogram") {
	my $programs = &get_programs();
	my $id = $cgi->param("id") || &fatal("need id");
	foreach my $program(@$programs) {
		if ($id eq $program->[16]) {
			my $raw = &osascript("tell application \"EyeTV\"\n" .
				"delete program id " . $program->[16] . "\n" .
				"end tell\n");
			&ok(join("\t", @$raw));
			exit(0);
		}
	}
	&fatal("no such program");
} elsif ($action eq "channels") {
	my $channels = &get_channels();
	&ok(join("\n", map { "$_\t" . $channels->{$_} } sort {
		$channels->{$a} <=> $channels->{$b}
	} keys %$channels));
} elsif ($action eq "datetime") {
	my @datetime = localtime();
	$datetime[4]++;
	$datetime[5] += 1900;
	&format_datetime(\@datetime);
	&ok(join("\t", reverse((@datetime)[0..5])));
} elsif ($action eq "programs") {
	&ok(&format_programs(&get_programs()));
} elsif ($action =~ /^delete(movie|schedule)$/) {
	my $type = ($1 eq "movie" ? 0 : 1);
	my $type_string = ($type == 0 ? "movie" : "schedule");
	my $html = &start_html("Delete " . $type_string);
	my @movies = @{&scan_archivedir($type)};
	my $id = $cgi->param("id") || "";
	$html .= "<table>\n<tr>\n";
	for (my $i = 1; $i <= $#fields; $i++) {
		if (exists $fields[$i]->{plist}->{$type}) {
			$html .= "<th>" . $fields[$i]->{display_header} .
			         "</th>\n";
		}
	}
	$html .= "</tr>\n";
	my $count = 0;
	my $zeile = 0;
	foreach my $movie(@movies) {
		next if ($id ne $movie->{id});
		$html .= "<tr>\n";
		for (my $i = 1; $i <= $#fields; $i++) {
			next if !exists $fields[$i]->{plist}->{$type};
			$html .= "<td class=\"" .
				$fields[$i]->{style} .
				"$zeile\">" .
				&{$fields[$i]->{display_value}}($movie, $type) .
				"</td>\n";
		}
		$html .= "</tr>\n";
		$zeile = 1 - $zeile;
		$count++;
	}
	$html .= "</tr>\n</table>\n<p>\n";
	my $back = "?action=" . $type_string;
	if ($count == 0) {
		$html .= "Nothing to delete!<br><a href=\"" . $back .
			"\">Back</a>\n";
	} else {
		$html .= "Delete this " . $type_string;
		$html .= "s" if ($count != 1);
		$html .= "?<br>\n<a href=\"?action=dodelete" . $type_string .
			"&id=" . $id . "\">Yes</a> <a href=\"" . $back .
			"\">No</a>\n";
	}
	$html .= "</p>\n";
	$html .= &end_html();
	&output_html($html);
} elsif ($action =~ /^dodelete(movie|schedule)$/) {
	my $type = ($1 eq "movie" ? 0 : 1);
	my $type_string = ($type == 0 ? "movie" : "schedule");
	my $id = $cgi->param("id") || &fatal("need id");
	&osascript("tell application \"EyeTV\"\n" .
		"delete " .
		($type == 0 ? "recording" : "program") .
		" id " . $id . "\n" .
		"end tell\n");
	my $redir = $script_name . "?action=" . $type_string;
	my $html = &start_html("Delete " . $type_string, $redir);
	$html .= "<p>If redirection doesn't work, click <a href=\"" .
		$redir . "\">here</a>.</p>\n";
	$html .= &end_html();
	&output_html($html);
} elsif ($action eq "m3umusic") {
	my $music = &traverse_dir($EXTRA_MUSIC_DIR, "", $EXTRA_MUSIC_URL,
	                          qw(mp3 wav wma ogg mp4 mp2 m4a mod midi mid
	                             aif aiff));
	&output_m3u("music", $music);
} elsif ($action eq "m3umovie") {
	my @movies = @{&get_allmovies(0)};
	my $field_index = $cgi->param("field_index") || -1;
	my $sort_order = $cgi->param("sort_order") || "";

	$sort_order = "desc" if !$sort_order;
	if (($field_index < 0) || ($field_index > $#fields) ||
		!defined $fields[$field_index]->{sort_func}) {
		$field_index = 2;
	}

	@movies = sort { &{$fields[$field_index]->{sort_func}} } @movies;
	@movies = reverse @movies if ($sort_order eq "desc");

	&output_m3u("movies", \@movies);
} else {
	my $type = ($action eq "schedule" ? 1 : 0);
	my @movies = ($type ?  @{&scan_archivedir($type)} : @{&get_allmovies($type)});
	my $field_index = $cgi->param("field_index") || -1;
	my $sort_order = $cgi->param("sort_order") || "";

	$sort_order = "desc" if !$sort_order;
	if (($field_index < 0) || ($field_index > $#fields) ||
		!defined $fields[$field_index]->{sort_func}) {
		$field_index = 2;
	}

	@movies = sort { &{$fields[$field_index]->{sort_func}} } @movies;
	@movies = reverse @movies if ($sort_order eq "desc");

	my $movie_count = @movies;
	my $total_size = 0;
	my $total_length = 0;

	foreach (@movies) {
		if (defined $_->{size} && ($_->{size} > 0)) {
			$total_size += $_->{size};
		}
		if (defined $_->{length} && ($_->{length} > 0)) {
			$total_length += $_->{length};
		}
	}

	my $html = &start_html(($type == 0 ? "Movies" : "Schedules"), undef,
	                       $field_index, $sort_order);
	$html .= "<p>" . $movie_count . " " .
		($type == 0 ? "movie" : "schedule") .
		($movie_count == 1 ? "" : "s") .
		", " . &display_value_length($total_length) . " play time";
	if ($type == 0) {
		$html .= ", " .	&display_value_size($total_size);
	}
	$html .= "</p>\n<table>\n<tr>\n";

	for (my $i = 0; $i <= $#fields; $i++) {
		if (exists $fields[$i]->{plist}->{$type}) {
			$html .= "<th>";
			my $updown;
			if (defined $fields[$i]->{sort_func}) {
				$html .= "<a href=\"?action=" .
                                        ($type == 0 ? "movie" : "schedule") .
                                        "&field_index=" . $i . "&sort_order=";
				if ($i != $field_index) {
					$html .= "asc";
					$updown = "";
				} elsif ($sort_order eq "asc") {
					$html .= "desc";
					$updown = " &uarr;";
				} else {
					$html .= "asc";
					$updown = " &darr;";
				}
				$html .= "\">";
			}
			$html .= $fields[$i]->{display_header};
			if (defined $fields[$i]->{sort_func}) {
				$html .= $updown . "</a>";
			}
			$html .= "</th>\n";
		}
	}

	$html .= "</tr>\n";

	my $zeile = 0;
	foreach my $movie(@movies) {
		$html .= "<tr>\n";
		foreach my $field(@fields) {
			next if !exists $field->{plist}->{$type};
			$html .= "<td class=\"" . $field->{style} .
				"$zeile\">" .
				&{$field->{display_value}}($movie, $type) .
				"</td>\n";
		}
		$html .= "</tr>\n";
		$zeile = 1 - $zeile;
	}

	$html .= "</tr>\n</table>\n";
	$html .= &end_html();
	&output_html($html);
}


sub get_channels() {
	my %channels = ();
	my $string = "";
	my $found_channel = 0;
	my $found_number = 0;
	my $channel;
	my $number;
	my $parser = XML::Parser->new(Handlers => {
		End => sub {
			my $element = $_[1];
			if ($element eq "string") {
				$channel = $string if $found_channel;
				$number = $string if $found_number;
				if (defined $channel && defined $number) {
					$channels{$channel} = $number;
					$channel = undef;
					$number = undef;
				}
			} elsif ($element eq "key") {
				$found_channel = ($string eq "channel name");
				$found_number = ($string eq "display number");
			}
		},
		Char => sub {
			$string = $_[1];
		},
	});
	eval {
		$parser->parsefile($CHANNELLIST);
	};
	&fatal("cannot parse channelist: $@") if $@;
	return \%channels;
}

sub get_programs() {
	my $raw = &osascript("tell application \"EyeTV\"\n" .
		"set progno to number of programs\n" .
		"set output to \"\"\n" .
		"repeat while progno > 0\n" .
		"set {" .
			"start time:starttime, " .
			"duration:durationseconds, " .
			"title:thetitle, " .
			"description:desc, " .
			"channel number:channelnumber, " .
			"station name:stationname, " .
			"input source:inputsource, " .
			"repeats:repeatperiod, " .
			"quality:qualitylevel, " .
			"enabled:isenabled, " .
			"unique ID:uid " .
		"} to program progno\n" .
		"set {" .
			"year:syear, " .
			"month:smonth, " .
			"day:sday, " .
			"hours:shour, " .
			"minutes:sminute, " .
			"seconds:ssecond" .
		"} to starttime\n" .
		"set output to output & " .
			"progno & (ASCII character 9) & " .
			"syear & (ASCII character 9) & " .
			"(smonth as integer) & (ASCII character 9) & " .
			"sday & (ASCII character 9) & " .
			"shour & (ASCII character 9) & " .
			"sminute & (ASCII character 9) & " .
			"ssecond & (ASCII character 9) & " .
			"durationseconds & (ASCII character 9) & " .
			"thetitle & (ASCII character 9) & " .
			"desc & (ASCII character 9) & " .
			"channelnumber & (ASCII character 9) & " .
			"stationname & (ASCII character 9) & " .
			"inputsource & (ASCII character 9) & " .
			"repeatperiod & (ASCII character 9) & " .
			"qualitylevel & (ASCII character 9) & " .
			"isenabled & (ASCII character 9) & " .
			"uid & (ASCII character 10)\n" .
		"set progno to progno - 1\n" .
		"end repeat\n" .
		"get output\n" .
	"end tell\n");

	my @programs = ();
	foreach my $line(@$raw) {
		chomp($line);
		next if !$line;
		my @fields = split(/\t/, $line);
		&format_datetime(\@fields);
		$fields[7] = int($fields[7]);
		$fields[16] = int($fields[16]);
		my ($starttime, $endtime) = &start_endtime((@fields)[1..7]);
		push @fields, ($starttime, $endtime);
		my $repeat = $fields[13];
		if ($repeat eq "none") {
			push @programs, \@fields;
		} else {
			my $repeats = &build_repeats(\@fields);
			for (my $day = -1; $day < 60; $day++) {
				&repeat_program(\@programs, \@fields, $repeats,
				                $day);
			}
		}
	}

	return \@programs;
}

sub repeat_program() {
	my ($programs, $fields, $repeats, $day) = @_;
	my @tmp = @$fields;
	$tmp[17] += $day * 86400;
	$tmp[18] += $day * 86400;
	my $dayofweek;
	($tmp[6], $tmp[5], $tmp[4], $tmp[3], $tmp[2], $tmp[1], $dayofweek) =
							localtime($tmp[17]);

	if (exists $repeats->{$dayofweek}) {
		$tmp[1] += 1900;
		$tmp[2]++;
		&format_datetime(\@tmp);
		push @$programs, \@tmp;
	}
}

sub build_repeats() {
	my $fields = shift;

	my $repeat = $fields->[13];
	my %repeat = ();
	if ($repeat eq "daily") {
		for (my $weekday = 0; $weekday < 8; $weekday++) {
			$repeat{$weekday} = undef;
		}
	} elsif ($repeat eq "weekdays") {
		for (my $weekday = 1; $weekday < 6; $weekday++) {
			$repeat{$weekday} = undef;
		}
	} else {
		my @weekdays = ("Sunday", "Monday", "Tuesday", "Wednesday",
		                "Thursday", "Friday", "Saturday", "Sunday");
		for (my $weekday = 0; $weekday <= $#weekdays; $weekday++) {
			if ($repeat =~ /$weekdays[$weekday]/) {
				$repeat{$weekday} = undef;
			}
		}
	}

	return \%repeat;
}

sub format_datetime() {
	my $fields = shift;
	for (my $i = 2; $i <= 6; $i++) {
		$fields->[$i] = sprintf("%02d", $fields->[$i]);
	}
}

sub format_programs() {
    my $programs = shift;
    return join("\n", map { join("\t", @$_) } sort {
            $a->[1] <=> $b->[1] ||
            $a->[2] <=> $b->[2] ||
            $a->[3] <=> $b->[3] ||
            $a->[4] <=> $b->[4] ||
            $a->[5] <=> $b->[5] ||
            $a->[6] <=> $b->[6] ||
            $a->[8] cmp $b->[8]
    } @$programs);
}

sub datetime_ampm() {
	my $hour = $_[3] % 12;
	$hour = 12 if !$hour;
	my $ampm = $_[3] < 12 ? "AM" : "PM";
	return "$_[1]/$_[2]/$_[0] $hour:$_[4]:$_[5] $ampm";
}

sub start_endtime() {
	my $starttime = timelocal($_[5], $_[4], $_[3], $_[2], $_[1] - 1,
	                          $_[0] - 1900);
	my $endtime = $starttime + $_[6];
	return ($starttime, $endtime);
}

sub osascript() {
	my $command = shift;

	my ($RHCGI, $RHAS, $WHCGI, $WHAS);
	pipe($RHCGI, $WHAS) || &fatal("pipe(): $!");
	pipe($RHAS, $WHCGI) || &fatal("pipe(): $!");

	my $pid = fork();
	&fatal("fork(): $!") if ($pid < 0);

	if (!$pid) {
		close($RHCGI);
		close($WHCGI);
		open(STDIN, ">&", $RHAS);
		open(STDOUT, ">&", $WHAS);
		open(STDERR, ">&", $WHAS);
		exec("sudo -u $TVUSER /usr/bin/osascript");
		die $!;
	}

	close($RHAS);
	close($WHAS);

	print $WHCGI $command;
	close($WHCGI);

	my @output = <$RHCGI>;
	do {
		$pid = waitpid(-1, WNOHANG);
		&fatal("osascript ($pid) returned $?: " .
		       join(" ", @output)) if (($pid > 0) && $?);
	} while ($pid > 0);

	return \@output;
}

sub ok() {
	my $output = shift;
	&output_text("OK\n" . $output);
}

sub fatal() {
	my $error = shift;

	foreach ($error) {
		s/[^\S ]//sgio;
	}

	&output_text("ERR\n" . $error);
	die $error;
}

sub output_generic() {
	my $data = join("", (@_)[2..$#_]);
	print STDOUT "Content-Type: $_[0]\r\n" .
	             "Content-Length: " . length($data) . "\r\n";
	if (defined $_[1]) {
		print STDOUT 'Content-Disposition: attachment; filename="',
		             $_[1], "\"\r\n";
	}
	print STDOUT "\r\n", $data;
}

sub output_m3u() {
	my ($filename, $movies) = @_;

	my $m3u = "#EXTM3U\n";
	foreach (@$movies) {
		next if !defined $_->{mpgfile};
		$m3u .= "#EXTINF:";
		if (defined $_->{length}) {
			$m3u .= int($_->{length});
		} else {
			$m3u .= "-1";
		}
		$m3u .= ",";
		if (defined $_->{title}) {
			$m3u .= $_->{title};
		} else {
			$m3u .= $_->{mpgfile};
		}
		$m3u .= "\nhttp://" . $SERVERNAME .
		        URI::Escape::uri_escape($_->{url} . "/" . $_->{mpgfile},
			                        '^A-Za-z0-9-._~/') . "\n";
	}

	&output_generic("application/x-mpegurl", $filename . ".m3u", $m3u);
}

sub output_text() {
	&output_generic("text/plain", undef, @_);
}

sub output_html() {
	&output_generic("text/html", undef, @_);
}

sub start_html() {
	my ($subtitle, $location, $field_index, $sort_order) = @_;
	if ($location) {
		$location = "<meta http-equiv=\"refresh\" content=\"0; URL=" .
		            $location . "\">\n";
	} else {
		$location = "";
	}
	my $html = <<EOF;
<html>
<head>
<title>AugTV</title>$location
<style type="text/css">
<!--
body {
  background-color:white;
  color:black;
  font-family:Arial, sans-serif;
  font-size:12px;
}
table {
  background-color:orange;
  border:none solid black;
  border-spacing:2px;
}
th, th a {
  background-color:#777;
  color:white;
  font-size:12px;
  font-weight:bold;
  text-decoration:none;
  white-space:nowrap;
}
td {
  padding:2px;
  font-size:12px;
}
td a {
  color:black;
}
td.zeile0, td.zeiler0, td.zeilen0, td.zeilem0 {
  background-color:#CCC;
}
td.zeile1, td.zeiler1, td.zeilen1, td.zeilem1 {
  background-color:#EEE;
}
td.zeiler0, td.zeiler1 {
  text-align:right;
}
td.zeilem0, td.zeilem1 {
  text-align:center;
}
td.zeilen0, td.zeilen1, td.zeiler0, td.zeiler1 {
  white-space:nowrap;
}
//-->
</style>
</head>
<body>
<h1>AugTV</h1>
<p>[<a href="?">Movies</a>] [<a href="?action=schedule">Schedules</a>]
EOF

	if (defined $field_index && defined $sort_order) {
		$html .= <<EOF;
<br>Playlists:
[<a href="?action=m3umovie&field_index=$field_index&sort_order=$sort_order">Movies</a>]
[<a href="?action=m3umusic">Music</a>]
EOF
	}

	$html .= <<EOF;
</p>
<h2>$subtitle</h2>
EOF

	return $html;
}

sub end_html() {
	return "</body>\n</html>";
}

sub get_allmovies() {
	my $type = shift;

	my $movies = &scan_archivedir($type);
	push @$movies, @{&traverse_dir($EXTRA_MOVIES_DIR, "", $EXTRA_MOVIES_URL,
	                               qw(avi mpg mpeg mkv m4v asf wmv mov))};

	return $movies;
}

sub traverse_dir() {
	my ($dir, $reldir, $url, @extensions) = @_;

	opendir(DH, $dir) || return [];
	my @entries = readdir(DH);
	closedir(DH);

	my @files;
	foreach my $entry(sort @entries) {
		my $absfile = $dir . "/" . $entry;
		my $relfile = $reldir . ($reldir ? "/" : "") . $entry;
		if (-d $absfile && ($entry ne ".") && ($entry ne "..")) {
			push @files, @{&traverse_dir($absfile, $relfile, $url,
			                             @extensions)};
		}
		if (-f $absfile && &matches_one($entry, @extensions)) {
			my @stat = stat($absfile);
			my ($sec, $min, $hour, $day, $mon, $year) =
							gmtime($stat[9]);
			my $start = sprintf("%04d-%02d-%02dT%02d:%02d:%02dZ",
			                    $year + 1900, $mon + 1, $day,
			                    $hour, $min, $sec);
			push @files, {
				url     => $url,
				mpgfile => $relfile,
				size    => $stat[7],
				start   => $start,
				title   => substr($entry, 0,
				                  rindex($entry, ".")),
			};
		}
	}

	return \@files;
}

sub scan_archivedir() {
	my $type = shift;
	my @movies = ();

	my $extension = ($type == 0 ? "eyetv" : "eyetvsched");

	opendir(DH, $ARCHIVEDIR) || die "$ARCHIVEDIR: $!";
	my @entries = readdir(DH);
	closedir(DH);

	foreach (@entries) {
		my $dir = "$ARCHIVEDIR/$_";
		if (-d $dir && !/^Live TV Buffer/) {
			my %movie = ();
			if (/\.$extension$/) {
				&scan_moviedir(\%movie, $_, $type);
				push @movies, \%movie;
			}
		}
	}

	return \@movies;
}

sub scan_moviedir() {
	my ($movie, $dir, $type) = @_;

	my $extension = ($type == 0 ? "eyetvr" : "eyetvp");

	opendir(DH, "$ARCHIVEDIR/$dir");
	my @entries = readdir(DH);
	closedir(DH);

	foreach (@entries) {
		my $file = "$ARCHIVEDIR/$dir/$_";
		if (-f $file) {
			&parse_movie($movie, $file, $type) if /\.$extension$/;
			if (/\.mpg$/) {
				$movie->{mpgfile} = "$dir/$_";
				$movie->{size} = -s $file;
				$movie->{url} = $ARCHIVEURL;
			}
		}
	}
}

sub parse_movie() {
	my ($movie, $file, $type) = @_;
	my $field;
        my $value = "";

	my $xml = XML::Parser->new(Handlers => {
                Start => sub {
                        $value = "";
                },
		End => sub {
			my $element = $_[1];
			$movie->{$field} = $value if $field;
			$field = undef;
			if ($element eq "key") {
				$field = &get_field($movie, $value, $type);
			}
                        $value = "";
		},
		Char => sub {
			$value .= $_[1];
		},
	});
	$xml->parsefile($file);
}

sub get_field() {
	my ($movie, $value, $type) = @_;

	foreach my $field(@fields) {
		if ($field->{plist}->{$type} &&
		    ($value eq $field->{plist}->{$type})) {
			return $field->{name};
		}
	}

	return undef;
}

sub display_value_length() {
	my $sec = $_[0] % 60;
	my $tmp = $_[0] / 60;
	my $min = $tmp % 60;
	my $hour = $tmp /= 60;
	return $cgi->escapeHTML(join(":", map { sprintf("%02d", $_) }
	                                  ($hour, $min, $sec)));
}

sub display_value_size() {
	my $size = $_[0] * 100;
	my @suffixes = qw(B kB MB GB TB);
	my $i;
	for ($i = 0; ($i <= $#suffixes) && ($size >= 102400); $i++) {
		$size /= 1024;
	}
	return $cgi->escapeHTML(sprintf("%.2f %s", $size / 100, $suffixes[$i]));
}

sub compare_string() {
	my ($a, $b, $field) = @_;
	my $a_val = $a->{$field} || "";
	my $b_val = $b->{$field} || "";

	return (lc($a_val) cmp lc($b_val));
}

sub compare_integer() {
	my ($a, $b, $field) = @_;
	my $a_val = $a->{$field} || 0;
	my $b_val = $b->{$field} || 0;

	return ($a_val <=> $b_val);
}

sub matches_one() {
	my ($filename, @extensions) = @_;
	my $idx = rindex($filename, ".");
	return 0 if ($idx < 0);
	my $ext = lc(substr($filename, $idx + 1));

	foreach (@extensions) {
		return 1 if (lc($_) eq $ext);
	}

	return 0;
}
