#!/usr/bin/perl -w

# $Rev: 31 $

# replace USER_RUNNING_EYETV by the username who runs EyeTV and
#     add the following line to /etc/sudoers:
# _www	ALL=(USER_RUNNING_EYETV) NOPASSWD: /usr/bin/osascript

use strict;
use CGI;
use POSIX ":sys_wait_h";
use Time::Local;
use XML::Parser;

our $TVUSER = "fjo";
our $CHANNELLIST = "/Library/Application Support/EyeTV/Shared/ChannelList.xml";
our $ARCHIVEDIR = "/Volumes/Backup 2/EyeTV Archive";
our $ARCHIVEURL = "/~fjo/eyetv"; # eg. symlink ~/Sites/eyetv to $ARCHIVEDIR

my $script_name = $ENV{SCRIPT_NAME} || "/cgi-bin/augtv.pl";
my $cgi = CGI->new();

my @fields = ({
	name => "Delete",
	sort_func => sub { 0 },
	default => "",
	style => "zeilem",
	plist => "id",
	display_func => sub {
		return "<a href=\"?action=deleterecording&id=" .
			$cgi->escapeHTML($_[0]->{Delete}) .
			"\">X</a>";
	},
}, {
	name => "Title",
	sort_func => sub { lc($a->{Title}) cmp lc($b->{Title}) },
	default => "",
	style => "zeilen",
	plist => "display title",
	display_func => sub {
		return "<a href=\"" . $ARCHIVEURL . "/" .
			$cgi->escapeHTML($_[0]->{mpgfile}) .
			"\">" .
			$cgi->escapeHTML($_[0]->{Title}) .
			"</a>";
	},
}, {
	name => "Recorded",
	sort_func => sub { $a->{Recorded} cmp $b->{Recorded} },
	default => "",
	style => "zeiler",
	plist => "actual start time",
	display_func => sub {
		# 2010-05-09T01:55:00Z
		if ($_[0]->{Recorded} =~ /^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})Z$/) {
			my ($sec, $min, $hour, $day, $mon, $year, @rest) =
				localtime(timegm($6, $5, $4, $3, $2 - 1, $1 - 1900));
			$mon++;
			$year += 1900;
			foreach ($sec, $min, $hour, $day, $mon) {
				$_ = sprintf("%02d", $_);
			}
			return $cgi->escapeHTML("$year/$mon/$day $hour:$min");
		} else {
			return $cgi->escapeHTML($_[0]->{Recorded});
		}
	},
}, {
	name => "Length",
	sort_func => sub { $a->{Length} <=> $b->{Length} },
	default => -1,
	style => "zeiler",
	plist => "actual duration",
	display_func => sub {
		return &display_func_length($_[0]->{Length});
	},
}, {
	name => "Size",
	sort_func => sub { $a->{Length} <=> $b->{Length} },
	default => -1,
	style => "zeiler",
	display_func => sub {
		return &display_func_size($_[0]->{Size});
	},
}, {
	name => "Channel",
	sort_func => sub { lc($a->{Channel}) cmp lc($b->{Channel}) },
	default => "",
	style => "zeilen",
	plist => "channel name",
	display_func => sub {
		return $cgi->escapeHTML($_[0]->{Channel});
	},
}, {
	name => "Description",
	sort_func => sub { lc($a->{Description}) cmp lc($b->{Description}) },
	default => "",
	style => "zeile",
	plist => "description",
	display_func => sub {
		return $cgi->escapeHTML($_[0]->{Description});
	},
});

my $action = $cgi->param("action") || "";

if ($action eq "record") {
	my $programs = &get_programs();
	my $station = $cgi->param("station") || &fatal("need station name");
	my $title = $cgi->param("title") || &fatal("need title");
	my $description = $cgi->param("description") || "";
	my $duration = $cgi->param("duration") || &fatal("need duration");
	&fatal("invalid duration") if ($duration !~ /^\d+$/);
	my $year = $cgi->param("year") || &fatal("need year");
	my $month = $cgi->param("month") || &fatal("need month");
	my $day = $cgi->param("day") || &fatal("need day");
	my $hour = $cgi->param("hour");
	&fatal("need hour") if !defined $hour;
	my $minute = $cgi->param("minute");
	&fatal("need minute") if !defined $minute;
	my $repeats = $cgi->param("repeats");
	&fatal("need repeats") if !defined $repeats;
	&fatal("invalid repeats $repeats") if (($repeats < 0) || ($repeats > 127));

	my $channels = &get_channels();
	my $channel = $channels->{$station};
	&fatal("unknown station $station") if !defined $channel;

	my ($starttime, $endtime) = &start_endtime($year, $month, $day, $hour,
		$minute, 0, $duration);
	foreach my $program(@$programs) {
		my $starttime0 = $program->[17];
		my $endtime0 = $program->[18];
		if (($starttime < $endtime0) && ($endtime > $starttime0)) {
			&fatal("overlaps with " . $program->[8]. " at " .
				$program->[4] . ":" . $program->[5] . " on " .
				$program->[11]);
		}
	}

	foreach ($title, $description) {
		s//Ae/sgo;
		s//Oe/sgo;
		s//Ue/sgo;
		s//ae/sgo;
		s//oe/sgo;
		s//ue/sgo;
		s//ss/sgo;
		s/[^A-Za-z0-9\,\.\-\;\:\_\<\>\&\(\)=\?\+\*\#\!\ ]/_/sgio;
	}
	
	my $uid = &osascript("tell application \"EyeTV\"\n" .
		"make new program with properties {" .
			"title:\"$title\", " .
			"repeats:$repeats, " .
			"description:\"$description\", " .
			"channel number:\"$channel\", " .
			"station name:\"$station\", " .
			"start time:date \"" . &datetime_ampm(
				$year, $month, $day, $hour, $minute, 0
			) . "\", " .
			"duration:$duration}\n" .
		"end tell\n");
	&ok(join("\t", @$uid));
} elsif ($action eq "deleteprogram") {
	my $programs = &get_programs();
	my $id = $cgi->param("id") || &fatal("need id");
	foreach my $program(@$programs) {
		if ($id eq $program->[16]) {
			my $raw = &osascript("tell application \"EyeTV\"\n" .
				"delete program id " . $program->[16] . "\n" .
				"end tell\n");
			&ok(join("\t", @$raw));
			exit(0);
		}
	}
	&fatal("no such program");
} elsif ($action eq "channels") {
	my $channels = &get_channels();
	&ok(join("\n", map { "$_\t" . $channels->{$_} } sort {
		$a <=> $b
	} keys %$channels));
} elsif ($action eq "datetime") {
	my @datetime = localtime();
	$datetime[4]++;
	$datetime[5] += 1900;
	&format_datetime(\@datetime);
	&ok(join("\t", reverse((@datetime)[0..5])));
} elsif ($action eq "programs") {
	&ok(&format_programs(&get_programs()));
} elsif ($action eq "deleterecording") {
	my $html = &start_html();
	my @movies = @{&scan_archivedir()};
	my $id = $cgi->param("id") || "";
	$html .= "<table>\n<tr>\n";
	for (my $i = 1; $i <= $#fields; $i++) {
		$html .= "<th>" . $fields[$i]->{name} . "</th>\n";
	}
	$html .= "</tr>\n";
	my $count = 0;
	my $zeile = 0;
	foreach my $movie(@movies) {
		if ($id eq $movie->{Delete}) {
			$html .= "<tr>\n";
			for (my $i = 1; $i <= $#fields; $i++) {
				$html .= "<td class=\"" . $fields[$i]->{style} .
					"$zeile\">" .
					&{$fields[$i]->{display_func}}($movie) .
					"</td>\n";
			}
			$html .= "</tr>\n";
			$zeile = 1 - $zeile;
			$count++;
		}
	}
	$html .= "</tr>\n</table>\n<p>\n";
	if ($count == 0) {
		$html .= "Nothing to delete!<br><a href=\"" .
			$script_name . "\">Back</a>\n";
	} else {
		if ($count == 1) {
			$html .= "Delete this movie?";
		} else {
			$html .= "Delete these movies?";
		}
		$html .= "<br>\n<a href=\"?action=dodeleterecording&id=" . $id .
			"\">Yes</a> <a href=\"" . $script_name . "\">No</a>\n";
	}
	$html .= "</p>\n";
	$html .= &end_html();
	&output_html($html);
} elsif ($action eq "dodeleterecording") {
	my $id = $cgi->param("id") || &fatal("need id");
	&osascript("tell application \"EyeTV\"\n" .
		"delete recording id " . $id . "\n" .
		"end tell\n");
	my $html = &start_html();
	$html .= "<p>If redirection doesn't work, click <a href=\"" .
		$script_name . "\">here</a>.</p>\n";
	$html .= &end_html($script_name);
	&output_html($html);
} else {
	my $html = &start_html();
	my @movies = @{&scan_archivedir()};
	my $sort_order = $cgi->param("sort_order") || "";
	my $field_index = $cgi->param("field_index") || -1;

	$sort_order = "desc" if !$sort_order;
	$field_index = 2 if (($field_index < 0) || ($field_index > $#fields));

	@movies = sort { &{$fields[$field_index]->{sort_func}} } @movies;
	@movies = reverse @movies if ($sort_order eq "desc");

	my $movie_count = @movies;
	my $total_size = 0;
	my $total_length = 0;

	foreach (@movies) {
		$total_size += $_->{Size} if ($_->{Size} > 0);
		$total_length += $_->{Length} if ($_->{Length} > 0);
	}

	$html .= "<p>" . $movie_count . " movie" .
		($movie_count == 1 ? "" : "s") .
		", " . &display_func_length($total_length) . " play time, " .
		&display_func_size($total_size) . "</p>\n" .
		"<table>\n" .
		"<tr>\n";

	for (my $i = 0; $i <= $#fields; $i++) {
		$html .= "<th><a href=\"?field_index=" . $i . "&sort_order=";
		my $updown;
		if ($i != $field_index) {
			$html .= "asc";
			$updown = "";
		} elsif ($sort_order eq "asc") {
			$html .= "desc";
			$updown = " &uarr;";
		} else {
			$html .= "asc";
			$updown = " &darr;";
		}
		$html .= "\">" . $fields[$i]->{name} . $updown . "</a></th>\n";
	}

	$html .= "</tr>\n";

	my $zeile = 0;
	foreach my $movie(@movies) {
		$html .= "<tr>\n";
		foreach my $field(@fields) {
			$html .= "<td class=\"" . $field->{style} . "$zeile\">" .
				&{$field->{display_func}}($movie) . "</td>\n";
		}
		$html .= "</tr>\n";
		$zeile = 1 - $zeile;
	}

	$html .= "</tr>\n</table>\n";
	$html .= &end_html();
	&output_html($html);
}


sub get_channels() {
	my %channels = ();
	my $string = "";
	my $found_channel = 0;
	my $found_number = 0;
	my $channel;
	my $number;
	my $parser = XML::Parser->new(Handlers => {
		End => sub {
			my $element = $_[1];
			if ($element eq "string") {
				$channel = $string if $found_channel;
				$number = $string if $found_number;
				if (defined $channel && defined $number) {
					$channels{$channel} = $number;
					$channel = undef;
					$number = undef;
				}
			} elsif ($element eq "key") {
				$found_channel = ($string eq "channel name");
				$found_number = ($string eq "display number");
			}
		},
		Char => sub {
			$string = $_[1];
		},
	});
	eval {
		$parser->parsefile($CHANNELLIST);
	};
	&fatal("cannot parse channelist: $@") if $@;
	return \%channels;
}

sub get_programs() {
	my $raw = &osascript("tell application \"EyeTV\"\n" .
		"set progno to number of programs\n" .
		"set output to \"\"\n" .
		"repeat while progno > 0\n" .
		"set {" .
			"start time:starttime, " .
			"duration:durationseconds, " .
			"title:thetitle, " .
			"description:desc, " .
			"channel number:channelnumber, " .
			"station name:stationname, " .
			"input source:inputsource, " .
			"repeats:repeatperiod, " .
			"quality:qualitylevel, " .
			"enabled:isenabled, " .
			"unique ID:uid " .
		"} to program progno\n" .
		"set {" .
			"year:syear, " .
			"month:smonth, " .
			"day:sday, " .
			"hours:shour, " .
			"minutes:sminute, " .
			"seconds:ssecond" .
		"} to starttime\n" .
		"set output to output & " .
			"progno & (ASCII character 9) & " .
			"syear & (ASCII character 9) & " .
			"(smonth as integer) & (ASCII character 9) & " .
			"sday & (ASCII character 9) & " .
			"shour & (ASCII character 9) & " .
			"sminute & (ASCII character 9) & " .
			"ssecond & (ASCII character 9) & " .
			"durationseconds & (ASCII character 9) & " .
			"thetitle & (ASCII character 9) & " .
			"desc & (ASCII character 9) & " .
			"channelnumber & (ASCII character 9) & " .
			"stationname & (ASCII character 9) & " .
			"inputsource & (ASCII character 9) & " .
			"repeatperiod & (ASCII character 9) & " .
			"qualitylevel & (ASCII character 9) & " .
			"isenabled & (ASCII character 9) & " .
			"uid & (ASCII character 10)\n" .
		"set progno to progno - 1\n" .
		"end repeat\n" .
		"get output\n" .
	"end tell\n");

	my @programs = ();
	foreach my $line(@$raw) {
		chomp($line);
		next if !$line;
		my @fields = split(/\t/, $line);
		&format_datetime(\@fields);
		$fields[7] = int($fields[7]);
		$fields[16] = int($fields[16]);
		my ($starttime, $endtime) = &start_endtime((@fields)[1..7]);
		push @fields, ($starttime, $endtime);
		my $repeat = $fields[13];
		if ($repeat eq "none") {
			push @programs, \@fields;
		} else {
			my $repeats = &build_repeats(\@fields);
			for (my $day = -1; $day < 60; $day++) {
				&repeat_program(\@programs, \@fields, $repeats,
									$day);
			}
		}
	}

	return \@programs;
}

sub repeat_program() {
	my ($programs, $fields, $repeats, $day) = @_;
	my @tmp = @$fields;
	$tmp[17] += $day * 86400;
	$tmp[18] += $day * 86400;
	my $dayofweek;
	($tmp[6], $tmp[5], $tmp[4], $tmp[3], $tmp[2], $tmp[1], $dayofweek) =
							localtime($tmp[17]);

	if (exists $repeats->{$dayofweek}){
		$tmp[1] += 1900;
		$tmp[2]++;
		&format_datetime(\@tmp);
		push @$programs, \@tmp;
	}
}

sub build_repeats() {
	my $fields = shift;

	my $repeat = $fields->[13];
	my %repeat = ();
	if ($repeat eq "daily") {
		for (my $weekday = 0; $weekday < 8; $weekday++) {
			$repeat{$weekday} = undef;
		}
	} elsif ($repeat eq "weekdays") {
		for (my $weekday = 1; $weekday < 6; $weekday++) {
			$repeat{$weekday} = undef;
		}
	} else {
		my @weekdays = ("Sunday", "Monday", "Tuesday", "Wednesday",
				"Thursday", "Friday", "Saturday", "Sunday");
		for (my $weekday = 0; $weekday <= $#weekdays; $weekday++) {
			$repeat{$weekday} = undef if
					($repeat =~ /$weekdays[$weekday]/);
		}
	}

	return \%repeat;
}

sub format_datetime() {
	my $fields = shift;
	for (my $i = 2; $i <= 6; $i++) {
		$fields->[$i] = sprintf("%02d", $fields->[$i]);
	}
}

sub format_programs() {
    my $programs = shift;
    return join("\n", map { join("\t", @$_) } sort {
            $a->[1] <=> $b->[1] ||
            $a->[2] <=> $b->[2] ||
            $a->[3] <=> $b->[3] ||
            $a->[4] <=> $b->[4] ||
            $a->[5] <=> $b->[5] ||
            $a->[6] <=> $b->[6] ||
            $a->[8] cmp $b->[8]
    } @$programs);
}

sub datetime_ampm() {
	my $hour = $_[3] % 12;
	$hour = 12 if !$hour;
	my $ampm = $_[3] < 12 ? "AM" : "PM";
	return "$_[1]/$_[2]/$_[0] $hour:$_[4]:$_[5] $ampm";
}

sub start_endtime() {
	my $starttime = timelocal($_[5], $_[4], $_[3], $_[2], $_[1] - 1,
								$_[0] - 1900);
	my $endtime = $starttime + $_[6];
	return ($starttime, $endtime);
}

sub osascript() {
	my $command = shift;

	my ($RHCGI, $RHAS, $WHCGI, $WHAS);
	pipe($RHCGI, $WHAS) || &fatal("pipe(): $!");
	pipe($RHAS, $WHCGI) || &fatal("pipe(): $!");

	my $pid = fork();
	&fatal("fork(): $!") if ($pid < 0);

	if (!$pid) {
		close($RHCGI);
		close($WHCGI);
		open(STDIN, ">&", $RHAS);
		open(STDOUT, ">&", $WHAS);
		open(STDERR, ">&", $WHAS);
		exec("sudo -u $TVUSER /usr/bin/osascript");
		die $!;
	}

	close($RHAS);
	close($WHAS);

	print $WHCGI $command;
	close($WHCGI);

	my @output = <$RHCGI>;
	do {
		$pid = waitpid(-1, WNOHANG);
		&fatal("osascript ($pid) returned $?: " .
				join(" ", @output)) if (($pid > 0) && $?);
	} while ($pid > 0);

	return \@output;
}

sub ok() {
	my $output = shift;
	&output_text("OK\n" . $output);
}

sub fatal() {
	my $error = shift;

	foreach ($error) {
		s/[^\S ]//sgio;
	}

	&output_text("ERR\n" . $error);
	die $error;
}

sub output_text() {
	&output_generic("text/plain", @_);
}

sub output_generic() {
	my $data = join("", (@_)[1..$#_]);
	print STDOUT "Content-Type: $_[0]\r\n" .
		"Content-Length: " . length($data) . "\r\n" .
		"\r\n" .
		$data;
}

sub output_html() {
	&output_generic("text/html", @_);
}

sub start_html() {
	my $location = shift;
	if ($location) {
		$location = "<meta http-equiv=\"refresh\" content=\"0; URL=" .
			$location . "\">\n";
	}
	my $html = <<EOF;
<html>
<head>
<title>AugTV</title>$location
<style type="text/css">
<!--
body {
  background-color:white;
  color:black;
  font-family:Arial, sans-serif;
  font-size:12px;
}
table {
  background-color:orange;
  border:none solid black;
  border-spacing:2px;
}
th, th a {
  background-color:#777;
  color:white;
  font-size:12px;
  font-weight:bold;
  text-decoration:none;
  white-space:nowrap;
}
td {
  padding:2px;
  font-size:12px;
}
td a {
  color:black;
}
td.zeile0, td.zeiler0, td.zeilen0, td.zeilem0 {
  background-color:#CCC;
}
td.zeile1, td.zeiler1, td.zeilen1, td.zeilem1 {
  background-color:#EEE;
}
td.zeiler0, td.zeiler1 {
  text-align:right;
}
td.zeilem0, td.zeilem1 {
  text-align:center;
}
td.zeilen0, td.zeilen1, td.zeiler0, td.zeiler1 {
  white-space:nowrap;
}
//-->
</style>
</head>
<body>
<h1>AugTV</h1>
EOF

	return $html;
}

sub end_html() {
	my $html = "</body>\n</html>";
	return $html;
}

sub scan_archivedir() {
	my @movies = ();

	opendir(DH, $ARCHIVEDIR) || die "$ARCHIVEDIR: $!";
	foreach (readdir(DH)) {
		my $dir = "$ARCHIVEDIR/$_";
		if (-d $dir && !/^Live TV Buffer/) {
			my %movie = map { $_->{name} => $_->{default} } @fields;
			&scan_moviedir(\%movie, $_);
			push @movies, \%movie if defined $movie{mpgfile};
		}
	}
	closedir(DH);

	return \@movies;
}

sub scan_moviedir() {
	my ($movie, $dir) = @_;

	opendir(DH2, "$ARCHIVEDIR/$dir");
	foreach (readdir(DH2)) {
		my $file = "$ARCHIVEDIR/$dir/$_";
		if (-f $file) {
			&parse_movie($movie, $file) if /\.eyetvr$/;
			if (/\.mpg$/) {
				$movie->{mpgfile} = "$dir/$_";
				$movie->{Size} = -s $file;
			}
		}
	}
	closedir(DH2);
}

sub parse_movie() {
	my ($movie, $file) = @_;
	my $field;
        my $value = "";

	my $xml = XML::Parser->new(Handlers => {
                Start => sub {
                        $value = "";
                },
		End => sub {
			my $element = $_[1];
			$movie->{$field} = $value if $field;
			$field = undef;
			if ($element eq "key") {
				foreach (@fields) {
					$field = $_->{name} if (exists $_->{plist} && ($value eq $_->{plist}));
				}
			}
                        $value = "";
		},
		Char => sub {
			$value .= $_[1];
		},
	});
	$xml->parsefile($file);
}

sub display_func_length() {
	my $sec = $_[0] % 60;
	my $tmp = $_[0] / 60;
	my $min = $tmp % 60;
	my $hour = $tmp /= 60;
	return $cgi->escapeHTML(join(":", map { sprintf("%02d", $_) } ($hour, $min, $sec)));
}

sub display_func_size() {
	my $size = $_[0] * 100;
	my @suffixes = qw(B kB MB GB TB);
	my $i;
	for ($i = 0; ($i <= $#suffixes) && ($size >= 102400); $i++) {
		$size /= 1024;
	}
	return $cgi->escapeHTML(sprintf("%.2f %s", $size / 100, $suffixes[$i]));
}