#!/usr/bin/perl -w

# $Rev: 23 $

# replace USER_RUNNING_EYETV by the username who runs EyeTV and
#     add the following line to /etc/sudoers:
# _www	ALL=(USER_RUNNING_EYETV) NOPASSWD: /usr/bin/osascript

use strict;
use CGI;
use POSIX ":sys_wait_h";
use Time::Local;
use XML::Parser;

our $TVUSER = "fjo";
our $CHANNELLIST = "/Library/Application Support/EyeTV/Shared/ChannelList.xml";

my $cgi = CGI->new();
my $action = $cgi->param("action") || "";

if ($action eq "record") {
	my $programs = &get_programs();
	my $station = $cgi->param("station") || &fatal("need station name");
	my $title = $cgi->param("title") || &fatal("need title");
	my $description = $cgi->param("description") || "";
	my $duration = $cgi->param("duration") || &fatal("need duration");
	&fatal("invalid duration") if ($duration !~ /^\d+$/);
	my $year = $cgi->param("year") || &fatal("need year");
	my $month = $cgi->param("month") || &fatal("need month");
	my $day = $cgi->param("day") || &fatal("need day");
	my $hour = $cgi->param("hour");
	&fatal("need hour") if !defined $hour;
	my $minute = $cgi->param("minute");
	&fatal("need minute") if !defined $minute;
	my $repeats = $cgi->param("repeats");
	&fatal("need repeats") if !defined $repeats;
	&fatal("invalid repeats $repeats") if (($repeats < 0) || ($repeats > 127));

	my $channels = &get_channels();
	my $channel = $channels->{$station};
	&fatal("unknown station $station") if !defined $channel;

	my ($starttime, $endtime) = &start_endtime($year, $month, $day, $hour,
		$minute, 0, $duration);
	foreach my $program(@$programs) {
		my $starttime0 = $program->[17];
		my $endtime0 = $program->[18];
		if (($starttime < $endtime0) && ($endtime > $starttime0)) {
			&fatal("overlaps with " . $program->[8]. " at " .
				$program->[4] . ":" . $program->[5] . " on " .
				$program->[11]);
		}
	}

	foreach ($title, $description) {
		s//Ae/sgo;
		s//Oe/sgo;
		s//Ue/sgo;
		s//ae/sgo;
		s//oe/sgo;
		s//ue/sgo;
		s//ss/sgo;
		s/[^A-Za-z0-9\,\.\-\;\:\_\<\>\&\(\)=\?\+\*\#\!\ ]/_/sgio;
	}
	
	my $uid = &osascript("tell application \"EyeTV\"\n" .
		"make new program with properties {" .
			"title:\"$title\", " .
			"repeats:$repeats, " .
			"description:\"$description\", " .
			"channel number:\"$channel\", " .
			"station name:\"$station\", " .
			"start time:date \"" . &datetime_ampm(
				$year, $month, $day, $hour, $minute, 0
			) . "\", " .
			"duration:$duration}\n" .
		"end tell\n");
	&ok(join("\t", @$uid));
} elsif ($action eq "delete") {
	my $programs = &get_programs();
	my $id = $cgi->param("id") || &fatal("need id");
	foreach my $program(@$programs) {
		if ($id eq $program->[16]) {
			my $raw = &osascript("tell application \"EyeTV\"\n" .
				"delete program id " . $program->[16] . "\n" .
				"end tell\n");
			&ok(join("\t", @$raw));
			exit(0);
		}
	}
	&fatal("no such program");
} elsif ($action eq "channels") {
	my $channels = &get_channels();
	&ok(join("\n", map { "$_\t" . $channels->{$_} } sort {
		$a <=> $b
	} keys %$channels));
} elsif ($action eq "datetime") {
	my @datetime = localtime();
	$datetime[4]++;
	$datetime[5] += 1900;
	&format_datetime(\@datetime);
	&ok(join("\t", reverse((@datetime)[0..5])));
} else {
	&ok(&format_programs(&get_programs()));
}


sub get_channels() {
	my %channels = ();
	my $string = "";
	my $found_channel = 0;
	my $found_number = 0;
	my $channel;
	my $number;
	my $parser = XML::Parser->new(Handlers => {
		End => sub {
			my $element = $_[1];
			if ($element eq "string") {
				$channel = $string if $found_channel;
				$number = $string if $found_number;
				if (defined $channel && defined $number) {
					$channels{$channel} = $number;
					$channel = undef;
					$number = undef;
				}
			} elsif ($element eq "key") {
				$found_channel = ($string eq "channel name");
				$found_number = ($string eq "display number");
			}
		},
		Char => sub {
			$string = $_[1];
		},
	});
	eval {
		$parser->parsefile($CHANNELLIST);
	};
	&fatal("cannot parse channelist: $@") if $@;
	return \%channels;
}

sub get_programs() {
	my $raw = &osascript("tell application \"EyeTV\"\n" .
		"set progno to number of programs\n" .
		"set output to \"\"\n" .
		"repeat while progno > 0\n" .
		"set {" .
			"start time:starttime, " .
			"duration:durationseconds, " .
			"title:thetitle, " .
			"description:desc, " .
			"channel number:channelnumber, " .
			"station name:stationname, " .
			"input source:inputsource, " .
			"repeats:repeatperiod, " .
			"quality:qualitylevel, " .
			"enabled:isenabled, " .
			"unique ID:uid " .
		"} to program progno\n" .
		"set {" .
			"year:syear, " .
			"month:smonth, " .
			"day:sday, " .
			"hours:shour, " .
			"minutes:sminute, " .
			"seconds:ssecond" .
		"} to starttime\n" .
		"set output to output & " .
			"progno & (ASCII character 9) & " .
			"syear & (ASCII character 9) & " .
			"(smonth as integer) & (ASCII character 9) & " .
			"sday & (ASCII character 9) & " .
			"shour & (ASCII character 9) & " .
			"sminute & (ASCII character 9) & " .
			"ssecond & (ASCII character 9) & " .
			"durationseconds & (ASCII character 9) & " .
			"thetitle & (ASCII character 9) & " .
			"desc & (ASCII character 9) & " .
			"channelnumber & (ASCII character 9) & " .
			"stationname & (ASCII character 9) & " .
			"inputsource & (ASCII character 9) & " .
			"repeatperiod & (ASCII character 9) & " .
			"qualitylevel & (ASCII character 9) & " .
			"isenabled & (ASCII character 9) & " .
			"uid & (ASCII character 10)\n" .
		"set progno to progno - 1\n" .
		"end repeat\n" .
		"get output\n" .
	"end tell\n");

	my @programs = ();
	foreach my $line(@$raw) {
		chomp($line);
		next if !$line;
		my @fields = split(/\t/, $line);
		&format_datetime(\@fields);
		$fields[7] = int($fields[7]);
		$fields[16] = int($fields[16]);
		my ($starttime, $endtime) = &start_endtime((@fields)[1..7]);
		push @fields, ($starttime, $endtime);
		my $repeat = $fields[13];
		if ($repeat eq "none") {
			push @programs, \@fields;
		} else {
			my $repeats = &build_repeats(\@fields);
			for (my $day = -1; $day < 60; $day++) {
				&repeat_program(\@programs, \@fields, $repeats,
									$day);
			}
		}
	}

	return \@programs;
}

sub repeat_program() {
	my ($programs, $fields, $repeats, $day) = @_;
	my @tmp = @$fields;
	$tmp[17] += $day * 86400;
	$tmp[18] += $day * 86400;
	my $dayofweek;
	($tmp[6], $tmp[5], $tmp[4], $tmp[3], $tmp[2], $tmp[1], $dayofweek) =
							localtime($tmp[17]);

	if (exists $repeats->{$dayofweek}){
		$tmp[1] += 1900;
		$tmp[2]++;
		&format_datetime(\@tmp);
		push @$programs, \@tmp;
	}
}

sub build_repeats() {
	my $fields = shift;

	my $repeat = $fields->[13];
	my %repeat = ();
	if ($repeat eq "daily") {
		for (my $weekday = 0; $weekday < 8; $weekday++) {
			$repeat{$weekday} = undef;
		}
	} elsif ($repeat eq "weekdays") {
		for (my $weekday = 1; $weekday < 6; $weekday++) {
			$repeat{$weekday} = undef;
		}
	} else {
		my @weekdays = ("Sunday", "Monday", "Tuesday", "Wednesday",
				"Thursday", "Friday", "Saturday", "Sunday");
		for (my $weekday = 0; $weekday <= $#weekdays; $weekday++) {
			$repeat{$weekday} = undef if
					($repeat =~ /$weekdays[$weekday]/);
		}
	}

	return \%repeat;
}

sub format_datetime() {
	my $fields = shift;
	for (my $i = 2; $i <= 6; $i++) {
		$fields->[$i] = sprintf("%02d", $fields->[$i]);
	}
}

sub format_programs() {
    my $programs = shift;
    return join("\n", map { join("\t", @$_) } sort {
            $a->[1] <=> $b->[1] ||
            $a->[2] <=> $b->[2] ||
            $a->[3] <=> $b->[3] ||
            $a->[4] <=> $b->[4] ||
            $a->[5] <=> $b->[5] ||
            $a->[6] <=> $b->[6] ||
            $a->[8] cmp $b->[8]
    } @$programs);
}

sub datetime_ampm() {
	my $hour = $_[3] % 12;
	$hour = 12 if !$hour;
	my $ampm = $_[3] < 12 ? "AM" : "PM";
	return "$_[1]/$_[2]/$_[0] $hour:$_[4]:$_[5] $ampm";
}

sub start_endtime() {
	my $starttime = timelocal($_[5], $_[4], $_[3], $_[2], $_[1] - 1,
								$_[0] - 1900);
	my $endtime = $starttime + $_[6];
	return ($starttime, $endtime);
}

sub osascript() {
	my $command = shift;

	my ($RHCGI, $RHAS, $WHCGI, $WHAS);
	pipe($RHCGI, $WHAS) || &fatal("pipe(): $!");
	pipe($RHAS, $WHCGI) || &fatal("pipe(): $!");

	my $pid = fork();
	&fatal("fork(): $!") if ($pid < 0);

	if (!$pid) {
		close($RHCGI);
		close($WHCGI);
		open(STDIN, ">&", $RHAS);
		open(STDOUT, ">&", $WHAS);
		open(STDERR, ">&", $WHAS);
		exec("sudo -u $TVUSER /usr/bin/osascript");
		die $!;
	}

	close($RHAS);
	close($WHAS);

	print $WHCGI $command;
	close($WHCGI);

	my @output = <$RHCGI>;
	do {
		$pid = waitpid(-1, WNOHANG);
		&fatal("osascript ($pid) returned $?: " .
				join(" ", @output)) if (($pid > 0) && $?);
	} while ($pid > 0);

	return \@output;
}

sub ok() {
	my $output = shift;
	&output_text("OK\n" . $output);
}

sub fatal() {
	my $error = shift;

	foreach ($error) {
		s/[^\S ]//sgio;
	}

	&output_text("ERR\n" . $error);
	die $error;
}

sub output_text() {
	&output_generic("text/plain", @_);
}

sub output_generic() {
	print STDOUT "Content-Type: $_[0]\n" .
		"Content-Length: " . length($_[1]) . "\n" .
		"\n" .
		$_[1];
}