#!/usr/bin/perl -w

# $Rev: 23 $

use strict;
use CGI;
use Time::Local;
use XML::Parser;

our $ARCHIVEDIR = "/Volumes/Backup 2/EyeTV Archive";
our $ARCHIVEURL = "/~fjo/eyetv"; # eg. symlink ~/Sites/eyetv to $ARCHIVEDIR

my @fields = ({
	name => "Title",
	sort_func => sub { lc($a->{Title}) cmp lc($b->{Title}) },
	default => "",
	style => "zeilen",
	plist => "display title",
	display_func => sub { @_ },
	mpg_link => 1,
}, {
	name => "Recorded",
	sort_func => sub { $a->{Recorded} cmp $b->{Recorded} },
	default => "",
	style => "zeiler",
	plist => "actual start time",
	display_func => sub {
		# 2010-05-09T01:55:00Z
		if ($_[0] =~ /^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})Z$/) {
			my ($sec, $min, $hour, $day, $mon, $year, @rest) =
				localtime(timegm($6, $5, $4, $3, $2 - 1, $1 - 1900));
			$mon++;
			$year += 1900;
			foreach ($sec, $min, $hour, $day, $mon) {
				$_ = sprintf("%02d", $_);
			}
			return "$year/$mon/$day $hour:$min";
		} else {
			@_;
		}
	},
}, {
	name => "Length",
	sort_func => sub { $a->{Length} <=> $b->{Length} },
	default => -1,
	style => "zeiler",
	plist => "actual duration",
	display_func => sub {
		my $sec = $_[0] % 60;
		my $tmp = $_[0] / 60;
		my $min = $tmp % 60;
		my $hour = $tmp /= 60;
		return join(":", map { sprintf("%02d", $_) } ($hour, $min, $sec));
	},
}, {
	name => "Size",
	sort_func => sub { $a->{Length} <=> $b->{Length} },
	default => -1,
	style => "zeiler",
	display_func => sub {
		my $size = $_[0] * 100;
		my @suffixes = qw(B kB MB GB TB);
		my $i;
		for ($i = 0; ($i <= $#suffixes) && ($size >= 102400); $i++) {
			$size /= 1024;
		}
		return sprintf("%.2f %s", $size / 100, $suffixes[$i]);
	},
}, {
	name => "Channel",
	sort_func => sub { lc($a->{Channel}) cmp lc($b->{Channel}) },
	default => "",
	style => "zeilen",
	plist => "channel name",
	display_func => sub { @_ },
}, {
	name => "Description",
	sort_func => sub { lc($a->{Description}) cmp lc($b->{Description}) },
	default => "",
	style => "zeile",
	plist => "description",
	display_func => sub { @_ },
});

my $cgi = CGI->new();

my $sort_order = $cgi->param("sort_order") || "";
my $field_index = $cgi->param("field_index") || -1;

if (!$sort_order && (($field_index < 0) || ($field_index > $#fields))) {
	$sort_order = "desc";
	$field_index = 1;
} else {
	$sort_order = "asc" if (($sort_order ne "asc") && ($sort_order ne "desc"));
	$field_index = 0 if (($field_index < 0) || ($field_index > $#fields));
}

my @movies = sort { &{$fields[$field_index]->{sort_func}} } @{&scan_archivedir()};
@movies = reverse @movies if ($sort_order eq "desc");

my $html = <<EOF;
<html>
<head>
<title>AugMovie</title>
<style type="text/css">
<!--
body {
  background-color:white;
  color:black;
  font-family:Arial, sans-serif;
  font-size:12px;
}
table {
  background-color:orange;
  border:none solid black;
  border-spacing:2px;
}
th, th a {
  background-color:#777;
  color:white;
  font-size:12px;
  font-weight:bold;
  text-decoration:none;
  white-space:nowrap;
}
td {
  padding:2px;
  font-size:12px;
}
td a {
  color:black;
}
td.zeile0, td.zeiler0, td.zeilen0 {
  background-color:#CCC;
}
td.zeile1, td.zeiler1, td.zeilen1 {
  background-color:#EEE;
}
td.zeiler0, td.zeiler1 {
  text-align:right;
}
td.zeilen0, td.zeilen1, td.zeiler0, td.zeiler1 {
  white-space:nowrap;
}
//-->
</style>
</head>
<body>
<h1>AugMovie</h1>
EOF

my $movie_count = @movies;
my $total_size = 0;
my $total_length = 0;

foreach (@movies) {
	$total_size += $_->{Size} if ($_->{Size} > 0);
	$total_length += $_->{Length} if ($_->{Length} > 0);
}

$html .= "<p>" . $movie_count . " movie" . ($movie_count == 1 ? "" : "s") .
	", " . $fields[2]->{display_func}($total_length) . " play time, " .
	$fields[3]->{display_func}($total_size) . "</p>\n" .
	"<table>\n" .
	"<tr>\n";

foreach (0..$#fields) {
	$html .= "<th><a href=\"?field_index=" . $_ . "&sort_order=";
	my $updown;
	if ($_ != $field_index) {
		$html .= "asc";
		$updown = "";
	} elsif ($sort_order eq "asc") {
		$html .= "desc";
		$updown = " &uarr;";
	} else {
		$html .= "asc";
		$updown = " &darr;";
	}
	$html .= "\">" . $fields[$_]->{name} . $updown . "</a></th>\n";
}

$html .= "</tr>\n";

my $zeile = 0;
foreach my $movie(@movies) {
	$html .= "<tr>\n";
	foreach my $field(@fields) {
		$html .= "<td class=\"" . $field->{style} . "$zeile\">";
		$html .= "<a href=\"" . $ARCHIVEURL . "/" .
			$cgi->escapeHTML($movie->{mpgfile}) .
			"\">" if $field->{mpg_link};
		$html .= $cgi->escapeHTML(&{$field->{display_func}}($movie->{$field->{name}}));
		$html .= "</a>" if $field->{mpg_link};
		$html .= "</td>\n";
	}
	$html .= "</tr>\n";
	$zeile = 1 - $zeile;
}

$html .= "</tr>\n</table>\n</body>\n</html>";

print "Content-Type: text/html\r\nContent-Length: " . length($html) .
	"\r\n\r\n" . $html;


sub scan_archivedir() {
	my @movies = ();

	opendir(DH, $ARCHIVEDIR) || die "$ARCHIVEDIR: $!";
	foreach (readdir(DH)) {
		my $dir = "$ARCHIVEDIR/$_";		
		if (-d $dir && !/^Live TV Buffer/) {
			my %movie = map { $_->{name} => $_->{default} } @fields;
			&scan_moviedir(\%movie, $_);
			push @movies, \%movie if defined $movie{mpgfile};
		}
	}
	closedir(DH);

	return \@movies;
}

sub scan_moviedir() {
	my ($movie, $dir) = @_;

	opendir(DH2, "$ARCHIVEDIR/$dir");
	foreach (readdir(DH2)) {
		my $file = "$ARCHIVEDIR/$dir/$_";
		if (-f $file) {
			&parse_movie($movie, $file) if /\.eyetvr$/;
			if (/\.mpg$/) {
				$movie->{mpgfile} = "$dir/$_";
				$movie->{Size} = -s $file;
			}
		}
	}
	closedir(DH2);
}

sub parse_movie() {
	my ($movie, $file) = @_;
	my ($field, $value);

	my $xml = XML::Parser->new(Handlers => {
		End => sub {
			my $element = $_[1];
			$movie->{$field} = $value if $field;
			$field = undef;
			if ($element eq "key") {
				foreach (@fields) {
					$field = $_->{name} if (exists $_->{plist} && ($value eq $_->{plist}));
				}
			}
		},
		Char => sub {
			$value = $_[1];
		},
	});
	$xml->parsefile($file);
}